﻿#
# 翻訳された xml ファイルと、オリジナルの dat をマージして
# 英語版の dat を作成する。
# 翻訳提出ファイルを作成した時の Command_Report.txt を利用する。
# 出力はオリジナルが格納されているディレクトリに行われる。
#
# COMMAND -DatDirectory ./dat -TranslatedDirectory ./en
#
# Command_Report.txt が無い場合のために、datフォルダ以下の Ja.ex.xml を
# 全て対象にする、-AllXml オプションがある。
#
# COMMAND -DatDirectory ./dat -TranslatedDirectory ./en -AllXml
#
param
(
    [string]$DatDirectory="./dat",
    [string]$TranslatedDirectory="./en",
    [switch]$AllXml = $false,
    [switch]$Help = $false
)

#---- ヘルプ
if ( $Help -eq $True )
{
Write-Host (`
@"
*** Usage of {0}
    Merge translated changelog xml file to original dat to output dat file.

Command
    {0} [options]

Options
    -DatDirectory <directory>         Specify original data directory.
    -TranslatedDirectory <directory>  Specify translated xml directory.
    -AllXml                           Merge all xml files.
    -Help                             Show help (this).
"@ `
-f $MyInvocation.MyCommand.Name )

    exit 0
}

#---- データファイルのあるフォルダチェック
# 引数指定のチェック
if ($DatDirectory -eq "")
{
    Write-Host "*** Error: DatDirectory is not specified."
    exit 1
}

# 存在チェック
if ( !(Test-Path $DatDirectory) )
{
    Write-Host "*** Error: Not found DatDirectory."
    exit 1
}

# ディレクトリであるかのチェック
if ( Test-Path $DatDirectory -PathType Leaf )
{
    Write-Host "*** Error: Specified file to DatDirectory option."
    exit 1
}

$ScriptPath = (Split-Path $MyInvocation.MyCommand.Path -Parent)

#----------------------------------------------------------------
#
function Merge-Dat( $name )
{
    $name -match "(.*)Ja.dat" | Out-Null
    $InFile = "$TranslatedDirectory/" + $matches[1] + "Ja.ex.xml"
    $OriginalFile = "$DatDirectory/$name"
    $OutFile = "$TranslatedDirectory/" + $matches[1] + "En.dat"

    PowerShell $ScriptPath\mergeLogForTranslation.ps1 -InFile $InFile -OriginalLog $OriginalFile -OutFile $OutFile -Detail
}

#----------------------------------------------------------------
# Main
#

#---- フォルダの中にある Ja.ex.xml を走査する指定かどうか
if ( $AllXml -eq $True )
{
    Get-ChildItem $DatDirectory | Foreach-Object `
    {
        #---- ディレクトリは除外
        if ( Test-Path "$DatDirectory/$_" -PathType Container )
        {
            return
        }

        #---- "Ja.ex.xml" で終わるものが対象
        if ( ([string]$_).EndsWith( "Ja.ex.xml" ) -eq $True )
        {
            $_ -match "(.*)Ja.ex.xml" | Out-Null
            $f = $matches[1] + "Ja.dat"
            Merge-Dat $f
        }
    }
}
else
{
    #---- レポート出力のためのファイル
    $ReportFile = "$DatDirectory/Command_Report.txt"

    #---- 読んでいき対象ファイルを調査
    Get-Content $ReportFile | Foreach-Object `
    {
        if ( $_.StartsWith( "# Name" ) )
        {
            $_ -match "# Name = (.*)" | Out-Null
            Merge-Dat $matches[1]
        }
    }
}
