﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Text.RegularExpressions;

namespace PackageCreator
{
    public interface IFileGetter
    {
        /// <summary>
        /// filepath にマッチするファイルのリストを絶対パスで返す。
        /// </summary>
        /// <param name="filepath">絶対パスで指定するマッチパターン</param>
        /// <returns>マッチした絶対パスの列挙</returns>
        IEnumerable<string> GetFiles(string filepath);
    }

    internal abstract class FileTree : IFileGetter
    {
        public IEnumerable<string> GetFiles(string filepath)
        {
            var filename = Path.GetFileName(filepath);
            if (filename == "*" || filename == "**")
            {
                return GetFilesForWildCard(Path.GetDirectoryName(filepath), filename == "**");
            }
            var fullPath = Path.GetFullPath(filepath);
            if (FileExists(fullPath))
            {
                return new[] { fullPath };
            }
            else
            {
                return Enumerable.Empty<string>();
            }
        }

        protected abstract IEnumerable<string> GetFilesForWildCard(string directoryPath, bool allDirectories);

        protected abstract bool FileExists(string filepath);
    }

    internal class FileTreeByFileSystem : FileTree
    {
        protected override IEnumerable<string> GetFilesForWildCard(string directoryPath, bool allDirectories)
        {
            var split = directoryPath.Split(Path.DirectorySeparatorChar);
            var parentDir = string.Join(Path.DirectorySeparatorChar.ToString(),
                                split.TakeWhile(s => !s.Contains("?") && !s.Contains("*")));

            IEnumerable<string> dirs;

            if (parentDir == directoryPath)
            {
                // パス文字列の末尾にしかワイルドカードが使われていない場合はそのまま
                dirs =  new List<string>(new[] { directoryPath });
            }
            else
            {
                var regex = new Regex(WildCard.CovertToRegularExpressionString(directoryPath) + "$");

                // 末尾以外にもワイルドカードが使われている場合は一致するものをすべて返す
                dirs = Directory.EnumerateDirectories(
                        parentDir,
                        "*",
                        SearchOption.AllDirectories
                        ).Where(s => regex.IsMatch(s));
            }

            return dirs.SelectMany(dir => Directory.EnumerateFiles(dir, "*", allDirectories ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly));
        }

        protected override bool FileExists(string filepath)
        {
            return File.Exists(filepath);
        }
    }

    internal class FileTreeBySpecificFilesImpl
    {
        private SortedSet<string> m_Files;

        public FileTreeBySpecificFilesImpl(IEnumerable<string> files)
        {
            var sortedSet = files as SortedSet<string>;
            this.m_Files = sortedSet != null && sortedSet.Comparer == StringComparer.Ordinal
                ? sortedSet
                : new SortedSet<string>(files, StringComparer.Ordinal);
        }

        public IEnumerable<string> GetFilesForWildCard(string directoryPath, bool allDirectories)
        {
            return from dir in GetDirectories(directoryPath)
                   from path in GetSubFiles(dir)
                   where allDirectories || Path.GetDirectoryName(path) == dir
                   select path;
        }

        private IEnumerable<string> GetDirectories(string sourceDir)
        {
            if(!sourceDir.Contains("*") && !sourceDir.Contains("?"))
            {
                return new[] { sourceDir };
            }

            var split = sourceDir.Split(Path.DirectorySeparatorChar);
            var parentDir = string.Join(Path.DirectorySeparatorChar.ToString(),
                                split.TakeWhile(s => !s.Contains("?") && !s.Contains("*")));


            int wildcardPosition = sourceDir.Length;

            if(sourceDir.Contains("*") && sourceDir.Contains("?"))
            {
                wildcardPosition = Math.Min(sourceDir.IndexOf("*"), sourceDir.IndexOf("?"));
            }
            else
            {
                if(sourceDir.Contains("*"))
                {
                    wildcardPosition = sourceDir.IndexOf("*");
                }
                else
                {
                    wildcardPosition = sourceDir.IndexOf("?");
                }
            }

            var fixedPath = sourceDir.Substring(0, wildcardPosition);

            // Programs/Alice/Sources/Libraries/hoge* を最短マッチさせたいので
            // Programs/Alice/Sources/Libraries/hoge.*?// となるように末尾にセパレータを追加
            var normalizedPath = sourceDir.EndsWith(Path.DirectorySeparatorChar.ToString()) ? sourceDir : sourceDir + Path.DirectorySeparatorChar;
            var regexString = WildCard.CovertToRegularExpressionString(normalizedPath).Replace("*", "*?");

            var dirs = from file in m_Files.GetViewBetween(parentDir + Path.DirectorySeparatorChar, parentDir + (char)(Path.DirectorySeparatorChar + 1))
                       where file.StartsWith(fixedPath) // FIXME: 正規表現でのマッチングは重いので事前にある程度絞る
                       select Regex.Match(file, regexString, RegexOptions.IgnoreCase).Value;

            // 末尾に追加したセパレータを除去
            return dirs.Where(e => e != string.Empty).Distinct().Select(e => Path.GetDirectoryName(e));
        }

        private IEnumerable<string> GetSubFiles(string sourceDir)
        {
            return m_Files.GetViewBetween(sourceDir + Path.DirectorySeparatorChar, sourceDir + (char)(Path.DirectorySeparatorChar + 1));
        }

        public bool FileExists(string filepath)
        {
            return m_Files.Contains(filepath);
        }
    }

    internal class FileTreeBySpecificFiles : FileTree
    {
        private FileTreeBySpecificFilesImpl m_Impl;

        public FileTreeBySpecificFiles(IEnumerable<string> files)
        {
            this.m_Impl = new FileTreeBySpecificFilesImpl(files);
        }

        protected override bool FileExists(string filepath)
        {
            return m_Impl.FileExists(filepath);
        }

        protected override IEnumerable<string> GetFilesForWildCard(string directoryPath, bool allDirectories)
        {
            return m_Impl.GetFilesForWildCard(directoryPath, allDirectories);
        }
    }

    internal class FileTreeByListFile : FileTree
    {
        private FileTreeBySpecificFilesImpl m_Files;

        public FileTreeByListFile(string rootPath, string listFilePath, Encoding encoding = null)
        {
            string[] files;
            if (encoding == null)
            {
                files = File.ReadAllLines(listFilePath);
            }
            else
            {
                files = File.ReadAllLines(listFilePath, encoding);
            }

            m_Files = new FileTreeBySpecificFilesImpl(files.Select(p => Path.GetFullPath(Path.Combine(rootPath, p))));
        }

        protected override IEnumerable<string> GetFilesForWildCard(string directoryPath, bool allDirectories)
        {
            return m_Files.GetFilesForWildCard(directoryPath, allDirectories);
        }

        protected override bool FileExists(string filepath)
        {
            return m_Files.FileExists(filepath);
        }
    }

    internal class MultiFileGetter : IFileGetter
    {
        private IFileGetter[] m_FileGetters;

        public MultiFileGetter(params IFileGetter[] fileGetters)
        {
            this.m_FileGetters = fileGetters;
        }

        public IEnumerable<string> GetFiles(string filepath)
        {
            return m_FileGetters.SelectMany(fileGetter => fileGetter.GetFiles(filepath)).Distinct();
        }
    }
}
