﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;

namespace NxAgingHelper
{
    public class SdevController : ISdevController
    {
        private HostBridgeTelnetClient m_Client;
        private string m_HostName;
        private string m_UserName = "root";
        private string m_Password = "root";

        private Task m_ReceiveTask;
        private CancellationTokenSource m_CancellationTokenSource;

        public void Open(string hostname)
        {
            m_HostName = hostname;

            try
            {
                m_Client = new HostBridgeTelnetClient(hostname, 23, new CancellationToken());

                var loginTask = Task.Run(async () =>
                {
                    bool ok = await m_Client.TryLoginAsync(m_UserName, m_Password, 3000);
                    if (!ok)
                    {
                        throw new Exception(string.Format("Failed to connect to SDEV ({0}): host bridge login error).", m_HostName));
                    }
                });
                loginTask.Wait();

                m_CancellationTokenSource = new CancellationTokenSource();
                m_ReceiveTask = Task.Run(async () =>
                {
                    while (!m_CancellationTokenSource.Token.IsCancellationRequested)
                    {
                        await m_Client.ReadAsync(TimeSpan.FromSeconds(1000));
                    }
                });
            }
            catch (Exception e)
            {
                throw new Exception(string.Format("Failed to connect to SDEV ({0}).", m_HostName), e);
            }
        }

        public void Close()
        {
            m_CancellationTokenSource.Cancel();
            m_ReceiveTask.Wait();
            m_Client.Dispose();
        }

        public void PressPowerButton(int holdTime)
        {
            Task.Run(async () =>
            {
                string command = string.Format("ngpio_test -d 1 -p 1; ngpio_test -p 1 -w 1; usleep {0}; ngpio_test -p 1 -w 0", holdTime * 1000);
                await m_Client.WriteLine(command);
            });
        }

        public void ReleasePowerButtonForClose()
        {
            var task = Task.Run(async () =>
            {
                string command = string.Format("ngpio_test -d 1 -p 1; ngpio_test -p 1 -w 0");
                await m_Client.WriteLine(command);

                // Power ボタンが正しくリリースされるのを待つために、とりあえず7秒ウェイト
                // TODO: 他のコマンドの実行と同期を取る
                await Task.Delay(7000);
            });
            task.Wait();
        }

        public void SwitchUsbPort(bool enable)
        {
            Task.Run(async () =>
            {
                string command = string.Format("ngpio_test -d 1 -p 4; ngpio_test -p 4 -w {0}", enable ? 0 : 1);
                await m_Client.WriteLine(command);
            });
        }

        public void BatteryEmulation(int voltage, int percentage)
        {
            Task.Run(async () =>
            {
                string commandForVoltage = string.Format("ni2c_fg_test -a 0x19 -w 0x{0,0:X4};", voltage);
                await m_Client.WriteLine(commandForVoltage);

                // コマンドの送信が完了するのを待つために1秒くらいウェイト
                await Task.Delay(1000);
                string commandForPercentage = string.Format("ni2c_fg_test -a 0x06 -w 0x{0,0:X4}", percentage);
                await m_Client.WriteLine(commandForPercentage);
            });
        }

        // PrimS.Telnet.Client そのままでは TryLogingAsync がパスワード入力後に返ってこなかったため ":" を "#" に変更するために継承
        private class HostBridgeTelnetClient : PrimS.Telnet.Client
        {
            public HostBridgeTelnetClient(string hostname, int port, System.Threading.CancellationToken token)
                : base(new PrimS.Telnet.TcpByteStream(hostname, port), token)
            {
            }
            /// <summary>
            /// Tries to login asynchronously.
            /// </summary>
            /// <param name="username">The username.</param>
            /// <param name="password">The password.</param>
            /// <param name="loginTimeOutMs">The login time out ms.</param>
            /// <returns>True if successful.</returns>
            public new async Task<bool> TryLoginAsync(string username, string password, int loginTimeOutMs)
            {
                try
                {
                    if (await this.IsTerminatedWith(loginTimeOutMs, ":"))
                    {
                        await this.WriteLine(username);
                        if (await this.IsTerminatedWith(loginTimeOutMs, ":"))
                        {
                            await this.WriteLine(password);
                        }

                        return await this.IsTerminatedWith(loginTimeOutMs, "#");
                    }
                }
                catch (Exception)
                {
                    // NOP
                }

                return false;
            }

            private async Task<bool> IsTerminatedWith(int loginTimeOutMs, string terminator)
            {
                return (await this.TerminatedReadAsync(terminator, TimeSpan.FromMilliseconds(loginTimeOutMs), 1)).TrimEnd().EndsWith(terminator);
            }
        }
    }
}
