﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;

namespace NxAgingHelper
{
    public interface IAsyncCommand : ICommand
    {
        CancellationToken CancellationToken { get; set; }

        void RunAsync(ControllerSet controllerSet);
        void Wait();
    }

    public abstract class AbstractAsyncCommand : IAsyncCommand
    {
        public CancellationToken CancellationToken { get; set; }

        public void Run(ControllerSet controllerSet)
        {
            RunAsync(controllerSet);
            Wait();
        }

        public abstract void RunAsync(ControllerSet controllerSet);
        public abstract void Wait();
    }

    public class ExecuteCommand : AbstractAsyncCommand
    {
        private IList<ICommand> m_CommandList;

        private Task m_Task;

        public ExecuteCommand(IList<ICommand> commandList)
        {
            m_CommandList = commandList;
        }

        public override void RunAsync(ControllerSet controllerSet)
        {
            m_Task = Task.Run(() =>
            {
                foreach (var command in m_CommandList)
                {
                    if (CancellationToken.IsCancellationRequested)
                    {
                        break;
                    }

                    if (command is IAsyncCommand)
                    {
                        (command as IAsyncCommand).CancellationToken = this.CancellationToken;
                    }

                    command.Run(controllerSet);
                }
            });
        }

        public override void Wait()
        {
            m_Task.TryWait(CancellationToken);
        }
    }

    public class ParallelExecuteCommand : AbstractAsyncCommand
    {
        private IList<ICommand> m_CommandList;

        private IList<Task> m_TaskList = new List<Task>();

        public ParallelExecuteCommand(IList<ICommand> commandList)
        {
            m_CommandList = commandList;
        }

        public override void RunAsync(ControllerSet controllerSet)
        {
            foreach (var command in m_CommandList)
            {
                var task = Task.Run(() =>
                {
                    if (command is IAsyncCommand)
                    {
                        (command as IAsyncCommand).CancellationToken = this.CancellationToken;
                    }

                    command.Run(controllerSet);
                });
                m_TaskList.Add(task);
            }
        }

        public override void Wait()
        {
            foreach (var task in m_TaskList)
            {
                task.TryWait(CancellationToken);
            }
        }
    }

    public class RepeatCommand : AbstractAsyncCommand
    {
        private IList<ICommand> m_CommandList;
        private int m_RepeatCount;

        private Task m_Task;

        public RepeatCommand(IList<ICommand> commandList, int repeatCount)
        {
            m_CommandList = commandList;
            m_RepeatCount = repeatCount;
        }

        public override void RunAsync(ControllerSet controllerSet)
        {
            m_Task = Task.Run(() =>
            {
                for (int i = 0; i < m_RepeatCount; i++)
                {
                    foreach (var command in m_CommandList)
                    {
                        if (CancellationToken.IsCancellationRequested)
                        {
                            break;
                        }

                        if (command is IAsyncCommand)
                        {
                            (command as IAsyncCommand).CancellationToken = this.CancellationToken;
                        }

                        command.Run(controllerSet);
                    }
                }
            });
        }

        public override void Wait()
        {
            m_Task.TryWait(CancellationToken);
        }
    }

    public class LoopCommand : AbstractAsyncCommand
    {
        private IList<ICommand> m_CommandList;

        private Task m_Task;

        public LoopCommand(IList<ICommand> commandList)
        {
            m_CommandList = commandList;
        }

        public override void RunAsync(ControllerSet controllerSet)
        {
            m_Task = Task.Run(() =>
            {
                while (true)
                {
                    foreach (var command in m_CommandList)
                    {
                        if (CancellationToken.IsCancellationRequested)
                        {
                            return;
                        }

                        if (command is IAsyncCommand)
                        {
                            (command as IAsyncCommand).CancellationToken = this.CancellationToken;
                        }

                        command.Run(controllerSet);
                    }
                }
            });
        }

        public override void Wait()
        {
            m_Task.TryWait(CancellationToken);
        }
    }

    public class WaitCommand : AbstractAsyncCommand
    {
        private int m_WaitMilliseconds;
        private Task m_Task;

        public WaitCommand(int waitMilliseconds)
        {
            m_WaitMilliseconds = waitMilliseconds;
        }

        public override void RunAsync(ControllerSet controllerSet)
        {
            m_Task = Task.Run(async () =>
            {
                await Task.Delay(m_WaitMilliseconds, CancellationToken);
            });
        }

        public override void Wait()
        {
            m_Task.TryWait(CancellationToken);
        }
    }
}
