﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Nintendo.NintendoSdkVsExtension.Base;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Input;

namespace Nintendo.NintendoSdkVsExtension.ViewModels
{
    /// <summary>
    /// 1 つの構成 (グリッドの行) を表す VM
    /// </summary>
    public class SdkConfigurationSettingsItem : Common.BindableBase, IDisposable
    {
        private bool _disposedValue = false;
        private Models.SdkVcProject _vcProject;
        private Models.SdkVcProjectConfiguration _vcProjectConfiguration;
        private SdkConfigurationDialogViewModel _parentViewModel;

        public bool IsSdkEnabled
        {
            get { return this._vcProjectConfiguration.IsSdkEnabled; }
            set { this._vcProjectConfiguration.IsSdkEnabled = value; }
        }

        public SdkRoot SdkRoot
        {
            get { return this._vcProjectConfiguration.SdkRoot; }
            set { this._vcProjectConfiguration.SdkRoot = value;  }
        }

        public string Spec
        {
            get { return this._vcProjectConfiguration.Spec; }
            set { this._vcProjectConfiguration.Spec = value; }
        }

        public SdkBuildType BuildType
        {
            get { return this._vcProjectConfiguration.BuildType; }
            set { this._vcProjectConfiguration.BuildType = value; }
        }

        public string ProjectName => Path.GetFileNameWithoutExtension(this._vcProject.ProjectFilePath);
        public string ProjectFilePath => this._vcProject.ProjectFilePath;
        public string Platform => this._vcProjectConfiguration.Platform;
        public string Configuration => this._vcProjectConfiguration.Configuration;
        public ICommand EditSdkRootCommand => this._parentViewModel.EditSdkRootCommand;
        public IEnumerable<string> AvailableSpecs => this._vcProjectConfiguration.AvailableSpecs;
        public IEnumerable<SdkBuildType> AvailableBuildTypes => BaseConstants.SdkBuildTypes;

        public SdkConfigurationSettingsItem(SdkConfigurationDialogViewModel parentViewModel, Models.SdkVcProject vcProject, Models.SdkVcProjectConfiguration vcProjectConfiguration)
        {
            this._parentViewModel = parentViewModel;
            this._vcProject = vcProject;
            this._vcProjectConfiguration = vcProjectConfiguration;
            this._vcProject.PropertyChanged += OnVcProjectPropertyChanged;
            this._vcProjectConfiguration.PropertyChanged += OnVcProjectConfigurationPropertyChanged;
        }

        #region IDisposable Support
        protected virtual void Dispose(bool disposing)
        {
            if (!_disposedValue)
            {
                if (disposing)
                {
                    this._vcProject.PropertyChanged -= OnVcProjectPropertyChanged;
                    this._vcProjectConfiguration.PropertyChanged -= OnVcProjectConfigurationPropertyChanged;
                }
                _disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion

        private void OnVcProjectPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            switch (e.PropertyName)
            {
                case nameof(this._vcProject.ProjectFilePath):
                    this.RaisePropretyChanged(nameof(this.ProjectFilePath));
                    this.RaisePropretyChanged(nameof(this.ProjectName));
                    break;
            }
        }

        private void OnVcProjectConfigurationPropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e)
        {
            switch (e.PropertyName)
            {
                case nameof(this._vcProjectConfiguration.Platform):
                    this.RaisePropretyChanged(nameof(this.Platform));
                    break;
                case nameof(this._vcProjectConfiguration.Configuration):
                    this.RaisePropretyChanged(nameof(this.Configuration));
                    break;
                case nameof(this._vcProjectConfiguration.IsSdkEnabled):
                    this.RaisePropretyChanged(nameof(this.IsSdkEnabled));
                    break;
                case nameof(this._vcProjectConfiguration.Spec):
                    this.RaisePropretyChanged(nameof(this.Spec));
                    break;
                case nameof(this._vcProjectConfiguration.BuildType):
                    this.RaisePropretyChanged(nameof(this.BuildType));
                    break;
                case nameof(this._vcProjectConfiguration.SdkRoot):
                    this.RaisePropretyChanged(nameof(this.SdkRoot));
                    break;
                case nameof(this._vcProjectConfiguration.AvailableSpecs):
                    this.RaisePropretyChanged(nameof(this.AvailableSpecs));
                    break;
            }
        }
    }

    /// <summary>
    /// SDK 構成マネージャダイアログを表す VM
    /// </summary>
    public class SdkConfigurationDialogViewModel : OkCancelDialogViewModelBase, IDisposable
    {
        private bool _disposedValue = false;
        private IEnumerable<SdkConfigurationSettingsItem> _items;
        private ICommand _editSdkRootCommand;
        private ChooseSdkRootViewModel _chooseSdkRootViewModel;

        public IEnumerable<SdkConfigurationSettingsItem> Items => this._items;

        /// <summary>
        /// SDK ルートを編集するコマンド （"..." ボタン向け）
        /// </summary>
        public ICommand EditSdkRootCommand =>
            this._editSdkRootCommand ?? (this._editSdkRootCommand = new Common.RelayCommand(OnEditSdkRoot));

        public ChooseSdkRootViewModel ChooseSdkRootViewModel
        {
            get { return _chooseSdkRootViewModel; }
            set { SetProperty(ref _chooseSdkRootViewModel, value); }
        }

        public SdkConfigurationDialogViewModel(Models.SdkSolution solution)
        {
            this._items = solution.VcProjects
                .SelectMany(v => v.ProjectConfigurations.Select(c => new SdkConfigurationSettingsItem(this, v, c)))
                .OrderBy(x => Tuple.Create(x.Configuration, x.Platform))
                .ToArray();
        }

        #region IDisposable Support
        protected virtual void Dispose(bool disposing)
        {
            if (!_disposedValue)
            {
                if (disposing)
                {
                    foreach (var x in this._items)
                    {
                        x.Dispose();
                    }
                }
                _disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion

        private void OnEditSdkRoot(object parameter)
        {
            var item = parameter as SdkConfigurationSettingsItem;
            if (item == null)
            {
                return;
            }

            var model = new Models.ChooseSdkRootDialogModel(item.SdkRoot);
            var vm = new ViewModels.ChooseSdkRootViewModel(model);
            vm.OkPressed += (s, e) =>
            {
                // TODO: SDK ルートが変わったら、スペックが範囲外になる可能性がある。
                // 勝手に修正するよりは、範囲外の値に対して 赤(!) を出して修正させるのがよい
                item.SdkRoot = model.ToSdkRoot();
            };

            // ChooseSdkRootViewModel
            this.ChooseSdkRootViewModel = vm;
        }
    }
}
