﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Base
{
    // spec は SDK から読み取らなければならない
    // ユーザーにも公開できるような xml ファイルか何かを用意したいが、とりあえず VcProjectUtility を見る

    /// <summary>
    /// あるパスにインストールされている SDK の情報
    /// </summary>
    public class SdkInformation
    {
        public static readonly SdkInformation Empty = new SdkInformation();
        public static readonly IEnumerable<string> DefaultSpecs = new string[] { "Generic" };

        public string RootPath { get; }
        public IEnumerable<string> Specs { get; }
        public string Version { get; }

        private SdkInformation()
        {
            this.RootPath = string.Empty;
        }

        public SdkInformation(SdkRoot sdkRoot)
            : this(sdkRoot.ToEvaluatedPath())
        {
        }

        public SdkInformation(string rootPath)
        {
            if (!IsValidSdkRoot(rootPath))
            {
                throw new InvalidSdkRootException(rootPath);
            }

            this.RootPath = rootPath;

            string vcProjectUtilityPath = GetVcProjectUtilityPath(rootPath);
            var specFiles = Directory.GetFiles(vcProjectUtilityPath, "NintendoSdkSpec_*.props");
            this.Specs = specFiles.Select(x => Path.GetFileNameWithoutExtension(x).Replace("NintendoSdkSpec_", string.Empty)).ToArray();

            // TODO: バージョンの取得
            this.Version = "not implemented";
        }

        public static bool IsValidSdkRoot(SdkRoot sdkRoot)
        {
            return IsValidSdkRoot(sdkRoot.ToEvaluatedPath());
        }

        public static bool IsValidSdkRoot(string rootPath)
        {
            // TODO: 正しい SDKRoot であることのまともな確認
            string vcProjectUtilityPath = GetVcProjectUtilityPath(rootPath);
            return (File.Exists(Path.Combine(rootPath, "NintendoSdkRootMark")) && Directory.Exists(vcProjectUtilityPath));
        }

        private static string GetVcProjectUtilityPath(string rootPath) =>
            Path.Combine(rootPath, @"Build\VcProjectUtility");
    }
}
