﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;
using System.Diagnostics;

namespace IFFRefGen.Src
{
    // HTML 要素リスト
    public class HtmlElementList
    {
        // テンプレートファイルの名前
        private readonly string TemplateFileName = "TmplElementList.html";
        // 保存パスのフォーマット文字列
        private readonly string SavePathFormat = "html/reference/element_list.html";

        // 要素リストノードの XPath
        private readonly string ElementListNodeXPath = "//div[@id='contents_list']";
        // 要素のフォーマット文字列
        private readonly string ElementFormat = "<{0}>";
        // リンクのフォーマット文字列
        private readonly string HrefFormat = "elements/{0}.html";

        // コンストラクタ
        public HtmlElementList()
        {
            this.Template = new XmlDocument();
        }

        // 初期化
        public void Initialize()
        {
            // テンプレートの読み込み
            string loadPath = GApp.Config.HtmlOutputConfig.TemplateElementFolder + this.TemplateFileName;
            this.Template.Load(loadPath);
        }

        // 生成
        public void Generate()
        {
            XsdElement[] elements = GXsd.Elements.ToArray<XsdElement>();
            Array.Sort<XsdElement>(elements,
                delegate(XsdElement lhs, XsdElement rhs)
                    { return lhs.Name.CompareTo(rhs.Name); });

            XmlNode contentsNode = this.Template.SelectSingleNode(this.ElementListNodeXPath);

            // インデックステーブル
            XmlElement indexTable = this.Template.CreateElement("table");
            contentsNode.AppendChild(indexTable);
            indexTable.SetAttribute("id", "index_table");
            XmlElement indexThead = this.Template.CreateElement("thead");
            indexTable.AppendChild(indexThead);
            XmlElement indexTr = this.Template.CreateElement("tr");
            indexThead.AppendChild(indexTr);

            XmlElement rootUl = this.Template.CreateElement("ul");
            contentsNode.AppendChild(rootUl);

            XmlElement letterUl = null;
            char firstLetter = ' ';
            int indexTableCount = 0;
            foreach (XsdElement element in elements)
            {
                // 頭文字処理
                if (element.Name[0] != firstLetter)
                {
                    firstLetter = element.Name[0];

                    XmlElement li = this.Template.CreateElement("li");
                    string letter = firstLetter.ToString().ToUpper();
                    li.InnerXml = "<strong>" + letter +
                        "　<a href=\"#index_table\" class=\"disable_chm\">↑</a></strong>";
                    string id = "id_" + letter;
                    li.SetAttribute("id", id);
                    rootUl.AppendChild(li);

                    letterUl = this.Template.CreateElement("ul");
                    li.AppendChild(letterUl);

                    // インデックステーブルへの追加
                    XmlElement indexTh = this.Template.CreateElement("th");
                    indexTr.AppendChild(indexTh);
                    indexTh.InnerXml =
                        "<strong><a href=\"#" + id + "\">" + letter + "</a></strong>";
                    indexTableCount++;
                    if ((indexTableCount % 8) == 0)
                    {
                        indexTr = this.Template.CreateElement("tr");
                        indexThead.AppendChild(indexTr);
                    }
                }

                // 要素の生成
                GenerateElement(element, letterUl);
            }

            // 保存
            string savePath = System.IO.Path.Combine(GApp.Config.HtmlOutputConfig.OutputFolder, this.SavePathFormat);

            GHtml.Save(this.Template, savePath);
        }

        // 要素の生成
        private void GenerateElement(XsdElement element, XmlNode parentNode)
        {
            XmlElement li = this.Template.CreateElement("li");
            parentNode.AppendChild(li);

            XmlElement a = this.Template.CreateElement("a");
            a.InnerText = string.Format(this.ElementFormat, element.Name);
            {
                XmlAttribute href = this.Template.CreateAttribute("href");
                href.InnerText = string.Format(this.HrefFormat, element.Name);
                a.Attributes.Append(href);
                XmlAttribute target = this.Template.CreateAttribute("target");
                target.InnerText = "main";
                a.Attributes.Append(target);
            }
            li.AppendChild(a);
        }

        // テンプレート
        public XmlDocument Template { get; private set; }
    }
}
