﻿using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Xml;
using System.Xml.Serialization;

namespace IFFRefGen.Src
{
    // リファレンスルートの設定
    public class ReferenceRootConfig
    {
        public string Title { get; set; }
        public string RootElementName { get; set; }
    }

    // HTML 出力設定
    public class HtmlOutputConfig
    {
        // コピーフォルダ
        public string TemplatePageFolder { get; set; }
        // テンプレートフォルダ
        public string TemplateElementFolder { get; set; }
        // 出力フォルダ
        public string OutputFolder { get; set; }
    }

    // コンフィグ
    public class Config
    {
        // スキーマフォルダ
        public string SchemaFolder { get; set; }
        // スキーマファイル
        public string SchemaFile { get; set; } = string.Empty;
        // スキーマ内のルートエレメント
        public string SchemaRootElement { get; set; }
        // ドキュメントフォルダ
        public string ContentFolder { get; set; }
        // サンプルフォルダ
        public string SampleFolder { get; set; }
        // リファレンスルートの設定
        public ReferenceRootConfig[] ReferenceRootConfigs { get; set; }
        // 付加情報要素
        [XmlArrayItem("ElementName")]
        public string[] AdditionalInformationElements { get; set; }
        // サンプルソート用テーブル
        [XmlArrayItem("Extension")]
        public string[] SampleSortTable { get; set; }
        // HTML 出力設定
        public HtmlOutputConfig HtmlOutputConfig { get; set; }

        // 読み込み
        public static Config Read(string path)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(Config));
            Config config;
            XmlReaderSettings readerSettings = new XmlReaderSettings();
            readerSettings.CloseInput = true;
            using (XmlReader reader = XmlReader.Create(
                new FileStream(path, FileMode.Open), readerSettings))
            {
                config = (Config)serializer.Deserialize(reader);
            }

            string rootFolder = System.IO.Path.GetDirectoryName(path);
            config.SampleFolder = System.IO.Path.Combine(rootFolder, config.SampleFolder);
            config.ContentFolder = System.IO.Path.Combine(rootFolder, config.ContentFolder);
            config.SchemaFolder = System.IO.Path.Combine(rootFolder, config.SchemaFolder);

            if (config.HtmlOutputConfig.OutputFolder != null)
            {
                config.HtmlOutputConfig.OutputFolder = System.IO.Path.Combine(rootFolder, config.HtmlOutputConfig.OutputFolder);
            }

            if (config.HtmlOutputConfig.TemplateElementFolder != null)
            {
                config.HtmlOutputConfig.TemplateElementFolder = System.IO.Path.Combine(rootFolder, config.HtmlOutputConfig.TemplateElementFolder);
            }

            if (config.HtmlOutputConfig.TemplatePageFolder != null)
            {
                config.HtmlOutputConfig.TemplatePageFolder = System.IO.Path.Combine(rootFolder, config.HtmlOutputConfig.TemplatePageFolder);
            }

            return config;
        }

        // 書き出し
        public static void Write(Config config, string path)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(Config));
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            settings.IndentChars = "\t";
            settings.NewLineOnAttributes = true;
            settings.CloseOutput = true;
            using (XmlWriter writer = XmlWriter.Create(path, settings))
            {
                serializer.Serialize(writer, config);
            }
        }
    }
}
