﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Xml;
using System.Xml.Serialization;

namespace GllSourceGenerator
{
    class CommandLineOption
    {
        public List<string> Inputs { get; set; }

        [Option( "--output-header", "Specify output header file path." )]
        public string OutputHeader { get; set; }

        [Option( "--output-source", "Specify output source file path." )]
        public string OutputSource { get; set; }

        [Option( "--add-header-code", "Add arbitrary code which is inserted into header file." )]
        public List< string > AddHeaderCode { get; set; }

        [Option( "--add-source-code", "Add arbitrary code which is inserted into source file." )]
        public List< string > AddSourceCode { get; set; }

        [Option( "--xsd", "Specify xml schema file path." )]
        public string XsdFile { get; set; }

        [Option( "--api", "Specify target api name." )]
        public string Api { get; set; }
    }

    class Program
    {
        static void Main( string[] args )
        {
            CommandLineOption options = new CommandLineOption(); ;
            if( !CommandLineParser.Parse( options, args ) )
            {
                Console.Error.WriteLine( "failed to parse command line option." );
                return;
            }
            if( options.Inputs.Count < 0 )
            {
                Console.Error.WriteLine( "no input file." );
                return;
            }
            if( options.Inputs.Count > 1 )
            {
                Console.Error.WriteLine( "only 1 input file is allowed." );
                return;
            }
            if( string.IsNullOrEmpty( options.Api ) )
            {
                Console.Error.WriteLine( "--api is required." );
            }

            Registry registry = null;
            using( FileStream fs = new FileStream( options.Inputs[ 0 ], FileMode.Open ) )
            {
                if( options.XsdFile == null )
                {
                    using( XmlReader xmlReader = XmlReader.Create( fs ) )
                    {
                        XmlSerializer serializer = new XmlSerializer( typeof( Registry ) );
                        registry = (Registry)serializer.Deserialize( xmlReader );
                    }
                }
                else
                {
                    using( XmlReader xsdReader = XmlReader.Create( options.XsdFile ) )
                    {
                        System.Xml.Schema.XmlSchema schema = System.Xml.Schema.XmlSchema.Read( xsdReader, null );
                        XmlReaderSettings settings = new XmlReaderSettings();
                        settings.CloseInput = true;
                        settings.ValidationType = ValidationType.Schema;
                        settings.Schemas.Add( schema );
                        using( XmlReader xmlReader = XmlReader.Create( fs, settings ) )
                        {
                            XmlSerializer serializer = new XmlSerializer( typeof( Registry ) );
                            registry = (Registry)serializer.Deserialize( xmlReader );
                        }
                    }
                }
            }
            registry.Resolve( options.Api );

            GlSourceGenerator generator = new GlSourceGenerator( registry );
            generator.AdditionalHeaderCode = options.AddHeaderCode;
            generator.AdditionalSourceCode = options.AddSourceCode;
            generator.HeaderFileName = options.OutputHeader;
            generator.Api = options.Api;
            Action<string, string> WriteFile = ( path, data ) =>
            {
                var directoryName = Path.GetDirectoryName( path );
                if( !Directory.Exists( directoryName ) )
                {
                    Directory.CreateDirectory( directoryName );
                }
                using( StreamWriter sw = new StreamWriter( path, false, Encoding.UTF8 ) )
                {
                    sw.Write( data );
                }
            };
            if( !string.IsNullOrEmpty( options.OutputHeader ) )
            {
                WriteFile( options.OutputHeader, generator.CreateHeader() );
            }
            if( !string.IsNullOrEmpty( options.OutputSource ) )
            {
                WriteFile( options.OutputSource, generator.CreateSource() );
            }
        }
    }
}
