﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GllSourceGenerator
{
    class GlSourceGenerator
    {
        private const string Copyright = @"/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/";

        private Registry m_Registry = null;

        private HashSet<Type> m_DefinedTypes = null;
        private HashSet<Command> m_DefinedCommands = null;
        private HashSet<Enum> m_DefinedEnums = null;

        public List< string > AdditionalHeaderCode { get; set; }

        public List< string > AdditionalSourceCode { get; set; }

        public string HeaderFileName { get; set; }

        public string Api { get; set; }

        public GlSourceGenerator( Registry registry )
        {
            m_Registry = registry;
        }

        public string CreateHeader()
        {
            m_DefinedTypes = new HashSet<Type>();
            m_DefinedEnums = new HashSet<Enum>();
            m_DefinedCommands = new HashSet<Command>();

            StringBuilder sb = new StringBuilder();
            sb.AppendLine( $@"{Copyright}

/**
* @file {System.IO.Path.GetFileName( HeaderFileName )}
*/

#pragma once

#include <stddef.h>

#include <nn/nn_Macro.h>
" );
            if( Api == "wgl" )
            {
                sb.AppendLine( "#include <nn/nn_Windows.h>" );
            }
            if( Api == "gl" )
            {
                sb.AppendLine( $@"#if defined( __gl_h_ ) || defined( __GL_H_ )
    #error Please include this file before ""gl.h""
#endif

#define __gl_h_
#define __GL_H_

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    #include <nn/nn_Windows.h>
    #define NN_GLL_GL_APIENTRY APIENTRY
    #define NN_GLL_GL_API WINGDIAPI
#endif

#if !defined( NN_GLL_GL_APIENTRY )
    #define NN_GLL_GL_APIENTRY
#endif
#if !defined( NN_GLL_GL_API )
    #define NN_GLL_GL_API
#endif
" );
            }
            if( AdditionalHeaderCode != null )
            {
                AdditionalHeaderCode.ForEach( code => sb.AppendLine( code ) );
            }

            sb.AppendLine( $@"
#ifdef __cplusplus
extern ""C"" {{
#endif

NN_GLL_API nngllResult {GetInitializeFunctionName()}() NN_NOEXCEPT;" );
            foreach( var feature in EnumerateTargetFeatures() )
            {
                WriteComment( sb, feature.comment );
                WriteFeatureHeader( sb, feature );
            }
            foreach( var extension in EnumerateTargetExtensions() )
            {
                WriteComment( sb, extension.comment );
                WriteFeatureHeader( sb, extension );
            }

            sb.AppendLine( @"
#ifdef __cplusplus
} // extern ""C""
#endif
" );
            return sb.ToString();
        }

        public string CreateSource()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine( Copyright );
            sb.AppendLine( "#include <algorithm>" ).AppendLine();
            if( !string.IsNullOrEmpty( HeaderFileName ) )
            {
                sb.AppendLine( $"#include <nn/gll/{System.IO.Path.GetFileName( HeaderFileName )}>" );
            }
            if( AdditionalSourceCode != null )
            {
                AdditionalSourceCode.ForEach( code => sb.AppendLine( code ) );
            }
            sb.AppendLine();

            // 関数ポインタの定義
            WriteCommandVariableDefinitions( sb );

            sb.AppendLine( "namespace {" ).AppendLine();

            // 機能がサポートされるかどうかの真偽値
            WriteIsSupportedVariableDefinitions( sb );

            // 指定された拡張がサポートされるかどうかの判定関数
            WriteSetExtensionEnabledFunction( sb );

            // 機能ごとのロード関数
            WriteLoadingFunctions( sb );

            // ロード関数
            WriteLoadingFunction( sb );

            // 機能がサポートされるかどうかの判定関数
            WriteCheckSupportedFeaturesFunction( sb );

            sb.AppendLine( "} // anonymous namespace" ).AppendLine();

            sb.AppendLine( "namespace nn {" );
            sb.AppendLine( "namespace gll {" );
            sb.AppendLine( "namespace detail {" ).AppendLine();

            // ミドルウェア情報
            sb.AppendLine( "void UseMiddleware() NN_NOEXCEPT;" ).AppendLine();

            sb.AppendLine( "} // detail" );
            sb.AppendLine( "} // gll" );
            sb.AppendLine( "} // nn" ).AppendLine();

            // 機能がサポートされるかどうかの取得関数
            WriteIsSupportedFunctions( sb );

            // メイン関数
            WriteInitializeFunction( sb );

            return sb.ToString();
        }

        private void WriteFeatureHeader( StringBuilder sb, IFeature feature )
        {
            sb.AppendLine( $@"
#if !defined({feature.name})
#define {feature.name}
" );

            var commands = EnumerateRequiredCommands( feature ).Where(
                command => m_DefinedCommands.Add( command ) ).ToList();
            // 型定義
            foreach( var type in EnumerateRequiredTypes( feature ) )
            {
                WriteType( sb, type );
            }
            // 関数に必要な型定義
            foreach( var command in commands )
            {
                WriteCommandRequireType( sb, command );
            }
            // 列挙定義
            foreach( var @enum in EnumerateRequiredEnums( feature ).Where( @enum => m_DefinedEnums.Add( @enum ) ) )
            {
                WriteEnum( sb, @enum );
            }
            // 関数型定義
            foreach( var command in commands )
            {
                WriteCommandTypeDefinition( sb, command );
            }
            if( IsVersion10( feature ) )
            {
                if( Api == "gl" )
                {
                    sb.AppendLine( "#if defined( NN_BUILD_CONFIG_OS_WIN32 )" );
                    foreach( var command in commands )
                    {
                        WriteCommandDeclaration( sb, command );
                    }
                    sb.AppendLine( "#else" );
                }
                else
                {
                    sb.AppendLine( "#if 0" );
                }
            }
            // 関数マクロ
            foreach( var command in commands )
            {
                WriteCommandPreprocessorDefinition( sb, command );
            }
            // 関数ポインタ変数の宣言
            foreach( var command in commands )
            {
                WriteCommandVariableDeclaration( sb, command );
            }
            if( IsVersion10( feature ) )
            {
                sb.AppendLine( "#endif // defined( NN_BUILD_CONFIG_OS_WIN32 )" );
            }
            sb.AppendLine( $"NN_GLL_API GLboolean {GetIsSupportedFunctionName( feature )}();" );
            sb.AppendLine( $"#endif // {feature.name}" );
        }

        private void WriteIsSupportedVariableDefinitions( StringBuilder sb )
        {
            foreach( var feature in EnumerateTargetFeaturesAndExtensions() )
            {
                WriteIsSupportedVariableDefinition( sb, feature );
            }
            sb.AppendLine();
        }

        private void WriteCommandVariableDefinitions( StringBuilder sb )
        {
            m_DefinedCommands = new HashSet<Command>();
            foreach( var feature in EnumerateTargetFeaturesAndExtensions() )
            {
                sb.AppendLine( "// " + feature.name );
                if( IsVersion10( feature ) )
                {
                    sb.AppendLine( "#if !defined( NN_BUILD_CONFIG_OS_WIN32 )" );
                }
                foreach( var command in EnumerateRequiredCommands( feature ).Where( command => m_DefinedCommands.Add( command ) ) )
                {
                    WriteCommandVariableDefinition( sb, command );
                }
                if( IsVersion10( feature ) )
                {
                    sb.AppendLine( "#endif // !defined( NN_BUILD_CONFIG_OS_WIN32 )" );
                }
                sb.AppendLine();
            }
        }

        private void WriteSetExtensionEnabledFunction( StringBuilder sb )
        {
            var extensionsList = EnumerateTargetExtensions().ToList();
            extensionsList.Sort( ( lhs, rhs ) => string.Compare( lhs.name, rhs.name ) );
            sb.AppendLine( "static const char* ExtensionNames[] = {" );
            for( int idxExtension = 0; idxExtension < extensionsList.Count; ++idxExtension )
            {
                sb.Append( $"    \"{extensionsList[ idxExtension ].name}\"" );
                if( idxExtension != extensionsList.Count - 1 )
                {
                    sb.Append( "," );
                }
                sb.AppendLine();
            }
            sb.AppendLine( "};" ).AppendLine();
            sb.AppendLine( "static GLboolean* const pIsSupported[] = {" );
            for( int idxExtension = 0; idxExtension < extensionsList.Count; ++idxExtension )
            {
                sb.Append( $"    &{GetIsSupportedVariableName( extensionsList[ idxExtension ] )}" );
                if( idxExtension != extensionsList.Count - 1 )
                {
                    sb.Append( "," );
                }
                sb.AppendLine();
            }
            sb.AppendLine( "};" );
            sb.AppendLine( @"
int GetExtensionIndex( const char* pExtensionName )
{
    struct Comp
    {
        bool operator()( const char* lhs, const char* rhs ) const
        {
            return strcmp( lhs, rhs ) < 0;
        }
    };
    auto pStart = ExtensionNames;
    auto pEnd = ExtensionNames + NN_ARRAY_SIZE( ExtensionNames );
    auto pFound = std::lower_bound( pStart, pEnd, pExtensionName, Comp() );
    if( pFound != pEnd && strcmp( *pFound, pExtensionName ) == 0 )
    {
        return static_cast< int >( std::distance( pStart, pFound ) );
    }
    return -1;
}

void SetExtensionEnabled( const char* pExtensionName )
{
    int idxExtension = GetExtensionIndex( pExtensionName );
    if( idxExtension >= 0 )
    {
        *( pIsSupported[ idxExtension ] ) = GL_TRUE;
    }
}

void CheckSpaceSeparatedExtensions( const char* pExtensions )
{
    char buf[ 128 ] = {};
    for( ; ; )
    {
        auto pNextSpace = strchr( pExtensions, ' ' );
        if( pNextSpace == nullptr )
        {
            SetExtensionEnabled( pExtensions );
            break;
        }
        auto length = pNextSpace - pExtensions;
        memcpy( buf, pExtensions, length * sizeof( char ) );
        buf[ length ] = '\0';
        SetExtensionEnabled( buf );
        pExtensions = pNextSpace + 1;
    }
}

void ClearSupportedFeatures()
{
    for( int idxFeature = 0; idxFeature < NN_ARRAY_SIZE( pIsSupported ); ++idxFeature )
    {
        *( pIsSupported[ idxFeature ] ) = GL_FALSE;
    }
}
" );
        }

        private void WriteLoadingFunctions( StringBuilder sb )
        {
            m_DefinedCommands = new HashSet<Command>();
            foreach( var feature in EnumerateTargetFeaturesAndExtensions() )
            {
                sb.AppendLine( $@"bool {GetLoadFunctionName( feature )}()
{{
    bool ret = true;" );
                if( IsVersion10( feature ) )
                {
                    sb.AppendLine( "#if !defined( NN_BUILD_CONFIG_OS_WIN32 )" );
                }
                foreach( var command in EnumerateRequiredCommands( feature ).Where( command => m_DefinedCommands.Add( command ) ) )
                {
                    sb.AppendLine( $"    ret &= ({GetCommandVariableName( command )} = reinterpret_cast< {GetCommandTypeName( command )} >( NN_GLL_GET_PROC_ADDRESS( \"{command.proto.name}\" ) ) ) != NULL;" );
                }
                if( IsVersion10( feature ) )
                {
                    sb.AppendLine( "#endif // !defined( NN_BUILD_CONFIG_OS_WIN32 )" );
                }
                sb.AppendLine( $@"
    return ret;
}} // NOLINT(impl/function_size)
" );
            }
        }

        private void WriteLoadingFunction( StringBuilder sb )
        {
            sb.AppendLine( "nngllResult InitializeFunctions()" );
            sb.AppendLine( "{" );
            foreach( var feature in EnumerateTargetFeaturesAndExtensions() )
            {
                sb.AppendLine( $"    {GetLoadFunctionName( feature )}();" );
            }
            sb.AppendLine().AppendLine( "    return nngllResult_Succeeded;" );
            sb.AppendLine( "} // NOLINT(impl/function_size)" ).AppendLine();
        }

        private void WriteIsSupportedFunctions( StringBuilder sb )
        {
            foreach( var feature in EnumerateTargetFeaturesAndExtensions() )
            {
                sb.AppendLine( $@"GLboolean {GetIsSupportedFunctionName( feature )}()
{{
    return {GetIsSupportedVariableName( feature )};
}}
" );
            }
        }

        private void WriteInitializeFunction( StringBuilder sb )
        {
            sb.AppendLine( $@"
nngllResult {GetInitializeFunctionName()}() NN_NOEXCEPT
{{
#if !defined( NN_GLL_CONFIG_DLL )
    nn::gll::detail::UseMiddleware();
#endif

    nngllResult result;
    if( ( result = InitializeFunctions() ) != nngllResult_Succeeded )
    {{
        return result;
    }}

    ClearSupportedFeatures();
    if( ( result = CheckSupportedFeatures() ) != nngllResult_Succeeded )
    {{
        return result;
    }}

    return nngllResult_Succeeded;
}}" );
        }

        private void WriteCheckSupportedFeaturesFunction( StringBuilder sb )
        {
            switch( Api  )
            {
                case "gl":
                    WriteCheckSupportedFeaturesFunctionGl( sb );
                    break;
                case "wgl":
                    WriteCheckSupportedFeaturesFunctionWgl( sb );
                    break;
                default:
                    WriteCheckSupportedFeaturesFunctionGl( sb );
                    break;
            }
        }

        private void WriteCheckSupportedFeaturesFunctionGl( StringBuilder sb )
        {
            // GL_MAJOR_VERSION, GL_MINOR_VERSION は GL 3.0 から
            // glGetStringi は GL 3.0 から
            sb.AppendLine( @"nngllResult CheckSupportedFeatures()
{
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    auto pGlGetString = glGetString;
#else
    auto pGlGetString = reinterpret_cast< glGetStringType >( NN_GLL_GET_PROC_ADDRESS( ""glGetString"" ) );
#endif
    if( pGlGetString == nullptr )
    {
        return nngllResult_FailedToGetGlVersion;
    }
    auto version = reinterpret_cast< const char* >( pGlGetString( GL_VERSION ) );
    char* pDot = nullptr;
    int major = static_cast< int >( strtol( version, &pDot, 10 ) );
    if( pDot == nullptr )
    {
        return nngllResult_InvalidGlVersion;
    }
    int minor = static_cast< int >( strtol( pDot + 1, nullptr, 10 ) );
    if( major < 1 || minor < 0 )
    {
        return nngllResult_InvalidGlVersion;
    }
" );
            foreach( var feature in EnumerateTargetFeatures() )
            {
                var numbers = feature.number.Split( '.' );
                var major = Convert.ToInt32( numbers[ 0 ] );
                var minor = Convert.ToInt32( numbers[ 1 ] );
                sb.AppendLine( $"    {GetIsSupportedVariableName( feature )} = ( ( major > {major} ) || ( major == {major} && minor >= {minor} ) ) ? GL_TRUE : GL_FALSE;" );
            }

            sb.AppendLine( @"
    if( major >= 3 )
    {
#if defined( NN_BUILD_CONFIG_OS_WIN32 )
        auto pGlGetIntegerv = glGetIntegerv;
#else
        auto pGlGetIntegerv = reinterpret_cast< glGetIntegervType >( NN_GLL_GET_PROC_ADDRESS( ""glGetIntegerv"" ) );
#endif
        auto pGlGetStringi = reinterpret_cast< glGetStringiType >( NN_GLL_GET_PROC_ADDRESS( ""glGetStringi"" ) );
        if( pGlGetIntegerv == nullptr || pGlGetStringi == nullptr )
        {
            return nngllResult_FailedToGetGlExtensions;
        }
        int extensionCount;
        pGlGetIntegerv( GL_NUM_EXTENSIONS, &extensionCount );
        for( int idxExtension = 0; idxExtension < extensionCount; ++idxExtension )
        {
            auto pExtension = reinterpret_cast< const char* >( pGlGetStringi( GL_EXTENSIONS, idxExtension ) );
            SetExtensionEnabled( pExtension );
        }
    }
    else
    {
        auto pExtensions = reinterpret_cast< const char* >( pGlGetString( GL_EXTENSIONS ) );
        CheckSpaceSeparatedExtensions( pExtensions );
    }

    return nngllResult_Succeeded;
}
" );
        }

        private void WriteCheckSupportedFeaturesFunctionWgl( StringBuilder sb )
        {
            sb.AppendLine( @"nngllResult CheckSupportedFeatures()
{
    g_WGL_VERSION_1_0 = GL_TRUE;
    const char* pExtensions = nullptr;
    auto pWglGetExtensionsStringARB = reinterpret_cast< wglGetExtensionsStringARBType >( NN_GLL_GET_PROC_ADDRESS( ""wglGetExtensionsStringARB"" ) );
    if( pWglGetExtensionsStringARB )
    {
        pExtensions = pWglGetExtensionsStringARB( wglGetCurrentDC() );
    }
    else
    {
        auto pWglGetExtensionsStringEXT = reinterpret_cast< wglGetExtensionsStringEXTType >( NN_GLL_GET_PROC_ADDRESS( ""wglGetExtensionsStringEXT"" ) );
        if( pWglGetExtensionsStringEXT )
        {
            pExtensions = pWglGetExtensionsStringEXT();
        }
    }
    if( pExtensions )
    {
        CheckSpaceSeparatedExtensions( pExtensions );
    }

    return nngllResult_Succeeded;
}
" );
        }

        private void WriteIsSupportedVariableDefinition( StringBuilder sb, IFeature feature )
        {
            sb.AppendLine( $"GLboolean {GetIsSupportedVariableName( feature )} = GL_FALSE;" );
        }

        private void WriteType( StringBuilder sb, Type type )
        {
            if( type == null )
            {
                return;
            }
            WriteType( sb, type.Requires );
            // インクルードは先頭で行う
            if( !type.Value.StartsWith( "#include" ) && m_DefinedTypes.Add( type ) )
            {
                WriteComment( sb, type.comment );
                if( !string.IsNullOrEmpty( type.Value ) ) // ダミーの型定義で空行ができないように
                {
                    sb.AppendLine( type.Value );
                }
            }
        }

        private void WriteCommandRequireType( StringBuilder sb, Command command )
        {
            if( command == null )
            {
                return;
            }
            WriteType( sb, command.proto.Type );
            if( command.param != null )
            {
                foreach( var param in command.param )
                {
                    WriteType( sb, param.Type );
                }
            }
        }

        private void WriteEnum( StringBuilder sb, Enum @enum )
        {
            if( @enum == null )
            {
                return;
            }
            WriteComment( sb, @enum.comment );
            sb.AppendLine( $"#define {@enum.name} {@enum.value} // NOLINT" );
        }

        private void WriteCommandTypeDefinition( StringBuilder sb, Command command )
        {
            WriteComment( sb, command.comment );
            var posApi = command.proto.Value.IndexOf( command.proto.name );
            sb.Append( $"typedef {command.proto.Value.Insert( posApi, $"( {GetApiEntry()} *" )}Type )( " );
            WriteParams( sb, command.param );
            sb.AppendLine( " );" );
        }

        private void WriteCommandDeclaration( StringBuilder sb, Command command )
        {
            WriteComment( sb, command.comment );
            var posApi = command.proto.Value.IndexOf( command.proto.name );
            sb.Append( $"NN_GLL_GL_API {command.proto.Value.Insert( posApi, $"{GetApiEntry()} " )}( " );
            WriteParams( sb, command.param );
            sb.AppendLine( " );" );
        }

        private void WriteParams( StringBuilder sb, Param[] param )
        {
            if( param == null )
            {
                return;
            }
            for( int idxParam = 0; idxParam < param.Length; ++idxParam )
            {
                if( idxParam > 0 )
                {
                    sb.Append( ", " );
                }
                sb.Append( param[ idxParam ].Value );
            }
        }

        private void WriteCommandPreprocessorDefinition( StringBuilder sb, Command command )
        {
            sb.AppendLine( $"#define {command.proto.name} {GetCommandVariableName( command )}" );
        }

        private void WriteCommandVariableDeclaration( StringBuilder sb, Command command )
        {
            sb.AppendLine( $"NN_GLL_API {GetCommandTypeName( command )} {GetCommandVariableName( command )};" );
        }

        private void WriteCommandVariableDefinition( StringBuilder sb, Command command )
        {
            sb.AppendLine( $"{GetCommandTypeName( command )} {GetCommandVariableName( command )} = NULL;" );
        }

        private IEnumerable<IFeature> EnumerateTargetFeaturesAndExtensions()
        {
            return EnumerateTargetFeatures().Cast<IFeature>().Concat( EnumerateTargetExtensions().Cast<IFeature>() );
        }

        private IEnumerable<Feature> EnumerateTargetFeatures()
        {
            return m_Registry.Feature.Where( feature => feature.api == Api );
        }

        private IEnumerable<Extension> EnumerateTargetExtensions()
        {
            foreach( var extensions in m_Registry.Extensions.Where( extensions => extensions.extension != null ) )
            {
                foreach( var extension in extensions.extension )
                {
                    var supported = extension.supported.Split( '|' );
                    if( supported.Contains( Api ) )
                    {
                        yield return extension;
                    }
                }
            }
        }

        private IEnumerable<Type> EnumerateRequiredTypes( IFeature feature )
        {
            foreach( var require in feature.Require )
            {
                foreach( var interfaceElementType in require.Type )
                {
                    yield return interfaceElementType.Type;
                }
            }
        }

        private IEnumerable<Enum> EnumerateRequiredEnums( IFeature feature )
        {
            foreach( var require in feature.Require )
            {
                foreach( var interfaceElementEnum in require.Enum )
                {
                    yield return interfaceElementEnum.Enum;
                }
            }
        }

        private IEnumerable<Command> EnumerateRequiredCommands( IFeature feature )
        {
            foreach( var require in feature.Require )
            {
                foreach( var interfaceElementCommand in require.Command )
                {
                    yield return interfaceElementCommand.Command;
                }
            }
        }

        private bool IsVersion10( IFeature feature )
        {
            return ( feature is Feature ) && ( feature as Feature ).number == "1.0";
        }

        private void WriteComment( StringBuilder sb, string comment )
        {
            if( !string.IsNullOrEmpty( comment ) )
            {
                sb.Append( "// " ).AppendLine( comment );
            }
        }

        private string GetCommandTypeName( Command command )
        {
            return command.proto.name + "Type";
        }

        private string GetCommandVariableName( Command command )
        {
            return "g_nngll" + command.proto.name.Substring( 2 );
        }

        private string GetIsSupportedFunctionName( IFeature feature )
        {
            return "nngllIsSupported_" + feature.name;
        }

        private string GetIsSupportedVariableName( IFeature feature )
        {
            return "g_" + feature.name;
        }

        private string GetLoadFunctionName( IFeature feature )
        {
            return "Initialize_" + feature.name;
        }

        private string GetInitializeFunctionName()
        {
            return "nngllInitialize" + GetCapitalizedApi();
        }

        private string GetCapitalizedApi()
        {
            return ( new System.Globalization.CultureInfo( "en-US", false ).TextInfo.ToTitleCase( Api ) );
        }

        private string GetApiEntry()
        {
            switch( Api )
            {
                case "gl":
                    return "NN_GLL_GL_APIENTRY";
                case "wgl":
                    return "WINAPI";
                default:
                    return "";
            }
        }
    }
}
