﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.IO;
using System.Xml;

using LOG = Common.Logger;

namespace Common
{
    /// <summary>
    /// doxygenが生成するXMLファイル index.xml を解析するためのクラス
    /// </summary>
    /// <remarks>
    /// このクラスの使い方について
    ///   1, 各要素を解析するためのハンドラを作る
    ///      作るのは名前空間・関数名・構造体・マクロ定義・変数・列挙体・列挙体メンバ・ファイルの8種類
    ///      ハンドラの型は名前空間用は XmlNamespaceFunc に、それ以外は XmlFunc に定義した型を使う
    ///      なお、 CompoundModeは現在メンバが名前空間／構造体／ファイルのどれに属しているかを確認するために用いる
    ///   2, index.xml を XmlTextReader で読み込む
    ///   3, XmlReadLoop() を呼び出す。引数は2で読み込んだ XmlTextReader のインスタンスと1で定義した8つのハンドラ
    /// </remarks>
    public class ParseDoxygenXml
    {
        // 解析モード
        private enum ParseMode
        {
            PARSE_OTHER,
            PARSE_NAMESPACE,
            PARSE_FUNCTION,
            PARSE_STRUCT,
            PARSE_FILE,
            PARSE_DEFINE,
            PARSE_VARIABLE,
            PARSE_ENUM,
            PARSE_ENUMVALUE
        }

        // compoundのモード
        public enum CompoundMode
        {
            COMPOUND_OTHER,
            COMPOUND_NAMESPACE,
            COMPOUND_STRUCT,
            COMPOUND_FILE,
        }

        /// <summary>
        /// 名前空間名解析ハンドラの定義型
        /// </summary>
        /// <param name="cmode">現在のCompoundモードが何であるか</param>
        /// <param name="refid">reference id</param>
        /// <param name="namespaceName">名前空間名</param>
        /// <param name="functionName">関数名</param>
        /// <returns></returns>
        public delegate bool XmlNamespaceFunc(CompoundMode cmode, string refid, string namespaceName, string functionName);

        /// <summary>
        /// 名前空間以外の名前解析ハンドラの定義型
        /// </summary>
        /// <param name="cmode">現在のCompoundモードが何であるか</param>
        /// <param name="member_refid">名前空間／構造体／ファイル名のreference id</param>
        /// <param name="refid">reference id</param>
        /// <param name="namespaceName">名前空間名</param>
        /// <param name="functionName">関数名</param>
        /// <returns></returns>
        public delegate bool XmlFunc(CompoundMode cmode, string member_refid, string refid, string namespaceName, string functionName);

        /// <summary>
        /// doxygen が生成したファイル index.xml の解析処理を行います
        /// </summary>
        /// <param name="reader">index.xmlを読み込んだ XmlTextReader のインスタンス</param>
        /// <param name="namespaceFunc">名前空間名解析ハンドラ</param>
        /// <param name="functionFunc">関数名解析ハンドラ</param>
        /// <param name="structFunc">構造体名解析ハンドラ</param>
        /// <param name="defineFunc">マクロ名解析ハンドラ</param>
        /// <param name="variableFunc">変数名解析ハンドラ</param>
        /// <param name="enumFunc">列挙体名解析ハンドラ</param>
        /// <param name="enumvalueFunc">列挙体メンバ名解析ハンドラ</param>
        /// <param name="fileFunc">ファイル名解析ハンドラ</param>
        /// <returns></returns>
        public static bool XmlReadLoop(XmlTextReader reader, XmlNamespaceFunc namespaceFunc, XmlFunc functionFunc, XmlFunc structFunc,
                                        XmlFunc defineFunc, XmlFunc variableFunc, XmlFunc enumFunc, XmlFunc enumvalueFunc, XmlFunc fileFunc)
        {
            string refid = null;
            string member_refid = null;
            string namespaceName = null;
            ParseDoxygenXml.ParseMode mode = ParseDoxygenXml.ParseMode.PARSE_OTHER;
            ParseDoxygenXml.CompoundMode cmode = ParseDoxygenXml.CompoundMode.COMPOUND_OTHER;

            while (reader.Read())
            {
                // 要素のみを解析する
                if (reader.NodeType != XmlNodeType.Element)
                {
                    continue;
                }

                // 要素 compound を探す
                if (reader.LocalName.Equals("compound"))
                {
                    ParseDoxygenXml.ParseCompound(ref refid, ref mode, ref cmode, reader);
                }
                else if (reader.LocalName.Equals("member"))
                {
                    ParseDoxygenXml.ParseMember(ref member_refid, ref mode, ref cmode, reader);
                }
                else if (reader.LocalName.Equals("name"))
                {
                    if (mode == ParseDoxygenXml.ParseMode.PARSE_NAMESPACE)
                    {
                        ParseDoxygenXml.ParseNamespaceName(ref namespaceName, refid, cmode, reader, namespaceFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_FUNCTION)
                    {
                        ParseDoxygenXml.ParseFunctionName(namespaceName, member_refid, refid, cmode, reader, functionFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_STRUCT)
                    {
                        ParseDoxygenXml.ParseStructName(ref namespaceName, member_refid, refid, cmode, reader, structFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_DEFINE)
                    {
                        ParseDoxygenXml.ParseDefineName(namespaceName, member_refid, refid, cmode, reader, defineFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_VARIABLE)
                    {
                        ParseDoxygenXml.ParseVariableName(namespaceName, member_refid, refid, cmode, reader, variableFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_ENUM)
                    {
                        ParseDoxygenXml.ParseEnumName(namespaceName, member_refid, refid, cmode, reader, enumFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_ENUMVALUE)
                    {
                        ParseDoxygenXml.ParseEnumValueName(namespaceName, member_refid, refid, cmode, reader, enumvalueFunc);
                    }
                    else if (mode == ParseDoxygenXml.ParseMode.PARSE_FILE)
                    {
                        ParseDoxygenXml.ParseFileName(namespaceName, member_refid, refid, cmode, reader, fileFunc);
                    }
                    mode = ParseDoxygenXml.ParseMode.PARSE_OTHER;
                }
                reader.MoveToElement();
            }

            return true;
        }

        private static bool ParseCompound(ref string refid, ref ParseMode mode, ref CompoundMode cmode, XmlTextReader reader)
        {
            string id = null;
            for (int i = 0; i < reader.AttributeCount; i++)
            {
                reader.MoveToAttribute(i);
                bool save_refid = false;
                if (reader.Name == "refid")
                {
                    id = reader.Value;
                }
                else if (reader.Name == "kind")
                {
                    if (reader.Value == "class" || reader.Value == "namespace")
                    {
                        mode = ParseMode.PARSE_NAMESPACE;
                        cmode = CompoundMode.COMPOUND_NAMESPACE;
                        save_refid = true;
                    }
                    else if (reader.Value == "struct")
                    {
                        mode = ParseMode.PARSE_STRUCT;
                        cmode = CompoundMode.COMPOUND_STRUCT;
                        save_refid = true;
                    }
                    else if (reader.Value == "file")
                    {
                        mode = ParseMode.PARSE_FILE;
                        cmode = CompoundMode.COMPOUND_FILE;
                        save_refid = true;
                    }
                    else
                    {
                        cmode = CompoundMode.COMPOUND_OTHER;
                    }
                }

                if (save_refid)
                {
                    refid = id;
                    //LOG.Log("refid = {0}\n", refid);
                }
            }

            return true;
        }

        private static bool ParseMember(ref string member_refid, ref ParseMode mode, ref CompoundMode cmode, XmlTextReader reader)
        {
            // 構造体メンバは解析対象外とする
            if (cmode == CompoundMode.COMPOUND_STRUCT)
            {
                return true;
            }

            string id = null;
            for (int i = 0; i < reader.AttributeCount; i++)
            {
                reader.MoveToAttribute(i);
                bool save_refid = false;
                if (reader.Name == "kind")
                {
                    if (reader.Value == "function")
                    {
                        mode = ParseMode.PARSE_FUNCTION;
                        save_refid = true;
                    }
                    else if (reader.Value == "define")
                    {
                        mode = ParseMode.PARSE_DEFINE;
                        save_refid = true;
                    }
                    else if (reader.Value == "variable")
                    {
                        mode = ParseMode.PARSE_VARIABLE;
                        save_refid = true;
                    }
                    else if (reader.Value == "enum")
                    {
                        mode = ParseMode.PARSE_ENUM;
                        save_refid = true;
                    }
                    else if (reader.Value == "enumvalue")
                    {
                        mode = ParseMode.PARSE_ENUMVALUE;
                        save_refid = true;
                    }
                }
                else if (reader.Name == "refid")
                {
                    id = reader.Value;
                }
                if (save_refid)
                {
                    member_refid = id;
                }
            }

            return true;
        }

        private static bool ParseNamespaceName(ref string namespaceName, string refid, CompoundMode cmode, XmlTextReader reader, XmlNamespaceFunc func)
        {
            // kind="namespace"の要素のnameを保存する
            namespaceName = reader.ReadString();
            return func(cmode, refid, namespaceName, null);
        }

        private static bool ParseFunctionName(string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="function"の要素のnameを保存する
            string functionName = reader.ReadString();
            return func(cmode, member_refid, refid, namespaceName, functionName);
        }

        private static bool ParseStructName(ref string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="struct"の要素のnameを保存する
            string structName = reader.ReadString();
            namespaceName = structName;     // 構造体メンバ変数で用いる
            return func(cmode, member_refid, refid, namespaceName, structName);
        }

        private static bool ParseDefineName(string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="define"の要素のnameを保存する
            string definitionName = reader.ReadString();
            return func(cmode, member_refid, refid, namespaceName, definitionName);
        }

        private static bool ParseVariableName(string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="variable"の要素のnameを保存する
            string variableName = reader.ReadString();
            return func(cmode, member_refid, refid, namespaceName, variableName);
        }

        private static bool ParseEnumName(string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="enum"の要素のnameを保存する
            string enumName = reader.ReadString();
            return func(cmode, member_refid, refid, namespaceName, enumName);
        }

        private static bool ParseEnumValueName(string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="enumvalue"の要素のnameを保存する
            string enumvalueName = reader.ReadString();
            return func(cmode, member_refid, refid, namespaceName, enumvalueName);
        }

        private static bool ParseFileName(string namespaceName, string member_refid, string refid, CompoundMode cmode, XmlTextReader reader, XmlFunc func)
        {
            // kind="file"の要素のnameを保存する
            string fileName = reader.ReadString();
            return func(cmode, member_refid, refid, namespaceName, fileName);
        }

        private enum ClassXmlParseMode
        {
            PARSE_MEMBERDEF,
            PARSE_COMPOUNDDEF,
            PARSE_OTHER
        }
        private enum ClassXmlInMemberParseMode
        {
            INMEMBER_PARSE_BRIEF_DESCRIPTION,
            INMEMBER_PARSE_DETAILED_DESCRIPTION,
            INMEMBER_PARSE_INBODY_DESCRIPTION,
            INMEMBER_PARSE_OTHER
        }
        private enum ClassXmlSectionParseMode
        {
            SECTION_PARSE_IN,
            SECTION_PARSE_OUT,
            SECTION_PARSE_BRIEF,
            SECTION_PARSE_OTHER
        }

        /// <summary>
        /// doxygen XMLファイル要素memberfunc解析ハンドラの定義型
        /// </summary>
        /// <param name="name">memberfunc内の属性名</param>
        /// <param name="value">属性に対応する値</param>
        /// <returns>解析を中止したい場合はfalseを返すようにする</returns>
        public delegate bool ClassXmlMemberdefAttributeFunc(string name, string value);

        /// <summary>
        /// doxygen XMLファイル要素のうちmemberfunc以外の要素の解析ハンドラ定義型
        /// </summary>
        /// <param name="value">要素に対応する値</param>
        /// <returns>解析を中止したい場合はfalseを返すようにする</returns>
        public delegate bool ClassXmlFunc(string value);

        public struct NamespaceEnableInfo
        {
            public bool IsEnabled;
            public string Name;
        }
        private static List<NamespaceEnableInfo> NamespaceEnableInfoList = new List<NamespaceEnableInfo>();

        private static bool CheckNamespaceEnabled(ref bool isEnable, string namespaceName)
        {
            foreach (NamespaceEnableInfo enableInfo in NamespaceEnableInfoList)
            {
                if (namespaceName == enableInfo.Name)
                {
                    isEnable = enableInfo.IsEnabled;
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// doxygen が生成した各種XMLファイルがドキュメント対象であるかを確認します
        /// </summary>
        /// <param name="reader">XMLファイルのパス</param>
        /// <param name="refid">XMLファイル内で検索するrefid(デバッグ目的のみで使う)</param>
        /// <returns>ドキュメント対象である場合はtrueを返す</returns>
        public static bool CheckClassXml(XmlTextReader reader, string namespaceName, string refid)
        {
            bool isEnable = false;
            if (CheckNamespaceEnabled(ref isEnable, namespaceName) == true)
            {
                return isEnable;
            }

            ClassXmlSectionParseMode sectMode = ClassXmlSectionParseMode.SECTION_PARSE_OTHER;
            string classBriefDescription = null;
            bool bRetVal = true;

            {
                while (reader.Read())
                {
                    switch (reader.NodeType)
                    {
                        case XmlNodeType.Element:
                            switch (reader.LocalName)
                            {
                                case "sectiondef":
                                    if (sectMode == ClassXmlSectionParseMode.SECTION_PARSE_OTHER || sectMode == ClassXmlSectionParseMode.SECTION_PARSE_OUT)
                                    {
                                        sectMode = ClassXmlSectionParseMode.SECTION_PARSE_IN;
                                    }
                                    break;
                                case "briefdescription":
                                    if (sectMode == ClassXmlSectionParseMode.SECTION_PARSE_OUT)
                                    {
                                        // classおよびnamespace自体がドキュメント対象外の場合は無効とする
                                        if (reader.IsEmptyElement == true)
                                        {
                                            bRetVal = false;
                                            goto quit;
                                        }
                                        classBriefDescription = reader.ReadString();
                                        sectMode = ClassXmlSectionParseMode.SECTION_PARSE_BRIEF;
                                    }
                                    break;
                                default:
                                    if (sectMode == ClassXmlSectionParseMode.SECTION_PARSE_BRIEF)
                                    {
                                        classBriefDescription += reader.ReadString();
                                    }
                                    break;
                            }
                            break;
                        case XmlNodeType.EndElement:
                            switch (reader.LocalName)
                            {
                                case "briefdescription":
                                    if (sectMode == ClassXmlSectionParseMode.SECTION_PARSE_BRIEF)
                                    {
                                        sectMode = ClassXmlSectionParseMode.SECTION_PARSE_OTHER;
                                        classBriefDescription += reader.ReadString();
                                        // classおよびnamespace自体がドキュメント対象外の場合は無効とする
                                        StringBuilder sb = new StringBuilder(classBriefDescription);
                                        sb.Replace("\n", string.Empty);
                                        sb.Replace(" ", string.Empty);
                                        if (sb.Length == 0)
                                        {
                                            bRetVal = false;
                                            goto quit;
                                        }
                                    }
                                    break;
                                case "sectiondef":
                                    if (sectMode == ClassXmlSectionParseMode.SECTION_PARSE_IN)
                                    {
                                        sectMode = ClassXmlSectionParseMode.SECTION_PARSE_OUT;
                                    }
                                    break;
                                default:
                                    break;
                            }
                            break;
                        case XmlNodeType.Text:
                            if (sectMode == ClassXmlSectionParseMode.SECTION_PARSE_BRIEF)
                            {
                                classBriefDescription += reader.ReadString();
                            }
                            break;
                    }
                }
            }

            // XmlTextReaderのバグで</briefdescription>が解析できない時があるためここでも判定する
            if (sectMode != ClassXmlSectionParseMode.SECTION_PARSE_OTHER)
            {
                System.Diagnostics.Debug.WriteLine(string.Format("briefdescription : {0}", namespaceName));
                bRetVal = false;
                goto quit;
            }

    quit:
            NamespaceEnableInfo enableInfo = new NamespaceEnableInfo();
            enableInfo.Name = namespaceName;
            enableInfo.IsEnabled = bRetVal;
            NamespaceEnableInfoList.Add(enableInfo);

            return bRetVal;
        }

        /// <summary>
        /// doxygen が生成した各種XMLファイルの解析処理を行います
        /// </summary>
        /// <param name="reader">doxygen XMLファイルを読み込んだ XmlTextReader のインスタンス</param>
        /// <param name="refid">XMLファイル内で検索するrefid</param>
        /// <param name="memberFunc">要素memberfunc解析ハンドラ</param>
        /// <param name="definitionFunc">要素definition解析ハンドラ</param>
        /// <param name="argsstringFunc">要素argsstring解析ハンドラ</param>
        /// <returns></returns>
        public static bool ClassXmlReadLoop(XmlTextReader reader, string refid, ClassXmlMemberdefAttributeFunc memberFunc,
                                                ClassXmlFunc definitionFunc, ClassXmlFunc argsstringFunc,
                                                ClassXmlFunc briefdescriptionFunc, ClassXmlFunc detaileddescriptionFunc,
                                                ClassXmlFunc inbodydescriptionFunc)
        {
            ClassXmlParseMode mode = ClassXmlParseMode.PARSE_OTHER;
            ClassXmlInMemberParseMode inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_OTHER;
            string briefdescription = null;
            string detaileddescription = null;
            string inbodydescription = null;

            {
                while (reader.Read())
                {
                    switch (reader.NodeType)
                    {
                        case XmlNodeType.Element:
                            switch (reader.LocalName)
                            {
                                case "memberdef":
                                    // refidチェック
                                    bool refidCheck = false;
                                    for (int i = 0; i < reader.AttributeCount; i++)
                                    {
                                        reader.MoveToAttribute(i);
                                        if (reader.Name == "id" && refid == reader.Value)
                                        {
                                            refidCheck = true;
                                            break;
                                        }
                                    }
                                    // ハンドラ呼び出し
                                    if (refidCheck)
                                    {
                                        for (int i = 0; i < reader.AttributeCount; i++)
                                        {
                                            reader.MoveToAttribute(i);
                                            if (reader.Name != "id")
                                            {
                                                if (memberFunc(reader.Name, reader.Value))
                                                {
                                                    mode = ClassXmlParseMode.PARSE_MEMBERDEF;
                                                }
                                                else
                                                {
                                                    mode = ClassXmlParseMode.PARSE_OTHER;
                                                    return false;
                                                }
                                            }
                                        }
                                    }
                                    break;
                                case "definition":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        if (definitionFunc(reader.ReadString()) == false)
                                        {
                                            return false;
                                        }
                                    }
                                    break;
                                case "argsstring":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        if (argsstringFunc(reader.ReadString()) == false)
                                        {
                                            return false;
                                        }
                                    }
                                    break;
                                case "briefdescription":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        briefdescription = reader.ReadString();
                                        inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_BRIEF_DESCRIPTION;
                                    }
                                    break;
                                case "detaileddescription":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        detaileddescription = reader.ReadString();
                                        inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_DETAILED_DESCRIPTION;
                                    }
                                    break;
                                case "inbodydescription":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        inbodydescription = reader.ReadString();
                                        inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_INBODY_DESCRIPTION;
                                    }
                                    break;
                                default:
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        switch (inMode)
                                        {
                                            case ClassXmlInMemberParseMode.INMEMBER_PARSE_BRIEF_DESCRIPTION:
                                                briefdescription += reader.ReadString();
                                                break;
                                            case ClassXmlInMemberParseMode.INMEMBER_PARSE_DETAILED_DESCRIPTION:
                                                detaileddescription += reader.ReadString();
                                                break;
                                            case ClassXmlInMemberParseMode.INMEMBER_PARSE_INBODY_DESCRIPTION:
                                                inbodydescription += reader.ReadString();
                                                break;
                                        }
                                    }
                                    break;
                            }
                            break;
                        case XmlNodeType.EndElement:
                            switch (reader.LocalName)
                            {
                                case "memberdef":
                                    if (mode != ClassXmlParseMode.PARSE_OTHER)
                                    {
                                        mode = ClassXmlParseMode.PARSE_OTHER;
                                    }
                                    break;
                                case "briefdescription":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_OTHER;
                                        briefdescription += reader.ReadString();
                                        if (briefdescriptionFunc(briefdescription) == false)
                                        {
                                            return false;
                                        }
                                    }
                                    break;
                                case "detaileddescription":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_OTHER;
                                        detaileddescription += reader.ReadString();
                                        if (detaileddescriptionFunc(detaileddescription) == false)
                                        {
                                            return false;
                                        }
                                    }
                                    break;
                                case "inbodydescription":
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        inMode = ClassXmlInMemberParseMode.INMEMBER_PARSE_OTHER;
                                        inbodydescription += reader.ReadString();
                                        if (inbodydescriptionFunc(inbodydescription) == false)
                                        {
                                            return false;
                                        }
                                    }
                                    break;
                                default:
                                    if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                                    {
                                        switch (inMode)
                                        {
                                            case ClassXmlInMemberParseMode.INMEMBER_PARSE_BRIEF_DESCRIPTION:
                                                briefdescription += reader.ReadString();
                                                break;
                                            case ClassXmlInMemberParseMode.INMEMBER_PARSE_DETAILED_DESCRIPTION:
                                                detaileddescription += reader.ReadString();
                                                break;
                                            case ClassXmlInMemberParseMode.INMEMBER_PARSE_INBODY_DESCRIPTION:
                                                inbodydescription += reader.ReadString();
                                                break;
                                        }
                                    }
                                    break;
                            }
                            break;
                        case XmlNodeType.Text:
                            if (mode == ClassXmlParseMode.PARSE_MEMBERDEF)
                            {
                                switch (inMode)
                                {
                                    case ClassXmlInMemberParseMode.INMEMBER_PARSE_BRIEF_DESCRIPTION:
                                        briefdescription += reader.ReadString();
                                        break;
                                    case ClassXmlInMemberParseMode.INMEMBER_PARSE_DETAILED_DESCRIPTION:
                                        detaileddescription += reader.ReadString();
                                        break;
                                    case ClassXmlInMemberParseMode.INMEMBER_PARSE_INBODY_DESCRIPTION:
                                        inbodydescription += reader.ReadString();
                                        break;
                                }
                            }
                            break;
                    }

                    //reader.MoveToElement();
                }
            }

            return true;
        }

        /// <summary>
        /// 関数のrefIdから所属名前空間のrefIdを解析します
        /// </summary>
        /// <param name="refid">関数のrefid</param>
        /// <returns>所属する名前空間(もしくはクラス)のrefid</returns>
        public static string GetNamespaceRefIdFromRefId(string refid)
        {
            int length = refid.Length;
            System.Diagnostics.Debug.Assert(refid.Substring(length - 34 - 1, 1) == "_", " ref id of function mult end with _(33 characters)");
            string[] refword = refid.Split('_');
            int refwordSize = refword.Length;
            if (refword[refwordSize - 2].Length == 33)
            {
                return refid.Substring(0, length - 69);
            }
            return refid.Substring(0, length - 35);
        }
    }
}
