﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

namespace Nintendo.DotNetLocalizer
{
    /// <summary>
    /// ユーティリティクラス。
    /// </summary>
    public sealed class Utility
    {
        private const string ResourcesExt = ".resources";

        public static void EnsureResourcesExt(string name)
        {
            if (!name.EndsWith(ResourcesExt, StringComparison.OrdinalIgnoreCase))
            {
                throw new InvalidOperationException(string.Format("{0} でないファイルが見つかりました。", ResourcesExt));
            }
        }

        public static string TrimResourcesExt(string name)
        {
            if (!name.EndsWith(ResourcesExt, StringComparison.OrdinalIgnoreCase))
            {
                throw new ArgumentException(ResourcesExt + "ではありません。");
            }
            return name.Substring(0, name.Length - ResourcesExt.Length);
        }

        /// <summary>
        /// エラーテスト。
        /// </summary>
        public static void TestError(bool condition, string message)
        {
            if (!condition)
            {
                throw new ApplicationException(message);
            }
        }

        /// <summary>
        /// エラーテスト。
        /// </summary>
        public static void TestError(bool condition, string format, params object[] args)
        {
            TestError(condition, string.Format(format, args));
        }

        /// <summary>
        /// ベースディレクトリと相対パスから絶対パスを取得。
        /// </summary>
        public static string GetFullPath(string basePath, string relativePath)
        {
            // ベースパスの末尾はディレクトリであることを
            // 判断するために '\' が必要
            if (!basePath.EndsWith("\\"))
            {
                basePath += "\\";
            }

            // 結合する
            Uri baseUri   = new Uri(basePath);
            Uri resultUri = new Uri(baseUri, relativePath);
            return resultUri.LocalPath;
        }

        /// <summary>
        /// CR+LF形式の改行をLF形式の改行に変換。
        /// </summary>
        public static string CRLFToLF(string input)
        {
            return input.Replace("\r\n", "\n");
        }

        /// <summary>
        /// LF形式の改行をCR+LF形式の改行に変換。
        /// </summary>
        public static string LFToCRLF(string input)
        {
            string str = input.Replace("\r", string.Empty);

            return str.Replace("\n", "\r\n");
        }

        public static string BackSlashToSlash(string path)
        {
            string newPath = path.Replace("\\\\", "/");
            newPath = newPath.Replace("\\", "/");

            return newPath;
        }
    }
}
