﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;

namespace Nintendo.DotNetLocalizer
{
    /// <summary>
    /// 出力クラス。
    /// </summary>
    public sealed class Output
    {
        // 出力抑制フラグ
        private static bool _silent = true;
        // トレース用ライター
        private static TextWriter _traceWriter = null;

        /// <summary>
        /// タイプコンストラクタ。
        /// </summary>
        static Output()
        {
            //_traceWriter = new StreamWriter(App.ExecutableDir + @"\log.txt");

            Application.ApplicationExit += new EventHandler(Event_Application_ApplicationExit);
        }

        /// <summary>
        /// 出力制御フラグ。
        /// </summary>
        public static bool Silent
        {
            get { return _silent;  }
            set { _silent = value; }
        }

        //---------------------------------------------------------------------
        // 書き込み
        //---------------------------------------------------------------------
        /// <summary>
        /// 書き込み。
        /// </summary>
        public static void Write(string message)
        {
            // 標準出力
            Console.Write(message);

            // トレースログ出力
            if (_traceWriter != null)
            {
                _traceWriter.Write(message);
                _traceWriter.Flush();
            }
        }

        /// <summary>
        /// 書き込み。標準エラー出力
        /// </summary>
        public static void ErrorWrite(string message)
        {
            // 標準出力
            Console.Error.Write(message);

            // トレースログ出力
            if (_traceWriter != null)
            {
                _traceWriter.Write(message);
                _traceWriter.Flush();
            }
        }

        /// <summary>
        /// 書式指定付き書き込み。
        /// </summary>
        public static void Write(string format, params object[] args)
        {
            Write(string.Format(format, args));
        }

        /// <summary>
        /// 書き込み＋改行。
        /// </summary>
        public static void WriteLine(string message)
        {
            Write(message + "\r\n");
        }

        /// <summary>
        /// 書式指定付き書き込み＋改行。
        /// </summary>
        public static void WriteLine(string format, params object[] args)
        {
            Write(string.Format(format, args) + "\r\n");
        }

        /// <summary>
        /// 書き込み＋改行。標準エラー出力
        /// </summary>
        public static void ErrorWriteLine(string message)
        {
            ErrorWrite(message + "\r\n");
        }

        //---------------------------------------------------------------------
        // 条件付き
        //---------------------------------------------------------------------
        /// <summary>
        /// 条件付き書き込み。
        /// </summary>
        public static void ConditionalWrite(string message)
        {
            if (!_silent)
            {
                Write(message);
            }
        }

        /// <summary>
        /// 条件付き書式指定付き書き込み。
        /// </summary>
        public static void ConditionalWrite(string format, params object[] args)
        {
            ConditionalWrite(string.Format(format, args));
        }

        /// <summary>
        /// 条件付き書き込み＋改行。
        /// </summary>
        public static void ConditionalWriteLine(string message)
        {
            ConditionalWrite(message + "\r\n");
        }

        /// <summary>
        /// 条件付き書式指定付き書き込み＋改行。
        /// </summary>
        public static void ConditionalWriteLine(string format, params object[] args)
        {
            ConditionalWrite(string.Format(format, args) + "\r\n");
        }

        //---------------------------------------------------------------------
        // デバッグ
        //---------------------------------------------------------------------
        /// <summary>
        /// デバッグ書き込み。
        /// </summary>
        [Conditional("DEBUG")]
        public static void DebugWrite(string message)
        {
            Write("DEBUG:" + message);
        }

        /// <summary>
        /// デバッグ書式指定付き書き込み。
        /// </summary>
        [Conditional("DEBUG")]
        public static void DebugWrite(string format, params object[] args)
        {
            DebugWrite(string.Format(format, args));
        }

        /// <summary>
        /// デバッグ書き込み＋改行。
        /// </summary>
        [Conditional("DEBUG")]
        public static void DebugWriteLine(string message)
        {
            DebugWrite(message + "\r\n");
        }

        /// <summary>
        /// デバッグ書式指定付き書き込み＋改行。
        /// </summary>
        [Conditional("DEBUG")]
        public static void DebugWriteLine(string format, params object[] args)
        {
            DebugWrite(string.Format(format, args) + "\r\n");
        }

        //---------------------------------------------------------------------
        // エラー
        //---------------------------------------------------------------------
        /// <summary>
        /// エラー書き込み。
        /// </summary>
        public static void Error(string message)
        {
            ErrorWriteLine("エラー:" + message);
        }

        /// <summary>
        /// 書式指定付きエラー書き込み。
        /// </summary>
        public static void Error(string format, params object[] args)
        {
            Error(string.Format(format, args));
        }

        //---------------------------------------------------------------------
        // 警告
        //---------------------------------------------------------------------
        /// <summary>
        /// 警告書き込み。
        /// </summary>
        public static void Warning(string message)
        {
            ErrorWriteLine("警告:" + message);
        }

        /// <summary>
        /// 書式指定付き警告書き込み。
        /// </summary>
        public static void Warning(string format, params object[] args)
        {
            Warning(string.Format(format, args));
        }

        #region イベントハンドラ
        /// <summary>
        /// アプリケーション終了時。
        /// </summary>
        private static void Event_Application_ApplicationExit(object sender, EventArgs e)
        {
            if (_traceWriter != null)
            {
                _traceWriter.Close();
            }
        }
        #endregion
    }
}
