﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using DocumentFormat.OpenXml;
using DocumentFormat.OpenXml.Packaging;
using DocumentFormat.OpenXml.Spreadsheet;

namespace Nintendo.DotNetLocalizer
{
    /// <summary>
    /// エクセル表生成クラス。
    /// </summary>
    public sealed class LocalizationMapWriter : IDisposable
    {
        /// <summary>
        /// 翻訳対象シート名
        /// </summary>
        public const string LocalizeSheetName = "翻訳対象項目";

        /// <summary>
        /// 未使用項目シート名
        /// </summary>
        public const string UnusedSheetName = "現在は使用されていない項目";

        /// <summary>
        /// ヘッダ行数
        /// </summary>
        public const int HeaderLineNum = 2;

        /// <summary>
        /// エクセルヘッダ（１シート目）
        /// </summary>
        private static readonly Dictionary<string, int> HeaderDic =
            new Dictionary<string, int>
            {
                { "ID", 0 },
                { "ProjectFile", 1 },
                { "ResourceFile", 2 },
                { "ResourceName", 3 },
                { "Image.ja-JP", 4 },
                { "Image.en-US", 5 },
                { "Text.ja-JP", 6 },
                { "Text.en-US", 7 }
            };

        /// <summary>
        /// エクセル１シート目のカラムヘッダ
        /// </summary>
        public static Dictionary<string, int> HeaderColumns
        {
            get
            {
                return HeaderDic;
            }
        }

        /// <summary>
        /// 出力ファイルパス
        /// </summary>
        private readonly string _outputPath;

        private readonly bool _snapshot;

        /// <summary>
        /// 翻訳数
        /// </summary>
        private int _localizedCount = 0;

        /// <summary>
        /// スプレッドシート
        /// </summary>
        private SpreadsheetDocument _doc = null;

        /// <summary>
        /// ワークブック
        /// </summary>
        private Workbook _book = null;

        /// <summary>
        /// 翻訳対象シート
        /// </summary>
        private Worksheet _wkSheet = null;

        /// <summary>
        /// 共有文字列テーブル
        /// </summary>
        private SharedStringTable _sharedStringTable = null;

        /// <summary>
        /// 共有文字列用辞書
        /// </summary>
        private readonly Dictionary<string, int> _sharedStringDic = new Dictionary<string, int>();

        /// <summary>
        /// ハイパーリンク用辞書
        /// </summary>
        private readonly Dictionary<string, string> _hyperlinkDic = new Dictionary<string, string>();

        /// <summary>
        /// スタイルシート
        /// </summary>
        private Stylesheet _stylesheet = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LocalizationMapWriter(string outputPath, bool snapshot)
        {
            _outputPath = outputPath;
            _snapshot = snapshot;

            _doc = SpreadsheetDocument.Create(_outputPath, SpreadsheetDocumentType.Workbook, true);

            //----------------------------------------------------------------
            // 翻訳対象項目用のブックを作成
            //----------------------------------------------------------------
            // シートを作成
            var workbookPart = _doc.AddWorkbookPart();
            workbookPart.Workbook = new Workbook();
            _book = workbookPart.Workbook;
            var shareStringPart = workbookPart.AddNewPart<SharedStringTablePart>();
            _sharedStringTable = new SharedStringTable();
            shareStringPart.SharedStringTable = _sharedStringTable;

            _wkSheet = new Worksheet(new SheetData(CreateHeaderRows()));

            var sheetPart = workbookPart.AddNewPart<WorksheetPart>();
            sheetPart.Worksheet = _wkSheet;

            // ワークブックとシートの関連付け
            workbookPart.Workbook.AppendChild<Sheets>(
                new Sheets(
                    new Sheet
                    {
                        Id = workbookPart.GetIdOfPart(_wkSheet.WorksheetPart),
                        SheetId = 1,
                        Name = LocalizeSheetName
                    }));

            // スタイルシートを作成し、ワークブックに追加します。
            _stylesheet = CreateStyleSheet();
            var stylesPart = workbookPart.AddNewPart<WorkbookStylesPart>();
            stylesPart.Stylesheet = _stylesheet;
        }

        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public void Dispose()
        {
            if (null != _doc)
            {
                _doc.Dispose();
                _doc = null;
            }
        }

        /// <summary>
        /// 実行。
        /// </summary>
        public void Execute(IEnumerable<LocalizationItem> strResJas, IEnumerable<LocalizationItem> strResEns, IDictionary<string, int> headerColumns)
        {
            if (null == _doc)
            {
                throw new ObjectDisposedException("SpreadsheetDocument");
            }

            // シートに文字列出力
            var stringCount = OutPutStringsToExcel(_wkSheet, strResJas, strResEns, headerColumns);

            // シートにヘッダ部分を書き込みます。
            var unlocalizedCount = stringCount - _localizedCount;
            WriteSheetHeader_(_wkSheet, stringCount, _localizedCount, unlocalizedCount, headerColumns);

            Output.ConditionalWriteLine(" 文字列：{0}", stringCount);
            Output.ConditionalWriteLine(" 翻訳済：{0}", _localizedCount);
            Output.ConditionalWriteLine(" 未翻訳：{0}", unlocalizedCount);
        }

        /// <summary>
        /// ファイル出力
        /// </summary>
        public void Write()
        {
            if (null == _doc)
            {
                throw new ObjectDisposedException("SpreadsheetDocument");
            }

            //----------------------------------------------------------------
            // ファイル出力
            //----------------------------------------------------------------

            Output.ConditionalWriteLine(
                                        "翻訳リスト '{0}' を作成しています...",
                                        Path.GetFileName(_outputPath));

            _book.Save();
        }

        /// <summary>
        /// カウント整数値を文字列に変換します。
        /// -1が指定されている場合は、---を返します。
        /// </summary>
        private string ConvertCountToString_(int count)
        {
            return count != -1 ? count.ToString() : "---";
        }

        /// <summary>
        /// エクセルシートのヘッダ部分を書き込みます。
        /// </summary>
        private void WriteSheetHeader_(Worksheet sheet, int stringCount, int localizedCount, int unlocalizedCount, IDictionary<string, int> headerColumns)
        {
            var sheetData = sheet.GetFirstChild<SheetData>();
            var rows = sheetData.Elements<Row>().Take(HeaderLineNum).ToArray();

            var enHeaders = new Cell[headerColumns.Count];
            enHeaders[headerColumns["ID"]] = CreateHeaderCell("ID");
            enHeaders[headerColumns["ProjectFile"]] = CreateHeaderCell("ProjectFile");
            enHeaders[headerColumns["ResourceFile"]] = CreateHeaderCell("ResourceFile");
            enHeaders[headerColumns["ResourceName"]] = CreateHeaderCell("ResourceName");
            enHeaders[headerColumns["Image.ja-JP"]] = CreateHeaderCell("Image.ja-JP");
            enHeaders[headerColumns["Image.en-US"]] = CreateHeaderCell("Image.en-US");
            enHeaders[headerColumns["Text.ja-JP"]] = CreateHeaderCell("Text.ja-JP");
            enHeaders[headerColumns["Text.en-US"]] = CreateHeaderCell("Text.en-US");
            sheetData.ReplaceChild(new Row(enHeaders), rows[0]);

            var jaHeaders = new Cell[headerColumns.Count];
            jaHeaders[headerColumns["ID"]] = CreateHeaderCell("ID");
            jaHeaders[headerColumns["ProjectFile"]] = CreateHeaderCell("プロジェクト名");
            jaHeaders[headerColumns["ResourceFile"]] = CreateHeaderCell("リソースファイル名");
            jaHeaders[headerColumns["ResourceName"]] = CreateHeaderCell("名前");
            jaHeaders[headerColumns["Image.ja-JP"]] = CreateHeaderCell("日本語画像");
            jaHeaders[headerColumns["Image.en-US"]] = CreateHeaderCell("英語画像");

            jaHeaders[headerColumns["Text.ja-JP"]] =
                CreateHeaderCell(
                    string.Format("日本語({0})", ConvertCountToString_(stringCount)));

            jaHeaders[headerColumns["Text.en-US"]] =
                CreateHeaderCell(
                    string.Format(
                        "英語(翻訳済:{0} 未翻訳:{1})",
                        ConvertCountToString_(localizedCount),
                        ConvertCountToString_(unlocalizedCount)));
            sheetData.ReplaceChild(new Row(jaHeaders), rows[1]);

            // 列幅最適化
            if (null == sheet.GetFirstChild<Columns>())
            {
                var columns = new Columns(
                    CreateColumn(headerColumns["ID"], 5),
                    CreateColumn(headerColumns["ProjectFile"], 20),
                    CreateColumn(headerColumns["ResourceFile"], 20),
                    CreateColumn(headerColumns["ResourceName"], 20),
                    CreateColumn(headerColumns["Image.ja-JP"], _snapshot ? 40 : 0),
                    CreateColumn(headerColumns["Image.en-US"], _snapshot ? 40 : 0),
                    CreateColumn(headerColumns["Text.ja-JP"], 50),
                    CreateColumn(headerColumns["Text.en-US"], 50));
                sheet.InsertBefore(columns, sheetData);
            }
        }

        /// <summary>
        /// セルに文字列を共有文字列として設定します。
        /// </summary>
        /// <param name="cell">設定するセル。</param>
        /// <param name="value">セルに設定する文字列。</param>
        private void SetSharedString(Cell cell, string value)
        {
            var index = InsertSharedStringItem(value);
            cell.CellValue = new CellValue(index.ToString());
            cell.DataType = CellValues.SharedString;
        }

        /// <summary>
        /// カラムを作成します。
        /// </summary>
        /// <param name="colIdx">カラムインデックス。</param>
        /// <param name="width">カラム幅。</param>
        private Column CreateColumn(int colIdx, int width)
        {
            var colNum = (uint)colIdx + 1;
            var column = new Column
            {
                Min = colNum,
                Max = colNum
            };
            if (width == 0)
            {
                column.Hidden = true;
            }
            else
            {
                column.CustomWidth = true;
                column.Width = width;
            }
            return column;
        }

        /// <summary>
        /// エクセル表に出力
        /// </summary>
        private int OutPutStringsToExcel(Worksheet sheet, IEnumerable<LocalizationItem> locItemsJa, IEnumerable<LocalizationItem> locItemsEn, IDictionary<string, int> headerColumns)
        {
            int stringCount = 0;

#if false
            Excel.Range mergeCell = null;
            try
            {
                mergeCell = Merge(
                                  sheet,
                                  line,
                                  line + strings.Count - 1,
                                  headerColumns["ResourceFile"],
                                  fileName
                                  );
                mergeCell.Interior.Color = System.Drawing.Color.FromArgb(192,192,192).ToArgb();
                // Merge(sheet, line, line + strings.Count - 1, headerColumns["Image.ja-JP"], " ");
                // Merge(sheet, line, line + strings.Count - 1, headerColumns["Image.en-US"], " ");
            }
            finally
            {
                if (mergeCell != null)
                {
                    System.Runtime.InteropServices.Marshal.ReleaseComObject(mergeCell);
                }
            }
#endif

            var sheetData = sheet.GetFirstChild<SheetData>();

            var enumResPairs = LocalizationItem.JoinLocalizeItemPair(locItemsJa, locItemsEn);
            foreach (var pair in enumResPairs)
            {
                // セル取得
                var lineNo = sheetData.ChildElements.Count + 1; // 行番号(1スタート)
                var cells = new Cell[headerColumns.Count];
                for (var i = 0; i < cells.Length; ++i)
                {
                    cells[i] = new Cell();
                }
                sheetData.AppendChild(new Row(cells));
                var cellID = cells[headerColumns["ID"]];
                var cellPro = cells[headerColumns["ProjectFile"]];
                var cellRes = cells[headerColumns["ResourceFile"]];
                var cellName = cells[headerColumns["ResourceName"]];
                var cellImgJP = cells[headerColumns["Image.ja-JP"]];
                var cellImgUS = cells[headerColumns["Image.en-US"]];
                var cellJpn = cells[headerColumns["Text.ja-JP"]];
                var cellEng = cells[headerColumns["Text.en-US"]];

                cellID.StyleIndex = (uint)CellStyleIndexes.ResourceId;
                cellPro.StyleIndex = (uint)CellStyleIndexes.ResourceId;
                cellRes.StyleIndex = (uint)CellStyleIndexes.ResourceId;
                cellName.StyleIndex = (uint)CellStyleIndexes.ResourceId;
                cellImgUS.StyleIndex = (uint)CellStyleIndexes.ImageLink;
                cellImgJP.StyleIndex = (uint)CellStyleIndexes.ImageLink;
                cellJpn.StyleIndex = (uint)CellStyleIndexes.NormalText;
                cellEng.StyleIndex = (uint)CellStyleIndexes.NormalText;

                // 書き込み
                cellID.CellValue = new CellValue((lineNo - HeaderLineNum).ToString()); // ヘッダ部を引く
                SetSharedString(cellPro, pair.Ja.AssemblyName);
                SetSharedString(cellRes, pair.Ja.ResourceFile);
                SetSharedString(cellName, pair.Ja.ResourceName);
                SetSharedString(cellJpn, pair.Ja.Text);

                if (_snapshot && pair.Ja.ResourceFile.EndsWith(".baml", StringComparison.OrdinalIgnoreCase))
                {
                    var bamlFiles = pair.Ja.ResourceFile.Split(new char[] { ':' }, 2);
                    var bamlPath = Path.Combine(pair.Ja.AssemblyName, bamlFiles[bamlFiles.Length - 1]);
                    var xamlPath = Path.ChangeExtension(bamlPath, ".xaml.jpg");
                    var cellStr = string.Format("{0}({1})", Path.GetFileNameWithoutExtension(bamlPath), Path.GetDirectoryName(bamlPath).Replace('\\', '/'));

                    var cellInfos = new[] {
                        new { Cell = cellImgJP, Column = "E", Lang = "ja" },
                        new { Cell = cellImgUS, Column = "F", Lang = "en" }
                    };
                    foreach (var cellInfo in cellInfos)
                    {
                        var xamlLangPath = Path.Combine("images", cellInfo.Lang, xamlPath);
                        var xamlLangFullPath = Path.Combine(Path.GetDirectoryName(_outputPath), xamlLangPath);
                        if (File.Exists(xamlLangFullPath))
                        {
                            SetHyperlink(cellInfo.Cell, xamlLangPath, cellInfo.Column, lineNo, cellStr);
                        }
                    }
                }

                // 翻訳テーブルから翻訳結果が検出できれば...
                string english = pair.En.Text;

                if (!string.IsNullOrEmpty(english))
                {
                    // new format
                    SetSharedString(cellEng, english);
                    _localizedCount++;
                }

                stringCount++;

                // 未入力箇所の色変更
                if (string.IsNullOrEmpty(english) || pair.En.IsNew)
                {
                    cellEng.StyleIndex = (uint)CellStyleIndexes.NewText;
                }
            }

            return stringCount;
        }

        /// <summary>
        /// ヘッダ用Rowオブジェクトを作成します。
        /// </summary>
        /// <returns>作成したRowオブジェクト配列。</returns>
        private Row[] CreateHeaderRows()
        {
            var rows = new Row[HeaderLineNum];
            for (var i = 0; i < rows.Length; ++i)
            {
                rows[i] = new Row();
            }
            return rows;
        }

        /// <summary>
        /// ヘッダ用セルを作成します。
        /// </summary>
        /// <param name="value">セルに設定する文字列。</param>
        /// <returns>作成したヘッダ用セル。</returns>
        private Cell CreateHeaderCell(string value)
        {
            return
                new Cell(new CellValue(value))
                {
                    DataType = CellValues.String,
                    StyleIndex = (uint)CellStyleIndexes.HeaderLine
                };
        }

        /// <summary>
        /// セルにハイパーリンクを設定します。
        /// </summary>
        /// <param name="cell">設定対象のセル。</param>
        /// <param name="path">ローカルファイルへのパス。</param>
        /// <param name="column">設定対象セルのカラムインデックス(0始まり)。</param>
        /// <param name="line">設定対象セルの行インデックス(0始まり)。</param>
        private void SetHyperlink(Cell cell, string path, string column, int line, string cellStr)
        {
            if (string.IsNullOrEmpty(path))
            {
                throw new ArgumentNullException("imagePath");
            }

            // 同じハイパーリンクリレーションシップがあるかどうかを調べ、なければ追加します。
            string refId;
            if (!_hyperlinkDic.TryGetValue(path, out refId))
            {
                var uri = new Uri(path, UriKind.RelativeOrAbsolute);
                var relShip = _wkSheet.WorksheetPart.AddHyperlinkRelationship(uri, true);
                _hyperlinkDic.Add(path, relShip.Id);
                refId = relShip.Id;
            }

            // 必要に応じて、ハイパーリンクコレクションをワークシートに追加。
            var hyperlinks = _wkSheet.Elements<Hyperlinks>().FirstOrDefault();
            if (null == hyperlinks)
            {
                hyperlinks = new Hyperlinks();
                _wkSheet.AppendChild(hyperlinks);
            }

            // ハイパーリンクを追加します。
            hyperlinks.AppendChild(
                new Hyperlink
                {
                    Reference = string.Format("{0}{1}", column, line),
                    Id = refId
                });

            // リンク先パスをセルのデータとして設定。
            SetSharedString(cell, cellStr/*path*/);
        }

        /// <summary>
        /// 指定した文字列を共有文字列テーブルに追加します。
        /// </summary>
        /// <param name="str">追加する文字列。</param>
        /// <returns>同じ文字列が既に存在している場合は、そのインデックス値、
        /// 存在していない場合は新たに追加し、最後のインデックス値を返します。</returns>
        private int InsertSharedStringItem(string str)
        {
            int idx;
            if (!_sharedStringDic.TryGetValue(str, out idx))
            {
                idx = _sharedStringDic.Count;
                // 一致するものがなかった場合は、追加。
                _sharedStringTable.AppendChild(
                    new SharedStringItem(
                        new Text(str) { Space = SpaceProcessingModeValues.Preserve }));
                _sharedStringDic.Add(str, idx);
            }

            return idx;
        }

        /// <summary>
        /// テキスト用のCellFormatオブジェクトを作成します。
        /// </summary>
        /// <param name="fillIdx">参照するFillへのインデックス値。</param>
        /// <returns>作成したCellFormatオブジェクト。</returns>
        private static CellFormat CreateTextCellFormat(int fillIdx)
        {
            return
                new CellFormat(
                    new Alignment
                    {
                        Vertical = VerticalAlignmentValues.Center,
                        WrapText = true
                    })
                {
                    ApplyAlignment = true,

                    FillId = (uint)fillIdx,
                    ApplyFill = true,

                    BorderId = 1,
                    ApplyBorder = true,

                    NumberFormatId = 49,
                    ApplyNumberFormat = true
                };
        }

        /// <summary>
        /// スタイルシートを作成します。
        /// </summary>
        /// <returns>Stylesheet オブジェクトを返します。</returns>
        private static Stylesheet CreateStyleSheet()
        {
            // Alignmentクラスを指定する場合は、Verticalプロパティを明示的に設定しないと、Bottomとなる。
            // Alignmentクラスを指定しない場合は、縦方向は中心になっている。
            return new Stylesheet(
                new Fonts(
                    new Font(                                           // Index 0 - The default font.
                        new FontSize { Val = 11 },
                        new FontName { Val = "ＭＳ Ｐゴシック" }),
                    new Font(                                           // Index 1 - ヘッダ行用
                        new Bold(),
                        new FontSize { Val = 11 },
                        new Color { Indexed = 9 },
                        new FontName { Val = "ＭＳ Ｐゴシック" }),
                    new Font(                                           // Index 2 - ハイパーリンク用
                        new Underline(),
                        new FontSize { Val = 11 },
                        new Color { Theme = 10 },
                        new FontName { Val = "ＭＳ Ｐゴシック" })),

                // Fillのコレクション
                // この並びは、FillIndexesと合わせること。
                new Fills(
                    new Fill(                                                           // The default fill.
                        new PatternFill { PatternType = PatternValues.None }),
                    new Fill(                                                           // The default fill of gray 125 (required)
                        new PatternFill { PatternType = PatternValues.Gray125 }),
                    new Fill(                                                           // ヘッダ行
                        new PatternFill(
                            new ForegroundColor { Rgb = new HexBinaryValue("FF333333") },
                            new BackgroundColor { Indexed = 64 })
                        { PatternType = PatternValues.Solid }),
                    new Fill(                                                           // ID-リソース
                        new PatternFill(
                            new ForegroundColor { Rgb = new HexBinaryValue("FFC0C0C0") },
                            new BackgroundColor { Indexed = 64 })
                        { PatternType = PatternValues.Solid }),
                    new Fill(                                                           // 新規翻訳テキスト
                        new PatternFill(
                            new ForegroundColor { Rgb = new HexBinaryValue("FFFF99CC") },
                            new BackgroundColor { Indexed = 64 })
                        { PatternType = PatternValues.Solid }),
                    new Fill(                                                           // 自動割当テキスト
                        new PatternFill(
                            new ForegroundColor { Rgb = new HexBinaryValue("FF99CCFF") },
                            new BackgroundColor { Indexed = 64 })
                        { PatternType = PatternValues.Solid }),
                    new Fill(                                                           // 新規コメント
                        new PatternFill(
                            new ForegroundColor { Rgb = new HexBinaryValue("FFFFFF00") },
                            new BackgroundColor { Indexed = 64 })
                        { PatternType = PatternValues.Solid }),
                    new Fill(                                                           // 修正テキスト
                        new PatternFill(
                            new ForegroundColor { Rgb = new HexBinaryValue("FF99CC00") },
                            new BackgroundColor { Indexed = 64 })
                        { PatternType = PatternValues.Solid })),
                new Borders(
                    new Border(                                     // Index 0 - The default border.
                        new LeftBorder(),
                        new RightBorder(),
                        new TopBorder(),
                        new BottomBorder(),
                        new DiagonalBorder()),
                    new Border(                                     // Index 1 - Applies a Left, Right, Top, Bottom border to a cell
                        new LeftBorder(
                            new Color { Indexed = 64 })
                        { Style = BorderStyleValues.Thin },
                        new RightBorder(
                            new Color { Indexed = 64 })
                        { Style = BorderStyleValues.Thin },
                        new TopBorder(
                            new Color { Indexed = 64 })
                        { Style = BorderStyleValues.Thin },
                        new BottomBorder(
                            new Color { Indexed = 64 })
                        { Style = BorderStyleValues.Thin },
                        new DiagonalBorder())),

                // セルスタイルフォーマットのコレクション
                new CellStyleFormats(
                    new CellFormat                                  // Index 0 - デフォルト
                    {
                        FontId = 0,
                        FillId = 0,
                        BorderId = 0
                    },
                    new CellFormat { FontId = 2 }),                 // Index 1 - ハイパーリンク

                // セルフォーマットのコレクション
                // この並びは、CellStyleIndexes を合わせること。
                new CellFormats(
                    new CellFormat                                          // デフォルト
                    {
                        FontId = 0,
                        FillId = 0,
                        BorderId = 0
                    },
                    new CellFormat(                                         // ヘッダ行
                        new Alignment
                        {
                            Vertical = VerticalAlignmentValues.Center,
                            Horizontal = HorizontalAlignmentValues.Center
                        })
                    {
                        ApplyAlignment = true,
                        FontId = 1,
                        ApplyFont = true,
                        FillId = (uint)FillIndexes.HeaderLine,
                        ApplyFill = true,
                        BorderId = 1,
                        ApplyBorder = true
                    },
                    new CellFormat                                          // ID-リソース名
                    {
                        FillId = (uint)FillIndexes.ResourceId,
                        ApplyFill = true,
                        BorderId = 1,
                        ApplyBorder = true
                    },
                    new CellFormat                                          // 画像
                    {
                        FontId = 2,
                        BorderId = 1,
                        ApplyBorder = true,
                        FormatId = 1            // セルスタイルフォーマットの指定
                    },
                    new CellFormat(                                         // 通常テキスト
                        new Alignment
                        {
                            Vertical = VerticalAlignmentValues.Center,
                            WrapText = true
                        })
                    {
                        ApplyAlignment = true,
                        BorderId = 1,
                        ApplyBorder = true,
                        NumberFormatId = 49,
                        ApplyNumberFormat = true
                    },

                    CreateTextCellFormat((int)FillIndexes.NewText),         // 新規翻訳テキスト
                    CreateTextCellFormat((int)FillIndexes.AutoText),        // 自動割当テキスト
                    CreateTextCellFormat((int)FillIndexes.NewComment),      // 新規コメント
                    CreateTextCellFormat((int)FillIndexes.CorrectedText))); // 修正テキスト
        }

        /// <summary>
        /// セルのFillへのインデックスのenum値
        /// </summary>
        private enum FillIndexes : uint
        {
            /// <summary>
            /// デフォルト
            /// </summary>
            Default,

            /// <summary>
            /// デフォルトのGray 125。
            /// サンプルなどによると必要なようだ。
            /// </summary>
            Gray125,

            /// <summary>
            /// ヘッダ行
            /// </summary>
            HeaderLine,

            /// <summary>
            /// IDやリソース名のセル
            /// </summary>
            ResourceId,

            /// <summary>
            /// 新規翻訳テキスト
            /// </summary>
            NewText,

            /// <summary>
            /// 自動で割当てられた翻訳テキスト
            /// </summary>
            AutoText,

            /// <summary>
            /// 新規コメント
            /// </summary>
            NewComment,

            /// <summary>
            /// 修正された翻訳テキスト
            /// </summary>
            CorrectedText
        }

        /// <summary>
        /// セルのスタイルへのインデックスのenum値
        ///
        /// 翻訳済みのxlsファイルからテキストをコピーする場合際に、Fillの値もコピーする処理を
        /// 行う場合は、未知のFillを追加する必要が出てくる可能性があります。
        /// (ここでいうテキストは、日本語テキスト、英語テキスト、開発者コメント、翻訳者コメント
        /// です。)
        /// 未知のFillを追加する場合は、そのFillを参照するCellFormatも新規に追加する必要があります。
        ///
        /// その作業をやりやすくするために、テキストセル用のCellFormatを
        /// 後ろの方にまとめています。
        ///
        /// 同じFillを参照するCellFormatがあるかどうかを探し、なければ
        /// 追加する、という処理を行っています。
        /// </summary>
        private enum CellStyleIndexes : uint
        {
            /// <summary>
            /// デフォルト
            /// </summary>
            Default,

            /// <summary>
            /// ヘッダ行
            /// </summary>
            HeaderLine,

            /// <summary>
            /// IDやリソース名のセル
            /// </summary>
            ResourceId,

            /// <summary>
            /// キャプチャ画像のリンク
            /// </summary>
            ImageLink,

            /// <summary>
            /// 通常テキスト
            /// </summary>
            NormalText,

            /// <summary>
            /// 新規翻訳テキスト
            /// </summary>
            NewText,

            /// <summary>
            /// 自動で割当てられた翻訳テキスト
            /// </summary>
            AutoText,

            /// <summary>
            /// 新規コメント
            /// </summary>
            NewComment,

            /// <summary>
            /// 修正された翻訳テキスト
            /// </summary>
            CorrectedText
        }
    }
}
