﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Reflection;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CommitContentChecker
{
    internal class CommitContentChecker
    {
        private GitExecuter m_GitExecuter { get; }
        private TargetExtensionListParser m_TargetExtensionListParser { get; }

        internal CommitContentChecker(string gitPath, TargetExtensionListParser targetExtensionListParser)
        {
            m_GitExecuter = new GitExecuter(gitPath);
            m_TargetExtensionListParser = targetExtensionListParser;
        }

        //文字コードがUTF-8でない、またはファイル書き込み・読み込みに問題があった場合
        //falseを返します。
        internal bool CheckCode(string codeFilename)
        {
            var checkOptions = m_TargetExtensionListParser.GetFileCheckOptions(codeFilename);

            // ファイルがチェック対象ではない場合は何もしない
            if (checkOptions == CheckOptions.None)
            {
                return true;
            }

            // --------------------------------
            // ファイル読み込み
            byte[] codeBytes = m_GitExecuter.GetStagedFileContent(codeFilename);
            bool existsBOM;

            // BOM チェック
            existsBOM = FormatterUtil.CheckBOM(codeBytes, checkOptions.HasFlag(CheckOptions.CheckBOM), $"{codeFilename}: ");
            if (checkOptions.HasFlag(CheckOptions.CheckBOM) && !existsBOM)
            {
                return false;
            }

            //--------------------------------
            // 自動整形
            //--------------------------------
            if (!checkOptions.HasFlag(CheckOptions.AutoFormatContents))
            {
                return true;
            }

            var encoding = new UTF8Encoding(existsBOM);
            // BOM 付きの場合は BOM のバイト列は無視してデコード
            string codeContent = encoding.GetString(codeBytes, encoding.GetPreamble().Length, codeBytes.Length - encoding.GetPreamble().Length);
            bool isFormatted = ContentFormatter.Format(ref codeContent, true, $"{codeFilename}: ").IsFormatted;

            // ファイルの修正がない場合、書き込み及び git add を行わない。
            if (!isFormatted)
            {
                return true;
            }

            //--------------------------------
            // ファイルへの書き込みと git add
            return UpdateFile(codeContent, codeFilename, existsBOM);
        }

        private bool UpdateFile(string codeContent, string codeFilename, bool existsBOM)
        {
            // stage されていない変更がある場合、ファイルが削除されていなければバックアップをとる。
            var status = m_GitExecuter.GetWorkTreeStatus(codeFilename);
            var backupFilename = string.Concat(codeFilename, ".backup");
            if (status != ' ')
            {
                Console.Out.WriteLine($"Changes not staged exist on '{codeFilename}'.");
                if (status != 'D')
                {
                    try
                    {
                        File.Copy(codeFilename, backupFilename, false);
                    }
                    catch (Exception exception)
                    {
                        Console.Error.WriteLine(codeFilename + ": Error: Failed to back up the file on disk");
                        Console.Error.WriteLine("Failed to auto-format.");
                        if (exception is IOException)
                        {
                            Console.Error.WriteLine($"'{backupFilename}' already exists. Please remove '{backupFilename}'.");
                        }
                        Console.Error.WriteLine(exception);
                        return false;
                    }
                    Console.Out.WriteLine($"Backed up '{codeFilename}' to '{backupFilename}'.");
                }
            }

            // ファイル書き出し及び git add
            if (!Utils.SaveFile(codeContent, codeFilename, existsBOM))
            {
                return false;
            }
            m_GitExecuter.Add(codeFilename);
            Console.Out.WriteLine("Staged " + codeFilename);

            // ファイルのバックアップをとった場合、ワークツリーのファイルに対して自動整形してバックアップを削除する。
            if (status != ' ')
            {
                // ワークツリー上でファイルが削除されていた場合は、自動整形のために書き出したファイルを削除する。
                if (status == 'D')
                {
                    try
                    {
                        File.Delete(codeFilename);
                    }
                    catch
                    {
                        // ファイル削除に失敗した場合は警告を出力してコミットを続行する。
                        Console.Out.WriteLine(codeFilename + ": Warning: Failed to delete the file on disk");
                    }
                    return true;
                }

                // バックアップからワークツリーのファイルを復元する。
                try
                {
                    File.Copy(backupFilename, codeFilename, true);
                    Console.Out.WriteLine($"Restored '{codeFilename}' from '{backupFilename}'.");
                }
                catch (Exception exception)
                {
                    Console.Error.WriteLine(codeFilename + ": Error: Failed to restore the file on disk");
                    Console.Error.WriteLine(exception);
                    Console.Error.WriteLine($"Please manually restore the file from '{backupFilename}'.");
                    return true;
                }
                try
                {
                    File.Delete(backupFilename);
                }
                catch
                {
                    // ファイル削除に失敗した場合は警告を出力してコミットを続行する。
                    Console.Out.WriteLine(backupFilename + ": Warning: Failed to delete file");
                }

                // ワークツリーのファイルを自動整形（失敗してもコミットは続行）
                string worktreeCodeContent;
                if (!Utils.LoadFile(codeFilename, existsBOM, out worktreeCodeContent))
                {
                    Console.Out.WriteLine(codeFilename + ": Warning: Failed to read from the file on disk");
                    Console.Out.WriteLine($"Failed to auto-format '{codeFilename}' on disk. (The file on disk remains unchanged. Continuing the operation...)");
                    return true;
                }
                if (!ContentFormatter.Format(ref worktreeCodeContent, true, $"{codeFilename}: ").IsFormatted)
                {
                    return true;
                }
                if (!Utils.SaveFile(worktreeCodeContent, codeFilename, existsBOM))
                {
                    Console.Out.WriteLine(codeFilename + ": Warning: Failed to write to the file on disk");
                    Console.Out.WriteLine($"Failed to auto-format '{codeFilename}' on disk. (The file on disk remains unchanged. Continuing the operation...)");
                    return true;
                }
            }
            return true;
        }
    }
}
