﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using Nintendo.ToolFoundation.IO;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Collections;

namespace CodeChecker
{
    internal class Program
    {
        internal static void Main(string[] args)
        {
            // コマンドライン引数を解析
            var parseResult = ParseCommandLine(args);
            if (parseResult.Params == null)
            {
                Environment.Exit(parseResult.ExitCode);
                return;
            }

            var @params = parseResult.Params;

            TargetExtensionListParser targetExtensionListParser = null;
            try
            {
                targetExtensionListParser = new TargetExtensionListParser(@params.TargetExtensionListFile);
            }
            catch (Exception)
            {
                Environment.Exit(1);
            }

            var files = @params.Recurse ?
                @params.Paths.SelectMany(p => Directory.EnumerateFiles(p, "*", SearchOption.AllDirectories)) :
                @params.Paths;

            bool checkResult = true;
            foreach (var file in files)
            {
                checkResult &= CodeChecker.CheckCode(file, targetExtensionListParser);
            }

            if (!checkResult)
            {
                Console.Error.WriteLine("Found coding check error.");
                Environment.Exit(1);
            }
            Console.Out.WriteLine("Success in coding check.");
            Environment.Exit(0);
        }

        private static ParseCommandLineResult ParseCommandLine(string[] args)
        {
            var s = new CommandLineParserSettings()
            {
                ApplicationDescription = "Check if specified file(s) conform to the Siglo coding conventions.",
            };

            try
            {
                CodeCheckerParams @params;
                if (!new CommandLineParser(s).ParseArgs(args, out @params))
                {
                    return new ParseCommandLineResult(0);
                }
                return new ParseCommandLineResult(@params);
            }
            catch (ArgumentException)
            {
                return new ParseCommandLineResult(1);
            }
        }

        private class ParseCommandLineResult
        {
            public CodeCheckerParams Params { get; }
            public int ExitCode { get; }

            public ParseCommandLineResult(CodeCheckerParams @params)
            {
                this.Params = @params;
            }

            public ParseCommandLineResult(int exitCode)
            {
                this.Params = null;
                this.ExitCode = exitCode;
            }
        }
    }
}
