﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VsSolutionLibrary
{
    internal class Dumper
    {
        // Guid を文字列に変換して返す。
        private static string GuidToString(Guid guid)
        {
            // *.sln では Guid を大文字で表現する。
            return guid.ToString("B").ToUpperInvariant();
        }

        private static void DumpHeader(StringBuilder stringBuilder, VsSolution solution)
        {
            switch (solution.Version)
            {
                case VsSolutionVersion.VisualStudio2012:
                    stringBuilder.Append(@"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2012
");
                    break;
                case VsSolutionVersion.VisualStudio2013:
                    stringBuilder.Append(@"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 2013
VisualStudioVersion = 12.0.40629.0
MinimumVisualStudioVersion = 10.0.40219.1
");
                    break;
                case VsSolutionVersion.VisualStudio2015:
                    stringBuilder.Append(@"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 14
VisualStudioVersion = 14.0.25123.0
MinimumVisualStudioVersion = 10.0.40219.1
");
                    break;
                case VsSolutionVersion.VisualStudio2017:
                    stringBuilder.Append(@"
Microsoft Visual Studio Solution File, Format Version 12.00
# Visual Studio 15
VisualStudioVersion = 15.0.26228.10
MinimumVisualStudioVersion = 10.0.40219.1
");
                    break;
                default:
                    throw new ArgumentException();
            }
        }

        private static void DumpProjects(StringBuilder stringBuilder, VsSolution solution)
        {
            foreach (Guid projectGuid in solution.Projects)
            {
                VsSolutionProjectProperty property = solution.GetProjectProperty(projectGuid);
                Guid projectTypeGuid = VsProjectTypeUtility.GetGuidFromProjectType(property.ProjectType);

                stringBuilder.AppendFormat("Project(\"{0}\") = \"{1}\", \"{2}\", \"{3}\"", GuidToString(projectTypeGuid), property.Name, property.Path, GuidToString(projectGuid));
                stringBuilder.AppendLine();

                if (property.Dependencies != null && property.Dependencies.Count() != 0)
                {
                    stringBuilder.AppendLine("\tProjectSection(ProjectDependencies) = postProject");
                    foreach (Guid guid in property.Dependencies)
                    {
                        stringBuilder.AppendFormat("\t\t{0} = {0}", GuidToString(guid));
                        stringBuilder.AppendLine();
                    }
                    stringBuilder.AppendLine("\tEndProjectSection");
                }

                stringBuilder.AppendLine("EndProject");
            }
        }

        private static void DumpGlobal(StringBuilder stringBuilder, VsSolution solution)
        {
            stringBuilder.AppendLine("Global");

            // SolutionConfigurationPlatforms
            if (solution.Configurations.Count() > 0)
            {
                stringBuilder.AppendLine("\tGlobalSection(SolutionConfigurationPlatforms) = preSolution");
                foreach (var configuration in solution.Configurations)
                {
                    stringBuilder.AppendFormat("\t\t{0} = {0}", configuration.ToString());
                    stringBuilder.AppendLine();
                }
                stringBuilder.AppendLine("\tEndGlobalSection");
            }

            // ProjectConfigurationPlatforms
            var projectsWithoutSolutionFolder = GetProjectWithoutSolutionFolder(solution);
            if (solution.Configurations.Count() > 0 && projectsWithoutSolutionFolder.Count() > 0)
            {
                stringBuilder.AppendLine("\tGlobalSection(ProjectConfigurationPlatforms) = postSolution");
                foreach (var projectGuid in projectsWithoutSolutionFolder)
                {
                    foreach (var solutionConfiguration in solution.Configurations)
                    {
                        VsProjectConfiguration projectConfiguration;
                        projectConfiguration = solution.GetProjectConfiguration(solutionConfiguration, projectGuid);

                        // ActiveCfg
                        stringBuilder.AppendFormat("\t\t{0}.{1}.ActiveCfg = {2}", GuidToString(projectGuid), solutionConfiguration.ToString(), projectConfiguration.ConfigurationPair.ToString());
                        stringBuilder.AppendLine();

                        // Build.0
                        if (projectConfiguration.EnableBuild)
                        {
                            stringBuilder.AppendFormat("\t\t{0}.{1}.Build.0 = {2}", GuidToString(projectGuid), solutionConfiguration.ToString(), projectConfiguration.ConfigurationPair.ToString());
                            stringBuilder.AppendLine();
                        }
                    }
                }
                stringBuilder.AppendLine("\tEndGlobalSection");
            }

            // SolutionProperties
            if (solution.SolutionProperties.Count() > 0)
            {
                stringBuilder.AppendLine("\tGlobalSection(SolutionProperties) = preSolution");
                foreach (var solutionProperty in solution.SolutionProperties)
                {
                    stringBuilder.AppendFormat("\t\t{0} = {1}", solutionProperty.Key, solutionProperty.Value);
                    stringBuilder.AppendLine();
                }
                stringBuilder.AppendLine("\tEndGlobalSection");
            }

            // NestedProjects
            if (solution.NestRelations.Count() > 0)
            {
                stringBuilder.AppendLine("\tGlobalSection(NestedProjects) = preSolution");
                foreach (var relation in solution.NestRelations)
                {
                    stringBuilder.AppendFormat("\t\t{0} = {1}", GuidToString(relation.Child), GuidToString(relation.Parent));
                    stringBuilder.AppendLine();
                }
                stringBuilder.AppendLine("\tEndGlobalSection");
            }

            stringBuilder.AppendLine("EndGlobal");
        }

        public static string Dump(VsSolution solution)
        {
            StringBuilder stringBuilder = new StringBuilder();

            DumpHeader(stringBuilder, solution);
            DumpProjects(stringBuilder, solution);
            DumpGlobal(stringBuilder, solution);

            return stringBuilder.ToString();
        }

        private static IEnumerable<Guid> GetProjectWithoutSolutionFolder(VsSolution solution)
        {
            return solution.Projects.Where(x => solution.GetProjectProperty(x).ProjectType != VsProjectType.SolutionFolder);
        }
    }
}
