<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Runs NxAgingHelperCommand with Bluetooth tests.  These are stress tests that should run for a long time.

    .DESCRIPTION
        - Runs all 3 applications in parallel:
           - Runs NxAgingHelperCommand
           - Runs testBluetooth_Pairing.nsp so the device can start pairing
           - Runs a series of ControlExtCon.exe commands on the US Writer device to automate button presses on the controller
        
#>

param
(
    [string]
    # SDK root directory
    $sigloRoot =
    ($env:NINTENDO_SDK_ROOT | ? { $_ } | % { $_.Trim() }),

    [string]
    # Device's IP address
    $TargetAddress =
    ($env:TARGET_ADDRESS | ? { $_ } | % { $_.Trim() }),
    
    [int]
    # Length of time to run in seconds
    $TimeToRunSeconds =
    ($env:TEST_NXAGING_TIME_SECONDS | ? { $_ } | % { $_.Trim() }),

    [string]
    # eg. Tests\Outputs\NX-NXFP2-a64\Tests\testBluetooth_Pairing\Develop\testBluetooth_Pairing.nsp
    $TestListApp =
    ($env:TESTRUNNER_EXE_LIST),

    [string]
    # eg. NxAgingConfig_Default.yml
    $NxAgingConfigFile =
    ($env:TEST_NXAGING_CONFIG_FILE),

    [string]
    # eg.  --enable-sdev-usb-event --enable-power-button-event
    $NxAgingCommands =
    ($env:TEST_NXAGING_COMMANDS),
    
    [string]
    # Port number of US Writer
    #   eg. com11
    $PortOfUsWriter =
    ($env:PORT_US_WRITER | ? { $_ } | % { $_.Trim() })
)


# go to the root directory
Set-Location -Path $sigloRoot

# constants
$NX_AGING_WORKING_DIR = "$sigloRoot\Integrate\Outputs\AnyCPU\Tools\NxAgingTools\NxAgingHelperCommand\Release"
$NX_AGING_EXE = "$sigloRoot\Integrate\Outputs\AnyCPU\Tools\NxAgingTools\NxAgingHelperCommand\Release\NxAgingHelperCommand.exe"


# Run NxAgingHelperCommand.exe, and then the tests in parallel for a specified duration
$JobList = @{
    
    # NxAgingHelperCommand
    "NxAgingHelperCommand" = [scriptblock]::Create("$NX_AGING_EXE --config $sigloRoot\Integrate\Sources\Tools\NxAgingTools\Configuration\$NxAgingConfigFile --event-log-path $NX_AGING_WORKING_DIR\NxAgingHelper_Event.log --enable-uart-log --uart-log-path $NX_AGING_WORKING_DIR\NxAgingHelper_Uart.log --sdev $TargetAddress --time $TimeToRunSeconds $NxAgingCommands");
    
    
    # US Writer commands
    "UsWriter" = {
        param($LocalSigloRoot,$LocalTimeToRunSeconds,$LocalNxAgingCommands,$LocalTestListApp,$LocalPortOfUsWriter)
        
        &   {
            # run commands on the US writer until time is up
            Set-Location -Path $LocalSigloRoot
            $timeToRunTestsSeconds = $LocalTimeToRunSeconds
            $stopWatch = [Diagnostics.Stopwatch]::StartNew()
            $elapsedTime = 0
            $elapsedTimeSeconds = 0

            # send US writer commands every 2 secs
            while ($elapsedTimeSeconds -lt $timeToRunTestsSeconds)
            {
                # delete all controller pairings
                $processDelete = [Diagnostics.Process]::Start("Tools\CommandLineTools\RunOnTarget.exe", "./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- controller delete-all-pairing")
                $processDelete.WaitForExit(5000)

                # start the pairing app on the device
                Write-Host "--- Starting $LocalTestListApp"
                Start-Process -FilePath "Tools\CommandLineTools\RunOnTarget.exe" -ArgumentList "$LocalTestListApp"

                # press pair button, then wait until hardware button is accepted
                Write-Host "Sending controller pair command"
                $processPair = [Diagnostics.Process]::Start("Integrate\Scripts\NX\GamePad\ControlExtCon.exe", "--port $LocalPortOfUsWriter reset-pair")
                $processPair.WaitForExit(20000)
                
                # press page button, then wait until harware button is accepted
                Write-Host "Sending controller page command"
                $processPage = [Diagnostics.Process]::Start("Integrate\Scripts\NX\GamePad\ControlExtCon.exe", "--port $LocalPortOfUsWriter reset-page")
                $processPage.WaitForExit(10000)
                
                # press home button, then wait until hardware button is accepted
                Write-Host "Sending controller Home button command"
                $processPower = [Diagnostics.Process]::Start("Integrate\Scripts\NX\GamePad\ControlExtCon.exe", "--port $LocalPortOfUsWriter reset-poweron")
                $processPower.WaitForExit(10000)
                
                $elapsedTime = $stopWatch.Elapsed
                $elapsedTimeSeconds = $elapsedTime.TotalSeconds
            }
        }
    }
}


# Report the status of running jobs this many seconds
$POLL_TIME = 600

try {

    # enable logging on the device
    Write-Host "Enable logging on the device SD card"
    & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug enable-sd-card-logging
    & Tools\CommandLineTools\ControlTarget.exe reset --target $TargetAddress
    & Tools\CommandLineTools\ControlTarget.exe connect
    
    # Start all of the jobs
    $Jobs = $(
        ForEach ($item in $JobList.GetEnumerator()) {
            Write-Host "Starting $($item.Name) as $($item.Value)"
            Start-Job -Name $item.Name -ScriptBlock $item.Value -ArgumentList $sigloRoot,$TimeToRunSeconds,$NxAgingCommands,$TestListApp,$PortOfUsWriter
        }
    )

    # Wait for all the jobs to finish
    # We could just use "Wait-Job $Jobs", except that with no output for hours, TC might think the job is hung.
    While($true) {
        Wait-Job $Jobs -Timeout $POLL_TIME | Out-Null
        $Running = $($Jobs | Where-Object {$_.State -eq "Running"})
        if ($Running) {
            Write-Host "Waiting for jobs: $($Running | Select -expand Name)"
        } else {
            Write-Host "Jobs Completed"
            Break
        }
    }
}

# This is in a finally block, so that any remaining jobs will be stopped if this script is interrupted.
finally {
    Set-Location -Path $sigloRoot

    # disable logging on the device
    Write-Host "Disable logging on the device SD card"
    & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- debug disable-sd-card-logging
    & Tools\CommandLineTools\ControlTarget.exe reset --target $TargetAddress
    & Tools\CommandLineTools\ControlTarget.exe connect
    
    # The normal output should be redirected to a log file, but startup errors may be displayed here.
    # Use the teamcity macros to differentiate output from each command.
    $Jobs | % {
        $JobName = $_.Name
        Write-Host "##teamcity[testStarted name='$JobName' captureStandardOutput='true']`n"
        Stop-Job $_
        $Job = Receive-Job $_ -Keep -Verbose
        $Job = $Job -join "`n"
        Write-Host $Job
        if ($Job.Error) {
            Write-Host "##teamcity[testFailed name='$JobName']"
        }

        if ($Job.PSEndTime) {
            $duration = ($Job.PSEndTime - $Job.PSBeginTime).TotalMilliseconds
        } else {
            $duration = 0
        }
        Write-Host "##teamcity[testFinished name='$JobName' duration='$duration']"
    }
}

