<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        - Prints out Bluetooth perf numbers

    .DESCRIPTION
        - Bluetooth perf numbers are printed out in TeamCity format for:
            - time to pair
            - time to connect
#>

param
(
    [string]
    # SDK root directory
    $SigloRoot =
    ($env:NINTENDO_SDK_ROOT | ? { $_ } | % { $_.Trim() }),

    [string]
    # Device's IP address
    $TargetAddress =
    ($env:TARGET_ADDRESS | ? { $_ } | % { $_.Trim() }),
    
    [int]
    # Number of times to run the tests to get perf numbers
    $NumTestIterations =
    ($env:NUM_TEST_ITERATIONS),

    [string]
    # eg. Tests\Outputs\NX-NXFP2-a64\Tests\testBluetooth_Pairing\Develop\testBluetooth_Pairing.nsp
    $TestListApp =
    ($env:TESTRUNNER_EXE_LIST),
    
    [string]
    # Port number of US Writer
    #   eg. com11
    $PortOfUsWriter =
    ($env:PORT_US_WRITER | ? { $_ } | % { $_.Trim() })
)


# Telnet port
$TelnetPort = 10023
# Number of bytes to read at a time
$BytesToRead = 1024
# Max time in milliseconds to wait for telnet log
$MaxWaitTimeMS = 10000
# Sleep increments in milliseconds to wait to check for telnet log
$SleepIncrementsMS = 500

# Controller pair command success string in Telnet logs
$PairSuccess = "[bluetooth] BSA_SEC_AUTH_CMPL_EVT received"
# Controller page command success string in Telnet logs
$PageSuccess = "[bluetooth] BSA_HH_OPEN_EVT"


# reads the telnet port until a specific message is received 
function ReadTelnetPort ($LocalTargetAddress,$LocalTelnetPort,$LocalTelnetMessage)
{
    # flag to indicate whether the telnet message we're looking for is found
    $flagFoundMessage = $false
    
    # Telnet into device port
    $socket = New-Object System.Net.Sockets.TCPClient($LocalTargetAddress,$LocalTelnetPort)
    if ($socket)
    {  
        $stream = $socket.GetStream()
        $buffer = New-Object System.Byte[] $BytesToRead
        $encoding = New-Object System.Text.AsciiEncoding
        
        $readResult = ""
        $numTimeLoop = $MaxWaitTimeMS / $SleepIncrementsMS
        for ($i = 0; ($i -lt $numTimeLoop); $i++)
        {
            $read = $stream.Read($buffer, 0, $BytesToRead)
            $readResult += ($encoding.GetString($buffer, 0, $read))

            if ($readResult.Contains($LocalTelnetMessage))
            {
                $flagFoundMessage = $true
                break
            }
            Start-sleep -Milliseconds $SleepIncrementsMS
        }

        $stream.Close()
        return $flagFoundMessage
    }
}


# prints out the strings for TeamCity
function PrintPerfNumbers($LocalKeyName,$LocalArrayInput)
{
    # count the number of 0 in array
    $numZeros = 0
    
    for ($i = 0; $i -lt $LocalArrayInput.Count; $i++)
    {
        if ($LocalArrayInput[$i] -eq 0)
        {
            $numZeros++;
        }
    }
    
    # print the average time
    $totalTime = ($LocalArrayInput | Measure-Object -sum).sum
    $averageTime = [math]::Round($totalTime / ($LocalArrayInput.Count - $numZeros))
    $maxTime = [math]::Round(($LocalArrayInput | Measure-Object -max).maximum)
    $successRate = [math]::Round(($LocalArrayInput.Count - $numZeros) / $LocalArrayInput.Count * 100)
    
    Write-Host ("##teamcity[buildStatisticValue key='{0}_avg_ms' value='{1}']" -f $LocalKeyName, $averageTime)
    Write-Host ("##teamcity[buildStatisticValue key='{0}_max_ms' value='{1}']" -f $LocalKeyName, $maxTime)
    Write-Host ("##teamcity[buildStatisticValue key='{0}_success_%' value='{1}']" -f $LocalKeyName, $successRate)
} 


# time parameters
$stopWatch = [Diagnostics.Stopwatch]::StartNew()
$elapsedTime = 0
$arrayPairTime = New-Object string[] $NumTestIterations
$arrayButtonTime = New-Object string[] $NumTestIterations

try
{
    # make sure Target Manager is connected
    & Tools\CommandLineTools\ControlTarget.exe connect

    # run tests until time has elapsed
    for ($currentIteration = 0; $currentIteration -lt $NumTestIterations; $currentIteration++)
    {
        # delete all controller pairings
        & Tools\CommandLineTools\RunOnTarget.exe ./Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp -- controller delete-all-pairing

        # start the pairing app on the device
        & Tools\CommandLineTools\RunOnTarget.exe --no-wait $TestListApp

        # press pair button on the controller, then verify in telnet log that the controller paired
        Write-Host "Sending controller pair command"
        $stopWatch.Reset()
        $stopWatch.Start()
        & Integrate\Scripts\NX\GamePad\ControlExtCon.exe --port $PortOfUsWriter reset-pair
        $flagTelnetPair = ReadTelnetPort $TargetAddress $TelnetPort $PairSuccess
        if ($flagTelnetPair -eq $true)
        {
            $arrayPairTime[$currentIteration] = $stopWatch.Elapsed.TotalMilliseconds
        }
        else
        {
            # use 0 as a failure number
            $arrayPairTime[$currentIteration] = 0
        }
        
        # US writer handles button presses immediately after each other with delay
        Start-sleep -s 3

        # press page button on the controller, then verify in telnet log is the expected result
        Write-Host "Sending controller page command"
        $stopWatch.Reset()
        $stopWatch.Start()
        & Integrate\Scripts\NX\GamePad\ControlExtCon.exe --port $PortOfUsWriter reset-page
        $flagTelnetPage = ReadTelnetPort $TargetAddress $TelnetPort $PageSuccess
        if ($flagTelnetPage -eq $true)
        {
            $arrayButtonTime[$currentIteration] = $stopWatch.Elapsed.TotalMilliseconds
        }
        else
        {
            # use 0 as a failure number
            $arrayButtonTime[$currentIteration] = 0
        }
    }
}

finally
{
    # reset the device
    & Tools\CommandLineTools\ControlTarget.exe reset --hard --target $TargetAddress    
}

# print the perf numbers
PrintPerfNumbers "pair" $arrayPairTime
PrintPerfNumbers "connect" $arrayButtonTime
