﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        修理用初期化イメージ(*.repair.initimg)を用いて修理システムの初期化を行います。

    .DESCRIPTION
        修理用初期化イメージ(*.repair.initimg)を用いて修理システムの初期化を行います。
#>

[CmdletBinding()]
Param
(
    [string]
    $KeyType = 'K5',

    [string]
    $ConnectionType = 'Manu',

    [string]
    $SignedType = 'Unsigned',

    [string]
    $BootType = 'ManuBoot',

    [string]
    $StorageSize = '32G',

    [string]
    $DeveloperType = 'Repair',

    [string]
    $BuildType = 'Develop',

    [string]
    $Platform = 'NX-NXFP2-a64',

    [int]
    $BootLoaderVersion = [int]::MaxValue,

    [string]
    $SkipBlVersionCheck = "false", # "true" or "false"

    [string]
    $DevicePath = ""
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../../Modules/Utility"

$SystemImagesDirectory = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/"

$DevicePathOption = ""
if(![System.String]::IsNullOrEmpty($DevicePath))
{
    $DevicePathOption = "--usb_device_path $DevicePath"
}

# K5, K6 の書き込みは RepairRecoveryImageWriter を使用
if ($KeyType -eq 'K6' -Or $KeyType -eq 'K5')
{
    if($SkipBlVersionCheck -eq "true")
    {
        $script:RepairRecoveryImageWriter    = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RepairRecoveryImageWriter/RepairRecoveryImageWriterSkipVerCheck.exe"
        $script:RepairRecoveryImageWriterRcm = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RepairRecoveryImageWriter/RepairRecoveryImageWriterSkipVerCheck.$KeyType.rcm"
        $script:BlVersionOption = ""
    }
    else
    {
        $script:RepairRecoveryImageWriter    = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RepairRecoveryImageWriter/RepairRecoveryImageWriter.exe"
        $script:RepairRecoveryImageWriterRcm = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RepairRecoveryImageWriter/RepairRecoveryImageWriter.$KeyType.rcm"
        $script:BlVersionOption = "--blVersion `"$BootLoaderVersion`""
    }

    if($KeyType -eq 'K6')
    {
        $script:ZeroBct               = "$(Get-NintendoSdkRootPath)/Externals/NxProdBinaries/Repair/BootLoader/Zero.bct"
        $script:SafeBct               = "$(Get-NintendoSdkRootPath)/Externals/NxProdBinaries/Repair/BootLoader/bl-safe.bct"
        $script:SafeBl                = "$(Get-NintendoSdkRootPath)/Externals/NxProdBinaries/Repair/BootLoader/bl-safe.bin"
        $script:Package2              = "$(Get-NintendoSdkRootPath)/Externals/NxProdBinaries/Repair/Package2Signed/p2repair-initialize-prod.bin"
        $script:SignedSystemPartition = "$(Get-NintendoSdkRootPath)/Externals/NxProdBinaries/Repair/SystemPartition/repair-ssp-prod.fatimg"
    }
    else
    {
        $SignedSystemPartitionName = "${DeveloperType}SSP-$ConnectionType-P2$SignedType"

        if($SignedType -eq "Signed")
        {
            $script:Package2 = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/Package2Signed/$BuildType/p2repair-initialize.bin"
            $script:SignedSystemPartition = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/SystemPartition/$SignedSystemPartitionName/$BuildType/$SignedSystemPartitionName.fatimg"
        }
        else
        {
            $script:Package2 = "${SystemImagesDirectory}/Package2Images/RepairPackage2ForInitialize/$BuildType/RepairPackage2ForInitialize.package2.img"
            $script:SignedSystemPartition = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/SystemPartition/$SignedSystemPartitionName/$BuildType/$SignedSystemPartitionName.fatimg"
        }
        $script:ZeroBct               = "$(Get-NintendoSdkRootPath)/Programs/Eris/Sources/InitialImages/BreakBct/Zero.bct"
        $script:SafeBct               = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Safe-$KeyType.bct"
        $script:SafeBl                = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Safe-$KeyType.bl"

    }

    Test-EnsureExistPath $RepairRecoveryImageWriter
    Test-EnsureExistPath $RepairRecoveryImageWriterRcm
    Test-EnsureExistPath $ZeroBct
    Test-EnsureExistPath $SafeBct
    Test-EnsureExistPath $SafeBl
    Test-EnsureExistPath $Package2
    Test-EnsureExistPath $SignedSystemPartition

    Invoke-CriticalCommand "& `"$RepairRecoveryImageWriter`" ``
        --rcm `"$RepairRecoveryImageWriterRcm`" ``
        --initimg `"$SignedSystemPartition`" ``
        --package2 `"$Package2`" ``
        --safebct `"$SafeBct`" ``
        --safebl `"$SafeBl`" ``
        $BlVersionOption ``
        --zerobct `"$ZeroBct`" ``
        $DevicePathOption `` "
}
else
{

    # その他の鍵世代は RcmWriter を使用
    $RcmWriter             = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.exe"
    $RcmWriterRcm          = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.$KeyType.rcm"
    $RcmWriterRcmOld       = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.$KeyType.old.rcm"
    $RcmWriterXml          = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RepairSystem.xml"
    $ZeroBct               = "$(Get-NintendoSdkRootPath)/Programs/Eris/Sources/InitialImages/BreakBct/Zero.bct"
    $SafeBct               = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Safe-$KeyType.bct"
    $SafeBl                = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/BootLoader/Safe-$KeyType.bl"
    $SignedSystemPartition = "${SystemImagesDirectory}/SystemPartition/Manufacture1stInitializeSystemPartition/$BuildType/Manufacture1stInitializeSystemPartition.fatimg"
    $RcmWriterAdditionalOptions = ""

    if($KeyType -eq 'M2')
    {
        $RcmWriterAdditionalOptions = "--old_rcm `"$RcmWriterRcmOld`""
    }

    if($SigndType -eq "Signed")
    {
        $Package2 = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/Package2Signed/$BuildType/p2repair-intialize.bin"
    }
    else
    {
        $Package2 = "${SystemImagesDirectory}/Package2Images/RepairPackage2ForInitialize/$BuildType/RepairPackage2ForInitialize.package2.img"
    }

    $OutputDirectory             = "$(Get-NintendoSdkRootPath)/Integrate/Outputs/RcmWriter"
    $OutputRcmWriterXml          = "$OutputDirectory/RepairSystem.xml"
    $OutputZeroBctPath           = "$OutputDirectory/Zero.bct"
    $OutputSafeBctPath           = "$OutputDirectory/Safe.bct"
    $OutputSafeBlPath            = "$OutputDirectory/Safe.bl"
    $OutputSignedSystemPartition = "$OutputDirectory/SignedSystemPartition.fatimg"
    $OutputPackage2              = "$OutputDirectory/Package2.img"

    Test-EnsureExistPath $RcmWriter
    Test-EnsureExistPath $RcmWriterRcm
    Test-EnsureExistPath $RcmWriterXml
    Test-EnsureExistPath $ZeroBct
    Test-EnsureExistPath $SafeBct
    Test-EnsureExistPath $SafeBl
    Test-EnsureExistPath $SignedSystemPartition
    Test-EnsureExistPath $Package2

    if (!(Test-Path $OutputDirectory))
    {
        mkdir $OutputDirectory > $null
    }

    # Copy files
    cp $RcmWriterXml          $OutputRcmWriterXml
    cp $ZeroBct               $OutputZeroBctPath
    cp $SafeBct               $OutputSafeBctPath
    cp $SafeBl                $OutputSafeBlPath
    cp $SignedSystemPartition $OutputSignedSystemPartition
    cp $Package2              $OutputPackage2

    # RcmWriter
    Invoke-CriticalCommand "& `"$RcmWriter`" --rcm `"$RcmWriterRcm`" --list `"$OutputRcmWriterXml`" $RcmWriterAdditionalOptions"
}

Wait-Seconds 10
