<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Initialize edev by system updater.

    .DESCRIPTION
        Initialize edev by system updater.
#>
[CmdletBinding()]
Param
(
    [string]
    $BuildType = 'Develop',
    
    [string]
    $Platform = 'NX-NXFP2-a64',
    
    [string]
    $KeyType = 'K5', # K1, K2, K3, K5, M1, M2, Auto
    
    [string]
    $BootType = 'Nand', # Nand, List
    
    [string]
    $ConnectionType = 'Usb',
    
    [string]
    $SignedType = 'Signed', # Unsigned, Signed
    
    [string]
    $StorageSize = '64G', # 32G, 64G 
    
    [string]
    $DeveloperType = 'Internal',  # Internal, Public

    [string]
    $AdditionalOptions = '',

    [Int]
    $DetectTargetRetryCount = 5,

    [string] 
    $Suffix = ''
)

$QspiConnectionType = "Usb"

Write-Host "Initialization Parameters:"
Write-Host "  BuildType:      $BuildType"
Write-Host "  Platform:       $Platform"
Write-Host "  KeyType:        $KeyType"
Write-Host "  ConnectionType: $ConnectionType"
Write-Host "  BootType:       $BootType"
Write-Host "  SignedType:     $SignedType"
Write-Host "  StorageSize:    $StorageSize"
Write-Host "  DeveloperType:  $DeveloperType"

if($Suffix.Length -ne 0)
{
    Write-Host "  Suffix:         $Suffix"
}

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}/../../Modules/Error"
Import-Module "${scriptDirectoryPath}/../../Modules/Path"
Import-Module "${scriptDirectoryPath}/../../Modules/Utility"
Import-Module "${scriptDirectoryPath}/../../Modules/Runner"
Import-Module "${scriptDirectoryPath}/../../Modules/UsbTarget"

$RunOnTarget        = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/RunOnTargetPrivate.exe"
$ControlTarget      = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
$InitialImageName   = "NX-$KeyType-$ConnectionType-$SignedType-$BootType-$StorageSize-$DeveloperType"

$InitialImagePath   = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/InitialImages/$InitialImageName/$BuildType/$InitialImageName$Suffix.initimg"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if($KeyType -eq "K4")
{
    $SystemUpdater = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/$Platform/TargetTools/SystemUpdaterHostFs/$BuildType/SystemUpdaterHostFs$KeyType.nsp"
}
else
{
    $SystemUpdater = "$(Get-NintendoSdkRootPath)/Programs/Chris/Outputs/$Platform/TargetTools/SystemUpdaterHostFs/$BuildType/SystemUpdaterHostFs.nsp"
}


Stop-ProcessesForSystemInitialize
Unregister-AllTargets

$RcmWriter          = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.exe"
$RcmWriterRcm       = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.$KeyType.rcm"
$RcmWriterRcmOld    = "$(Get-NintendoSdkRootPath)/Externals/NxSystemImages/RcmWriter/RcmWriter.$KeyType.old.rcm"
$RcmWriterAdditionalOptions = ""
if($KeyType -eq 'M1')
{
    $RcmWriterXml   = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-NxDev-WithBootConfig-M1.xml"
}
elseif($KeyType -eq 'M2')
{
    $RcmWriterXml   = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-NxDev-WithBootConfig-M2.xml"
    $RcmWriterAdditionalOptions = "--old_rcm `"$RcmWriterRcmOld`""
}
else
{
    $RcmWriterXml   = "$(Get-NintendoSdkRootPath)/Integrate/Sources/Configs/RcmWriter/RecoverySystem-NxDev-WithBootConfig.xml"
}
$BootConfig   = "$(Get-NintendoSdkRootPath)/Externals/NxBootConfigs/bc-dummy.bin"
$RecoveryWriterName = "RecoveryWriter-$KeyType-$QspiConnectionType-$SignedType"
$RecoveryWriter = "$(Get-NintendoSdkRootPath)/Programs/Eris/Outputs/$Platform/SystemImages/QspiBootImages/$RecoveryWriterName/$BuildType/$RecoveryWriterName$Suffix.qspi.img"

$OutputDirectory     = "$(Get-NintendoSdkRootPath)/Integrate/Outputs/RcmWriter"
$OutputConfig        = "$OutputDirectory/RecoverySystem-NxDev-WithBootConfig.xml"
$OutputBootConfig    = "$OutputDirectory/BootConfig.img"
$OutputQspiBootImage = "$OutputDirectory/QspiBoot.img"

# Test existing files
Test-EnsureExistPath $SystemUpdater
Test-EnsureExistPath $RunOnTarget
Test-EnsureExistPath $ControlTarget
Test-EnsureExistPath $InitialImagePath
Test-EnsureExistPath $RcmWriterRcm
Test-EnsureExistPath $RecoveryWriter
Test-EnsureExistPath $BootConfig

if (!(Test-Path $OutputDirectory)) {
    New-Item $OutputDirectory -itemType Directory | Out-Null
}

# Copy files

cp $RcmWriterXml $OutputConfig
cp $RecoveryWriter $OutputQspiBootImage
cp $BootConfig $OutputBootConfig

# RcmWriter
Write-Host "Write by Rcm: $RecoveryWriter"
Invoke-CriticalCommand "& `"$RcmWriter`" --rcm `"$RcmWriterRcm`" --list `"$OutputConfig`" $RcmWriterAdditionalOptions"

# Detect target
Write-Host "Find USB target device..."
Get-SigleUsbTargetName -RetryCount $DetectTargetRetryCount  | Out-Null

# Write
Invoke-CriticalCommand "& `"$RunOnTarget`" run --target usb --verbose --failure-timeout 600 --pattern-success-exit `"Succeeded initializing the system.`" `"$SystemUpdater`" '--' --input `"$InitialImagePath`" --enable-erase-emmc $AdditionalOptions"
