﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Kernel Update

    .DESCRIPTION
        Sending a kernel image on serial port, boot it.
#>

param
(
    [parameter(Mandatory=$true)]
    [string]
    # Path to NxAging config
    $ConfigPath
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\..\..\Modules\Error"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Path"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Utility"
Import-Module "${scriptDirectoryPath}\..\..\Modules\Yaml"
Import-Module "${scriptDirectoryPath}\..\..\Modules\NxAging"

$ResultDirectoryPath = "$(Get-NintendoSdkRootPath)\Integrate\Outputs\NxAgingTools"

$NxAgingHelperCommand = "$(Get-NintendoSdkRootPath)\Integrate\Outputs\AnyCPU\Tools\NxAgingTools\NxAgingHelperCommand\Release\NxAgingHelperCommand.exe"
$CommonArgs = "--enable-uart-log --enable-sdev-usb-event --enable-power-button-event"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

$configText = (Get-Content $ConfigPath) -join [System.Environment]::NewLine
$config = (ConvertFrom-Yaml $configText).yaml

# NxAgingHelperCommand の起動をスキップする設定なら、そのまま正常終了
if ($config.RunNxAgingHelper -eq "false")
{
    Write-Host "Skip executing NxAginHelperCommand."
    exit 0
}

# エージングの実行時間を取得
$Duration = $config.Duration

# ターゲットを文字列配列として取得
$TargetList = @() 
foreach($node in $config.Target.ChildNodes)
{
    $TargetList += $node.InnerXml
}
$TargetList = @(Filter-TargetByInitializeLog -targets $TargetList -filter "SUCCESS")

if ($TargetList.Count -eq 0)
{
    throw "No valid target is specified."
}

# ConfigPath の一覧を取得
if ((Split-Path -IsAbsolute $config.DefaultConfig) -or $config.DefaultConfig.StartsWith("\\"))
{
    $DefaultConfigPath = $config.DefaultConfig
}
else
{
    # 絶対パスとネットワークパス以外は、NINTENDO_SDK_ROOT からの相対パスとして扱う
    $DefaultConfigPath = "$(Get-NintendoSdkRootPath)\" + $config.DefaultConfig
}

$TargetSpecificConfig = @{}
if ($config.TargetSpecificConfig -ne $null)
{
    foreach ($node in $config.TargetSpecificConfig.ChildNodes)
    {
        foreach ($targetNode in $node.Target.ChildNodes)
        {
            $target = $targetNode.InnerXml
            if ((Split-Path -IsAbsolute $node.Path) -or $node.Path.StartsWith("\\"))
            {
                $TargetSpecificConfig[$target] = $node.Path
            }
            else
            {
                # 絶対パスとネットワークパス以外は、NINTENDO_SDK_ROOT からの相対パスとして扱う
                $TargetSpecificConfig[$target] = "$(Get-NintendoSdkRootPath)\" + $node.Path
            }
        }
    }
}

# 並列で NxAgingHelper を実行
$jobs = @()
foreach ($target in $TargetList)
{
    # 設定ファイルからターゲットに対応する ConfigPath を取得
    $configPath = $DefaultConfigPath
    if ($null -ne $TargetSpecificConfig[$target])
    {
        $configPath = $TargetSpecificConfig[$target]
    }

    $eventLogPath = "$ResultDirectoryPath\NxAgingTools_${target}_Event.log"
    $uartLogPath = "$ResultDirectoryPath\NxAgingTools_${target}_Uart.log"

    $paramList = @($NxAgingHelperCommand, $CommonArgs, $Duration, $target, $configPath, $eventLogPath, $uartLogPath)
    $jobs += Start-Job -ArgumentList $paramList -ScriptBlock {
        param($NxAgingHelperCommand, $CommonArgs, $Duration, $target, $configPath, $eventLogPath, $uartLogPath)
        Write-Host "`"$NxAgingHelperCommand`" $CommonArgs --time $Duration --sdev $target --config $ConfigPath --event-log-path $eventLogPath --uart-log-path $uartLogPath"
        Invoke-Expression "& `"$NxAgingHelperCommand`" $CommonArgs --time $Duration --sdev $target --config $ConfigPath --event-log-path $eventLogPath --uart-log-path $uartLogPath"
        if ($LastExitCode -ne 0)
        {
            # 例外を ScriptBlock の外で直接 catch はできないが、
            # throw することでジョブの State が Failed になる
            # Wait-Job の後で State をチェックして Failed ならそこで例外を投げる
            throw "Failed to start NxAgingHelperCommand for $target (ExitCode = $LastExitCode)."
        }
    }
}

# スクリプト完了を待つ
foreach ($job in $jobs)
{
    Wait-Job $job.Id
    $result = Receive-Job $job.Id | Write-Host
    if ($job.State -eq "Failed")
    {
        throw "Failed to execute NxAgingHelperCommand."
    }
}

Write-Host "Done."
