﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Merge TestRunner reports

    .DESCRIPTION
        Make a integrated report file from multiple TestRunner reports
#>

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

$templatePath = (Get-NintendoSdkIntegrateRootPath)
$templatePath = [IO.Path]::Combine($templatePath, "Sources")
$templatePath = [IO.Path]::Combine($templatePath, "Tools")
$templatePath = [IO.Path]::Combine($templatePath, "TestRunner")
$templatePath = [IO.Path]::Combine($templatePath, "TestRunner")
$templatePath = [IO.Path]::Combine($templatePath, "Resources")
$templatePath = [IO.Path]::Combine($templatePath, "Report.template.html")

if ( -not (Test-Path $templatePath -PathType Leaf) )
{
    $templateName =[IO.Path]::GetFileName($templatePath)

    $message = "${templateName} not found at '${templatePath}'"

    throw New-Object "System.IO.FileNotFoundException" $message
}

$outputPath = (Get-NintendoSdkIntegrateOutputRootPath)
$outputPath = [IO.Path]::Combine($outputPath, "TestRunner")

if ( -not (Test-Path $outputPath -PathType Container) )
{
    New-Item $outputPath -ItemType Directory | Out-Null
}

$outputUri = New-Object "Uri" "$outputPath\"

function Get-EscapedPath([string]$path)
{
    return $path.Replace("\", "\\")
}

function Get-Header([string]$summaryPath)
{
    $header = New-Object "Collections.Generic.Dictionary[string, object]"

    Get-Content $summaryPath | % {
        switch -regex -casesensitive ($_)
        {
            "^Test date\s+: (.+)"       { $header["date"]       = $matches[1] }
            "^Test root\s+: (.+)"       { $header["path"]       = $matches[1] }
            "^Platforms?\s+: (.+)"      { $header["platforms"]  = $matches[1] }
            "^BuildTypes?\s+: (.+)"     { $header["builds"]     = $matches[1] }
            "^Modules?\s+: (.+)"        { $header["modules"]    = $matches[1] }
            "^Categor(y|ies)\s+: (.+)"  { $header["categories"] = $matches[2] }
            "^Branch name\s+: (.+)"     { $header["branch"]     = $matches[1] }
            "^Commit hash\s+: (.+)"     { $header["commit"]     = $matches[1] }
        }
    }

    $header["path"] = Get-EscapedPath $header["path"]

    ("branch", "commit") | % {
        $key = $_

        if ( -not $header.ContainsKey($key) )
        {
            $header[$key] = [string]::Empty
        }
    }

    ("platforms", "builds", "modules", "categories") | % {
        $key = $_

        $list = New-Object "Collections.Generic.List[string]"

        $value = [string]::Empty

        if ( $header.TryGetValue($key, [ref]$value) )
        {
            $value -split ", " | % { $list.Add($_) }
        }

        $header[$key] = $list
    }

    return ,$header
}

function Test-ResultCodeSuccess([string]$resultCode)
{
    return $resultCode -ceq "PASS"
}

function Test-ResultCodeSerious([string]$resultCode)
{
    if ( Test-ResultCodeSuccess $resultCode )
    {
        return $false
    }

    if ( ("FAIL", "SKIP", "NO_ENTRIES") -contains $resultCode )
    {
        return $false
    }

    return $true
}

function Test-ResultCodeSkip([string]$resultCode)
{
    if ( Test-ResultCodeSuccess $resultCode )
    {
        return $false
    }

    if ( ("FAIL", "TIME_OUT", "ERROR") -contains $resultCode )
    {
        return $false
    }

    return $true
}

function Get-ListName([xml]$listContext)
{
    $builder = New-Object "Text.StringBuilder"

    $builder.Append(
        (Get-EscapedPath $listContext.ListContext.ListName)) | Out-Null

    if ( 1 -lt $listContext.ListContext.StatementCount )
    {
        $builder.AppendFormat(
            " [{0:D3}]", $listContext.ListContext.StatementId) | Out-Null
    }

    if ( Test-ResultCodeSuccess $listContext.ListContext.ResultCode )
    {
        $list = New-Object "Collections.Generic.List[string]"

        if ( 1 -lt $listContext.ListContext.PlatformCount )
        {
            $list.Add($listContext.ListContext.PlatformName)
        }

        if ( 1 -lt $listContext.ListContext.BuildTypeCount )
        {
            $list.Add($listContext.ListContext.BuildTypeName)
        }

        if ( 0 -lt $list.Count )
        {
            $builder.AppendFormat(" ({0})", ($list -join ", ")) | Out-Null
        }
    }

    return ,$builder.ToString()
}

function Get-ListResult([xml]$listContext)
{
    $builder = New-Object "Text.StringBuilder"

    $builder.AppendLine("{") | Out-Null

    $type = if ( Test-ResultCodeSerious $listContext.ListContext.ResultCode )
    {
        "ERROR"
    }
    else
    {
        $listContext.ListContext.ResultCode
    }

    $builder.AppendLine("type: `"${type}`",") | Out-Null

    $builder.AppendLine("number: `"000`",") | Out-Null

    $builder.AppendLine([string]::Format(
        "result: `"{0}`",", $listContext.ListContext.ResultCode)) | Out-Null

    $builder.AppendLine("log: `"`",") | Out-Null

    $xmlUri = New-Object "Uri" $listContext.ListContext.Path

    $builder.AppendLine([string]::Format(
        "xml: `"{0}`",", $outputUri.MakeRelativeUri($xmlUri))) | Out-Null

    $builder.AppendLine("duration: `"0`",") | Out-Null

    $builder.AppendLine([string]::Format(
        "name: `"{0}`"",
        [IO.Path]::GetFileName($listContext.ListContext.ListName))) | Out-Null

    $builder.AppendLine("}") | Out-Null

    return ,$builder.ToString()
}

function Get-TestResult([xml]$testContext)
{
    $builder = New-Object "Text.StringBuilder"

    $builder.AppendLine("{") | Out-Null

    $type = if ( Test-ResultCodeSerious $testContext.TestContext.ResultCode )
    {
        "ERROR"
    }
    else
    {
        $testContext.TestContext.ResultCode
    }

    $builder.AppendLine("type: `"${type}`",") | Out-Null

    $number = if ( $testContext.TestContext.RoleType -cne "Observer" )
    {
        ([Uint32]$testContext.TestContext.TestId).ToString("D3")
    }
    else
    {
        "OBS"
    }

    $builder.AppendLine("number: `"${number}`",") | Out-Null

    $builder.AppendLine([string]::Format(
        "result: `"{0}`",", $testContext.TestContext.ResultCode)) | Out-Null

    if ( Test-ResultCodeSkip $testContext.TestContext.ResultCode )
    {
        $builder.AppendLine("log: `"`",") | Out-Null
    }
    else
    {
        $logUri = New-Object "Uri" $testContext.TestContext.LogPath

        $builder.AppendLine([string]::Format(
            "log: `"{0}`",", $outputUri.MakeRelativeUri($logUri))) | Out-Null
    }

    $xmlUri = New-Object "Uri" $testContext.TestContext.Path

    $builder.AppendLine([string]::Format(
        "xml: `"{0}`",", $outputUri.MakeRelativeUri($xmlUri))) | Out-Null

    $builder.AppendLine([string]::Format(
        "duration: `"{0}`",", $testContext.TestContext.Duration)) | Out-Null

    $builder.AppendLine([string]::Format(
        "name: `"{0}`"", $testContext.TestContext.TestName)) | Out-Null

    $builder.AppendLine("}") | Out-Null

    return ,$builder.ToString()
}

function Get-FullResult($tests)
{
    $builder = New-Object "Text.StringBuilder"

    $builder.AppendLine("{") | Out-Null

    $builder.AppendLine(
        [string]::Format("path: `"{0}`",", $tests["path"])) | Out-Null

    $builder.AppendLine("tests: [") | Out-Null

    $builder.Append($tests["tests"] -join ",") | Out-Null

    $builder.AppendLine("]") | Out-Null

    $builder.AppendLine("}") | Out-Null

    return ,$builder.ToString()
}

function Get-JsonReport($report)
{
    $builder = New-Object "Text.StringBuilder"

    $builder.AppendLine("{") | Out-Null

    $builder.AppendLine(
        [string]::Format("name: `"{0}`",", $report["name"])) | Out-Null

    $builder.AppendLine(
        [string]::Format("date: `"{0}`",", $report["date"])) | Out-Null

    $builder.AppendLine(
        [string]::Format("path: `"{0}`",", $report["path"])) | Out-Null

    $builder.Append("platforms: [ ") | Out-Null

    $platforms = $report["platforms"] | % { "`"${_}`"" }

    $builder.Append($platforms -join ", ") | Out-Null

    $builder.AppendLine(" ],") | Out-Null

    $builder.Append("builds: [ ") | Out-Null

    $builds = $report["builds"] | % { "`"${_}`"" }

    $builder.Append($builds -join ", ") | Out-Null

    $builder.AppendLine(" ],") | Out-Null

    $builder.Append("modules: [ ") | Out-Null

    $modules = $report["modules"] | % { "`"${_}`"" }

    $builder.Append($modules -join ", ") | Out-Null

    $builder.AppendLine(" ],") | Out-Null

    $builder.Append("categories: [ ") | Out-Null

    $categories = $report["categories"] | % { "`"${_}`"" }

    $builder.Append($categories -join ", ") | Out-Null

    $builder.AppendLine(" ],") | Out-Null

    $builder.AppendLine(
        [string]::Format("branch: `"{0}`",", $report["branch"])) | Out-Null

    $builder.AppendLine(
        [string]::Format("commit: `"{0}`",", $report["commit"])) | Out-Null

    $builder.AppendLine("results: [") | Out-Null

    $builder.Append($report["results"] -join ",") | Out-Null

    $builder.AppendLine("]") | Out-Null

    $builder.AppendLine("}") | Out-Null

    return ,$builder.ToString()
}

$reports = New-Object "Collections.Generic.List[string]"

Get-ChildItem $outputPath -Include Summary.txt -Recurse | % {
    $summaryPath = $_.FullName

    $report = New-Object "Collections.Generic.Dictionary[string, object]"

    $reportPath = [IO.Path]::GetDirectoryName($summaryPath)
    $report["name"] = [IO.Path]::GetFileName($reportPath)

    $header = Get-Header $summaryPath
    $header.Keys | % { $report[$_] = $header[$_] }

    $report["results"] = New-Object "Collections.Generic.List[string]"

    $tests = $null

    Get-ChildItem $reportPath -Include *.result.xml -Recurse | % {
        $contextPath = $_.FullName

        $context = [xml](Get-Content $contextPath)

        if ($contextPath.EndsWith(".list.result.xml"))
        {
            if ($tests -ne $null)
            {
                $report["results"].Add((Get-FullResult $tests))
            }

            $tests = New-Object "Collections.Generic.Dictionary[string, object]"
            $tests["path"] = Get-ListName $context
            $tests["tests"] = New-Object "Collections.Generic.List[string]"

            if ( -not (Test-ResultCodeSuccess $context.ListContext.ResultCode) )
            {
                $tests["tests"].Add((Get-ListResult $context))
            }
        }
        else
        {
            $tests["tests"].Add((Get-TestResult $context))
        }
    }

    if ($tests -ne $null)
    {
        $report["results"].Add((Get-FullResult $tests))
    }

    $reports.Add((Get-JsonReport $report))
}

if ( 0 -eq $reports.Count )
{
    $Host.UI.WriteWarningLine("TestRunner report was not found")
}
else
{
    $reports = $reports -join ","

    $htmlPath = [IO.Path]::Combine($outputPath, "Report.html")

    $html = Get-Content $templatePath
    $html = $html | % { $_.Replace("/*DUMMY_REPORTS*/", $reports) }
    $html | Out-File $htmlPath -Encoding UTF8
}
