﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Kernel Update

    .DESCRIPTION
        Sending a kernel image on serial port, boot it.
#>

param
(
    [string]
    # The platform name
    $Platform = "NXFP2-a64",

    [string]
    # Debug/Develop/Release
    $BuildType = "Develop",

    [string]
    # Auto / K1 / K2 / ...
    $KeyType = "Auto",

    [string]
    # Nand / List (FromHost)
    $BootType = "Nand",

    [string]
    # Hb (HostBridge) / Usb
    $ConnectionType = "Hb",

    [string]
    # Signed / Unsigned kernel
    $SignedType = "Unsigned",

    [string]
    # 32G / 64G
    $StorageSize = "64G",

    [string]
    # Public / Internal
    $DeveloperType = "Internal",

    [switch]
    $SkipHelloWorld = $false,

    [string]
    $AddressPattern = $env:TARGET_ADDRESS_PATTERN
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\BootConfig"
Import-Module "${scriptDirectoryPath}\Modules\Target"
Import-Module "${scriptDirectoryPath}\Modules\Runner"

$RestartHostBridge  = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Recovery/Restart-HostBridge.ps1"

if (-not $AddressPattern)
{
    # If AddressPattern is not specified and env:TARGET_ADDRESS_PATTERN is not difined
    $AddressPattern = "169.*";
}

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

function Invoke-TegraFlash()
{
    $FlashMinRoot = "$(Get-NintendoSdkRootPath)\ToolChains\flash-min-fp2"
    $mkkernelimgPath = "$FlashMinRoot\bin\mkkernelimg"
    $controlTargetPath = "D:\Tools\ControlTarget\ControlTarget.exe"

    $kernelDir = "$(Get-NintendoSdkRootPath)\Programs/Chris/Outputs/${Platform}-kernel/Kernel/${BuildType}/"

    Start-Process $mkkernelimgPath -ArgumentList "--kernel $kernelDir/CODE_ENTRY --ramdisk $kernelDir/INITIAL_PROCESS -o ./kernel.img" -NoNewWindow -Wait

    Start-Process $controlTargetPath -ArgumentList "force-recovery" -NoNewWindow -Wait

    Start-Sleep 3

    $pythonDir  = "$(Get-NintendoSdkRootPath)\Externals\Binaries\Python27"
    $pythonPath = "${pythonDir}\python.exe"

    $WriteKernelArgument  = "$FlashMinRoot/bin/tegraflash.py "
    $WriteKernelArgument += "--applet $FlashMinRoot/nvtboot_recovery.bin "
    $WriteKernelArgument += "--bct $FlashMinRoot/bct_p2530_e01.cfg "
    $WriteKernelArgument += "--bl $FlashMinRoot/cboot_default.bin "
    $WriteKernelArgument += "--cfg $FlashMinRoot/flash_t210_android_sdmmc_fb.xml "
    $WriteKernelArgument += "--chip 0x21 "
    $WriteKernelArgument += "--dtb $FlashMinRoot/tegra210-jetson-e-p2595-0000-a00-00.dtb "
    $WriteKernelArgument += "--nct $FlashMinRoot/nct.bin "
    $WriteKernelArgument += "--odmdata 0x94000 "
    $WriteKernelArgument += "--cmd `"write LNX kernel.img; reboot`""

    Start-Process $pythonPath -ArgumentList "$WriteKernelArgument" -NoNewWindow -Wait
}

function Invoke-SystemUpdate()
{
    param
    (
        [string]
        $Platform
    )

    $ControlTarget = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTarget.exe"
    $ControlTargetPrivate = "$(Get-NintendoSdkRootPath)/Tools/CommandLineTools/ControlTargetPrivate.exe"

    # ターゲットをリストアップ
    $TargetList = Get-Targets -ConnectionType $ConnectionType

    # 署名済みカーネルを使うかどうか
    if (($SignedType -eq "Signed") -or ($DeveloperType -eq "Public"))
    {
        $TargetList | Where-Object {

            $TargetIp = $_

            # TORIAEZU: CI で失敗する場合があるので 5 回までリトライする
            for ($i=0; $i -lt 5; $i++)
            {
                try
                {
                    # TargetManager が接続中だと takeover できないので切断しておく
                    Invoke-Expression "& `"$ControlTarget`" disconnect"

                    # 署名済みカーネルを使う場合、署名検証を有効にするために Dummy BootConfig を書く
                    Write-Host "Write Dummy BootConfig"
                    Write-DummyBootConfig $TargetIp
                    Write-Host "Done."
                    break;
                }
                catch [Exception]
                {
                    $num = $i + 1
                    Write-Host "Failed to write dummy BootConfig. Retry... ($num/5)"

                    # TORIAEZU: ワークアラウンドのため、失敗した場合には HostBridge ごと再起動する
                    Invoke-Expression "& `"$RestartHostBridge`" -AddressPattern $TargetIp -WaitBoot"
                    if($LastExitCode -ne 0)
                    {
                        # 再起動に失敗した場合はエラー終了
                        Write-ErrorMessage "Command failed.`n    Command=$Expression`n    ExitCode = $LastExitCode"
                        exit 1
                    }
                }
            }
        }
    }

    #
    # パラメータに応じて初期化コマンドを生成
    #
    $AdditionalOptions = "-SkipStopProcessesForCi"
    
    if ($ConnectionType -eq "Usb")
    {
        $InitializeScript = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Initialize/Initialize-Edev-Full-Raw.ps1"
        
        # HostBridge がないと KeyType が取れないため、K5 に固定
        $KeyType = "K5"

        try
        {
            Invoke-CriticalCommand "& `"$ControlTargetPrivate`" power-off-phidgets"
            Invoke-CriticalCommand "& `"$ControlTargetPrivate`" force-recovery"
        }
        catch [Exception]
        {
            exit 1
        }
    }
    else
    {
        $InitializeScript = "$(Get-NintendoSdkRootPath)/Integrate/Scripts/NX/Initialize/Initialize-Sdev-Full-Raw"    
        if ($DeveloperType -eq "Public")
        {
            $SignedType = "Signed"
        }

        if ($SkipHelloWorld)
        {
            $AdditionalOptions += " -SkipHelloWorld"
        }
    }

    # 初期化用に Target Manager を再起動
    Stop-ProcessesForSystemInitialize -SkipStopTargetManager
    Unregister-AllTargets
    Invoke-Expression "& `"${scriptDirectoryPath}\Start-TargetManager.ps1`" -ForInitialize"

    #
    # 初期化
    #

    # 並列で初期化スクリプトを実行
    $jobs = @()
    $TargetList | Where-Object {
        $TargetIp = $_

        $paramList = @($InitializeScript, $TargetIp, $BuildType, $Platform, $BootType, $SignedType, $StorageSize, $DeveloperType, $AdditionalOptions)
        $jobs += Start-Job -ArgumentList $paramList -ScriptBlock {
            param($InitializeScript, $TargetIp, $BuildType, $Platform, $BootType, $SignedType, $StorageSize, $DeveloperType, $AdditionalOptions)
            Invoke-Expression "& `"$InitializeScript`" -AddressPattern $TargetIp -BuildType $BuildType -Platform $Platform -BootType $BootType -SignedType $SignedType -StorageSize $StorageSize -DeveloperType $DeveloperType $AdditionalOptions"
            if ($LastExitCode -ne 0)
            {
                # 例外を ScriptBlock の外で直接 catch はできないが、
                # throw することでジョブの State が Failed になる
                # Wait-Job の後で State をチェックして Failed ならそこで例外を投げる
                throw "Command failed.`n& `"$InitializeScript`" -AddressPattern $TargetIp -BuildType $BuildType -Platform $Platform -BootType $BootType -SignedType $SignedType -StorageSize $StorageSize -DeveloperType $DeveloperType $AdditionalOptions`nExitCode = $LastExitCode"
            }
        }
    }

    # スクリプト完了を待つ
    foreach ($job in $jobs)
    {
        Wait-Job $job.Id
        $result = Receive-Job $job.Id | Write-Host
        if ($job.State -eq "Failed")
        {
            throw "Failed to update the system."
        }
    }
}


# Update Kernel
Write-Host "== Update kernel image ==="

switch -Regex ($Platform)
{
    "jetson-tk2"
    {
        Invoke-TegraFlash $Platform
    }
    "jetson-tk2-a64"
    {
        Invoke-TegraFlash $Platform
    }
    "NXFP2-a32|NX-NXFP2-a32"
    {
        Invoke-SystemUpdate "NX-NXFP2-a32"
    }
    "NXFP2-a64|NX-NXFP2-a64"
    {
        Invoke-SystemUpdate "NX-NXFP2-a64"
    }
}

Write-Host "Done."

exit 0
