﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Sleep Test on Target

    .DESCRIPTION
        Run Sleep and Wake Test.
#>

param
(
    [string]
    # Target Address Pattern
    $AddressPattern = $env:TARGET_ADDRESS_PATTERN,

    [string]
    # The platform name
    $Platform = "NX64",

    [int]
    # The number of seconds before it times out.
    $Timeout = 90,

    [string]
    # The build type
    $BuildType = "Develop",

    [string]
    # Test path
    $Path,

    [string]
    # DevMenu path
    $DevMenuPath,

    [bool]
    $RunDevMenu = $true,

    [int]
    $TimeOfFirstPowerButtonPush = 5,

    [int]
    $TimeOfSecondPowerButtonPush = 20,

    [int]
    $TimeOfSleepBeforeTest = 10,

    [int]
    $TimeOfSleepAfterTest = 0,

    [string]
    $TargetName,

    [string]
    $TargetInterface = "Ethernet",

    [string]
    $TargetAddress,

    [string]
    $SleepTriggerType = "PowerButton"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\HostBridge"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\Utility"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($AddressPattern))
{
    # If AddressPattern is not specified and env:TARGET_ADDRESS_PATTERN is not
    # difined
    $AddressPattern = "^169\.*"
}

$VsPlatform = switch ($Platform)
{
    "NX32"      { "NX-NXFP2-a32" }
    "NXFP2-a32" { "NX-NXFP2-a32" }
    "NX64"      { "NX-NXFP2-a64" }
    "NXFP2-a64" { "NX-NXFP2-a64" }
    default     { $_ }
}

if ([string]::IsNullOrEmpty($Path))
{
    $Path = Get-NintendoSdkRootPath
    $Path = [IO.Path]::Combine($Path, "Tests\Outputs")
    $Path = [IO.Path]::Combine($Path, $VsPlatform)
    $Path = [IO.Path]::Combine($Path, "Tests\SleepWakePerformanceChecker")
    $Path = [IO.Path]::Combine($Path, $BuildType)
    $Path = [IO.Path]::Combine($Path, "SleepWakePerformanceChecker.nsp")
}

if ([string]::IsNullOrEmpty($DevMenuPath))
{
    $DevMenuPath = Get-NintendoSdkRootPath
    $DevMenuPath = [IO.Path]::Combine($DevMenuPath, "Programs\Iris\Outputs")
    $DevMenuPath = [IO.Path]::Combine($DevMenuPath, "NX-NXFP2-a64")
    $DevMenuPath = [IO.Path]::Combine($DevMenuPath, "TargetTools\DevMenu")
    $DevMenuPath = [IO.Path]::Combine($DevMenuPath, $BuildType)
    $DevMenuPath = [IO.Path]::Combine($DevMenuPath, "DevMenu.nsp")
}

if ([string]::IsNullOrEmpty($TargetName))
{
    if ("Ethernet" -cne $TargetInterface)
    {
        throw "TargetName not specified"
    }
    else
    {
        $TargetName = Get-SigleTargetAddress -AddressPattern $AddressPattern

        $TargetAddress = $TargetName
    }
}

if ([string]::IsNullOrEmpty($TargetAddress))
{
    if ("Ethernet" -ceq $TargetInterface)
    {
        throw "TargetAddress not specified"
    }
}

switch ( $SleepTriggerType )
{
    "PowerButton" { break }
    "LowBattery"  { break }
    default       { throw "Invalid sleep trigger type. Specify `"PowerButton`" or `"LowBattery`"." }
}

# TORIAEZU :
#  dll が読めないので、暫定対応として、Outputs 以下の ControlTarget を実行
$ControlTarget = Get-NintendoSdkRootPath
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Programs\Chris\Outputs")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "x86")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Tools\RunnerTools")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "Release")
$ControlTarget = [IO.Path]::Combine($ControlTarget, "ControlTargetPrivate.exe")

$RunOnTarget = Get-NintendoSdkRootPath
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "Tools")
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "CommandLineTools")
$RunOnTarget = [IO.Path]::Combine($RunOnTarget, "RunOnTarget.exe")

$ReadHbLog = Get-NintendoSdkRootPath
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "Tools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "CommandLineTools")
$ReadHbLog = [IO.Path]::Combine($ReadHbLog, "ReadHostBridgeLog.exe")

$logDir = $scriptDirectoryPath
$logDir = $logDir.Replace(
    (Get-NintendoSdkIntegrateRootPath),
    (Get-NintendoSdkIntegrateOutputRootPath))
$logDir = [IO.Path]::Combine($logDir, [IO.Path]::GetFileNameWithoutExtension($scriptPath))
$logDir = [IO.Path]::Combine($logDir, ((Get-Date -Format "yyyyMMddHHmmss") + "-" + $TargetName))
[void](New-Item $logDir -ItemType Directory -ErrorAction "Stop")

$logFile = [IO.Path]::GetFileNameWithoutExtension($Path)
$logFile = [IO.Path]::Combine($logDir, ($logFile + ".log"))

Write-Host "Start ${Path}"
Write-Host "logFile: ${logFile}"

# FromHost 起動など、spsm が psc のモジュール登録を待たない場合は、
# すぐにテストを走らせるとモジュールが死ぬため待つ
Start-Sleep -Seconds $TimeOfSleepBeforeTest

# SA としてテストを起動 (してすぐ RunOnTarget は終了)
$command  = ("&", "`"$RunOnTarget`"", "`"$Path`"")
$command += ("--target", $TargetName)
$command += ("--verbose", "--no-wait", "--failure-timeout", "$Timeout")
$command += ("--suppress-auto-kill")
$command = $command -join " "
Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host
if (0 -ne $LastExitCode)
{
    Invoke-Expression "& `"$ControlTarget`" reset --target $TargetName"

    throw ("Failed to run " + [IO.Path]::GetFileName($Path))
}

$TestResult = 1
$BackgroundJobResult = 1

if ("Ethernet" -ieq $TargetInterface)
{
    # スリープに入る
    $list = ($ControlTarget, $TargetName, $TargetAddress,
                $TimeOfFirstPowerButtonPush, $TimeOfSecondPowerButtonPush, $SleepTriggerType )
    $job = Start-Job -ArgumentList $list -ScriptBlock {
        param($path, $name, $addr, $time1, $time2, $trigger)
        if ( $trigger -eq "PowerButton")
        {
            Start-Sleep -Seconds $time1
            $subCommand  = ("&", "`"$path`"", "press-power-button")
            $subCommand += ("--target", $name)
            $subCommand = $subCommand -join " "
            Invoke-Expression $subCommand
        }
        elseif ( $trigger -eq "LowBattery" )
        {
            Start-Sleep -Seconds $time1
            $subCommand  = ("&", "`"$path`"", "set-battery-level-for-sc7", "sleep-required")
            $subCommand += ("--ip-addr", $addr)
            $subCommand += ("--target", $name)
            $subCommand = $subCommand -join " "
            Invoke-Expression $subCommand
        }

        if( $trigger -eq "LowBattery" )
        {
            Start-Sleep -Seconds $time1
            $subCommand  = ("&", "`"$path`"", "set-battery-level-for-sc7", "enough")
            $subCommand += ("--target", $name)
            $subCommand = $subCommand -join " "
            Invoke-Expression $subCommand
        }

        Start-Sleep -Seconds $time2
        $subCommand  = ("&", "`"$path`"")
        $subCommand += ("press-power-button-for-sc7")
        $subCommand += ("--ip-addr", $addr)
        $subCommand = $subCommand -join " "
        Invoke-Expression $subCommand
    }

    # HostBridge Log の監視
    $command  = ("&", "`"$ReadHbLog`"")
    $command += ("--address", $TargetAddress)
    $command += ("--verbose", "--failure-timeout", "$Timeout")
    $command += ("--pattern-success-exit", "`"\[  PASSED  \] .+ tests?.`"")
    $command += ("--pattern-failure-exit", "`"\[  FAILED  \]`"")
    $command += ("--pattern-failure-exit", "`"\[TestOnTarget\] FAILED`"")
    $command += ("--pattern-failure-exit", "`"Assertion Failure:`"")
    $command += ("--pattern-failure-exit", "`"Break\(\) called`"")
    $command = $command -join " "
    Invoke-Expression $command | Set-Content -Passthru $logFile | Write-Host

    $ReadHostBridgeLogResult = $LastExitCode

    # Background Job の完了待ちと結果のチェック
    Wait-Job -id $job.Id
    if($job.State -eq "Completed")
    {
        $BackgroundJobResult = 0
    }

    Write-Output "ReadHostBridgeLogResult = $ReadHostBridgeLogResult"
    Write-Output "BackgroundJobResult     = $BackgroundJobResult"

    $TestResult = $ReadHostBridgeLogResult + $BackgroundJobResult
}
elseif ("USB" -ieq $TargetInterface)
{
    # USB 接続の場合は Phidgets を使う
    # power-on-phidgets コマンドで Power ボタンをちょん押しするので、スリープ・ウェイクできる
    $command  = ("&", "`"$ControlTarget`"", "power-on-phidgets")
    $command = $command -join " "

    # スリープに入る
    Start-Sleep -Seconds $TimeOfFirstPowerButtonPush
    Invoke-Expression $command
    $TestResult = $LastExitCode

    # スリープ復帰
    if (0 -eq $TestResult)
    {
        Start-Sleep -Seconds $TimeOfSecondPowerButtonPush
        Invoke-Expression $command
        $TestResult = $LastExitCode
    }

    Start-Sleep -Seconds 5

    # USB 接続の場合は、テスト終了の待ち時間として ControlTarget と同様に 20 秒待つようにする。
    $TimeOfSleepAfterTest = 20
}
else
{
    Write-Warning "$TargetInterface is not supported"
}

# USB 接続の場合は、今のところテストの終了を待っていないので、すぐ DevMenu を起動するとテストが途中で終了してしまう
# ワークアラウンドとして、適当にウェイトを入れる
Start-Sleep -Seconds $TimeOfSleepAfterTest

if ((0 -ne $TestResult) -or ("true" -eq $env:REQUIRE_TARGET_RESET))
{
    # スリープ状態で失敗した場合、
    # TargetManager を落とさないとリセットできないため、一度落とす
    Invoke-Expression "& `"${scriptDirectoryPath}\Stop-TargetManager.ps1`""

    Invoke-CriticalCommand "& `"$ControlTarget`" reset --target $TargetName"
}

# SA がいなくなった後なので DevMenu を起動しておく
if ($RunDevMenu)
{
    $DevMenuCommandSystem = "$(Get-NintendoSdkRootPath)\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommandSystem\$BuildType\DevMenuCommandSystem.nsp"
    Invoke-Expression "& `"$RunOnTarget`" --target $TargetName $DevMenuCommandSystem systemprogram launch 0x0100000000002064"
}

# Pass exitCode to TestRunner's result
Write-Output "TestResult              = $TestResult"
exit $TestResult
