﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Invoke Kernel on U-Boot

    .DESCRIPTION
        Sending a kernel image on serial port, boot it.
#>

param
(
    [string]
    # The path to PwrUSBDll.dll
    $PowerUsbDllFile,

    [int]
    # The number of seconds before it times out.
    $Timeout = 900,

    [string]
    # The build type
    $BuildType = "Develop"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\Modules\Error"
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\PowerUsb"

trap [Exception]
{
    Write-ErrorRecord $_
    exit 1
}

if ([string]::IsNullOrEmpty($PowerUsbDllFile))
{
    throw "'PowerUsbDllFile' is required"
}
Import-PowerUsbDll $PowerUsbDllFile

$ttpmacro = Get-TTPMacroPath
$ttlFile = "$(Get-TeraTermProMacrosRootPath)\InvokeKernelOnUBoot.ttl"
$iniFile = "$(Get-TeraTermProConfigurationsRootPath)\bdsl-imx6.ini"

$chrisDir  = "$(Get-NintendoSdkRootPath)\Programs\Chris"
$kernelDir = "${chrisDir}\Outputs\bdsl-imx6-kernel\Kernel"
$buildTypeDir = "${kernelDir}\${BuildType}"
$procFile = "${buildTypeDir}\INITIAL_PROCESS"
$codeFile = "${buildTypeDir}\CODE_ENTRY"
if ((-not (Test-Path $procFile)) -or  (-not (Test-Path $codeFile)))
{
    Write-ErrorMessage "Kernel images not found"
    exit 1
}

$ports = New-Object "System.Collections.Generic.List[String]"
foreach ($port in [System.IO.Ports.SerialPort]::GetPortNames())
{
    $ports.Add(($port -creplace "^COM"))
}
if (0 -eq $ports.Count)
{
    Write-ErrorMessage "Serial port not found"
    exit 1
}
$ports.Sort()

$logDir = [System.IO.Path]::GetFileNameWithoutExtension($scriptPath)
$logDir = "${scriptDirectoryPath}\${logDir}"
$logDir = $logDir.Replace((Get-NintendoSdkIntegrateRootPath), "")
$logDir = (Get-NintendoSdkIntegrateOutputRootPath) + $logDir
$logDir = $logDir + "\" + (Get-Date -Format "yyyyMMddHHmmss")
[void](New-Item $logDir -ItemType Directory –ErrorAction "Stop")
$logFile = $logDir + "\" + $ports[0] + ".log"

$params = $ttlFile, $logFile, $ports[0], $iniFile, $procFile, $codeFile, $BuildType
$procInfo = New-Object "System.Diagnostics.ProcessStartInfo"
$procInfo.UseShellExecute = $false
$procInfo.FileName = $ttpmacro
$procInfo.Arguments = "/V " + (($params | % { "`"$_`"" }) -join " ")

try
{
    Initialize-PowerUsb
    Set-PowerUsbPortState 1 $false
    Start-Sleep 1
    Set-PowerUsbPortState 1 $true
    $proc = New-Object "System.Diagnostics.Process"
    $proc.StartInfo = $procInfo
    [void]$proc.Start()
    $fs = $null
    $sr = $null
    for ($i = 0; ($i -lt $Timeout) -and (-not $proc.HasExited); ++$i)
    {
        Start-Sleep 1
        if ($null -ne $sr)
        {
            Write-Host ($sr.ReadToEnd()) -NoNewLine
        }
        else
        {
            if ((Test-Path $logFile))
            {
                $mode = [System.IO.FileMode]::Open
                $access = [System.IO.FileAccess]::Read
                $share = [System.IO.FileShare]::ReadWrite
                $fs = [System.IO.File]::Open($logFile, $mode, $access, $share)
                $sr = New-Object "System.IO.StreamReader" $fs
            }
        }
    }
    $exitCode = $proc.ExitCode
    if (-not $proc.HasExited)
    {
        taskkill /PID $proc.Id /T /F
        $exitCode = 2
    }
    $proc.WaitForExit()
    if (0 -ne $exitCode) { Write-ErrorMessage "Failed to boot" }
    exit $exitCode
}
finally
{
    if ($null -ne $sr) { $sr.Close() }
    if ($null -ne $fs) { $fs.Close() }
    if ($null -ne $proc) { $proc.Close() }
    Set-PowerUsbPortState 1 $false
    Set-PowerUsbPortState 2 $false
    Set-PowerUsbPortState 3 $false
    Close-PowerUsb
}
