<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Crate Process List Module

    .DESCRIPTION
        This file defines functions for creating process list
#>

#
#    COMMAND [-InvokeType <RunTests|Build|BuildSamples|Default>] [true|false]
#
#       -InvokeType <type>          Oasis の必要性を判別するための起動文脈
#

param
(
    # Check if Visual Studio is running
    [bool]$CheckVisualStudioRunning = $true,
    [ValidateSet("RunTests", "Build", "BuildSamples", "Default")]$InvokeType = "Default"
)

#
# Import submodules
#
$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
Import-Module "${scriptDirectoryPath}\Modules\Path"
Import-Module "${scriptDirectoryPath}\Modules\VisualStudio"


#
# Functions
#

function Get-InstalledOasisVersion()
{
    foreach ( $software in Get-ChildItem("HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall") )
    {
        $name = $software.GetValue("DisplayName")
        if ( $name -ne $null -and $name.startsWith("Nintendo Target Manager and VSI-NX") )
        {
            return $software.GetValue("DisplayVersion")
        }
    }
}

function Get-RepositoryOasisVersion()
{
    # ファイル名 "TargetManager.X.X.X.XXXXX.msi" からバージョン番号 X.X.X.XXXXX を抜き出す
    $filename = [System.IO.Path]::GetFileName($(Get-OasisInstallerPath))
    $version = $filename.Substring($filename.IndexOf(".") + 1, $filename.LastIndexOf(".") - $filename.IndexOf(".") - 1)

    return $version.Split("_")[0] # hotfix バージョンを抜く
}

#
# Entry point
#

# check oasis dependency by InvokeType
switch($InvokeType)
{
    "RunTests"
    {
        Write-Host "Installation is skipped."
        exit 0
    }
    "Build"
    {
        Write-Host "Installation is skipped."
        exit 0
    }
    "BuildSamples"
    {
    }
    "Default"
    {
    }
    default
    {
        Write-Host "Error: unknown InvokeType: ${InvokeType}"
        exit 1
    }
}

# do install
$installedVersion = Get-InstalledOasisVersion
$repositoryVersion = Get-RepositoryOasisVersion
if ($installedVersion -eq $null -or $installedVersion -ne $repositoryVersion)
{
    if ($CheckVisualStudioRunning -and $(Test-VisualStudioRunning))
    {
        Write-Host "Error: Visual Studio is running"
        exit 1
    }

    if ($installedVersion -ne $null)
    {
        Write-Host "'${installedVersion}' is installed."
    }
    Write-Host "Installing ${repositoryVersion}..."

    # [INSTALL] TargetManager.XXX.msi (インストール前に、古いバージョンをアンインストールします)
    $expression = "cmd /c `"$(Get-NintendoSdkRootPath)\Integrate\Packages\InstallScript\TargetManager\Install.bat`" `"$(Get-OasisInstallerPath)`""
    $output = Invoke-Expression $expression
    if ($LASTEXITCODE -ne 0)
    {
        Write-Host "Error: '${expression}' failed ($LASTEXITCODE)."
        exit ${LASTEXITCODE}
    }

    # [INSTALL] VsiForNX.XXX.[msi|vsix] (インストール前に、古いバージョンをアンインストールを行います)
    $expression = "cmd /c `"$(Get-NintendoSdkRootPath)\Integrate\Packages\InstallScript\VsiForNX\Install.bat`" `"$(Get-NintendoSdkRootPath)\Externals\Oasis\Installer`""
    $output = Invoke-Expression $expression
    if ($LASTEXITCODE -ne 0)
    {
        Write-Host "Error: '${expression}' failed ($LASTEXITCODE)."
        exit ${LASTEXITCODE}
    }
}
else
{
    Write-Host "Install is not necessary. '${installedVersion}' is already installed."
}

Write-Host "Done."