﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Disassemble object file.

    .DESCRIPTION
        Disassemble all LOAD sections of specified file.
#>
param
(
    [parameter(Mandatory=$True)][string] $InputFile,
    [parameter(Mandatory=$True)][string] $OutputFile,
    [string] $TargetSectionAttribute = "CODE",
    [string] $HeaderOptions,
    [switch] $EnableSymbolReplace,
    [switch] $DisableFullContentsDump
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

Import-Module "${ScriptDirectoryPath}\Modules\Error"
Import-Module "${ScriptDirectoryPath}\Modules\Path"
Import-Module "${ScriptDirectoryPath}\Modules\BuildSettings"

if (!(Test-Path $InputFile -PathType Leaf))
{
    Write-Host "Error: $InputFile does not found."
    exit 1
}

function Make-SymbolTable
{
    param
    (
        $DisassembledFile
    )

    $symbolTable = @()

    foreach ($l in $DisassembledFile)
    {
        # 逆アセンブル結果からシンボルのシグネチャを探す
        if ($l -match "^([0-9a-f]+) <(.+)>:")
        {
            # Matches[1] = address, $Matches[2] = name
            $symbolAddress = $Matches[1];
            $symbolName = $Matches[2];

            # シンボルテーブルとしてシンボル名と行数の情報を登録
            $symbolTable += @{
                address = ${symbolAddress};
                name = ${symbolName};
            }
        }
    }

    $symbolTable
}

# ToolChains のパスを解決
$toolChainsRoot = "$(Get-NintendoSdkRootPath)\ToolChains"
if ($Env:SIGLO_TOOLCHAINS_ROOT){
    $toolChainsRoot = $Env:SIGLO_TOOLCHAINS_ROOT
}

# Rynda のバージョンを取得
$ryndaVersion = Get-BuildToolChainVersion("Rynda")

# objdump のパスを取得
$objdumpTool = "$toolChainsRoot\$ryndaVersion\nx\aarch64\bin\aarch64-nintendo-nx-elf-objdump.exe"

# objdump -h --wide <file> でヘッダを出力し、LOAD が含まれる行の二番目の要素をセクション名として抜き出す
$targetSections = (& $objdumpTool -h --wide $InputFile | Select-String $TargetSectionAttribute | Foreach-Object { $_.ToString().TrimStart().Split(" ")[1] })

# 出力ファイルを作成
New-Item -ItemType file $OutputFile -Force | Out-Null

# ダンプファイルを別途保存
if (! $DisableFullContentsDump)
{
    $OutputFile2 = $OutputFile -replace "\.dasm$",".dump"
    New-Item -ItemType file $OutputFile2 -Force | Out-Null
}

if ($HeaderOptions)
{
    # ヘッダの出力指定があれば先頭にヘッダを出力。ただし3行目まではファイル名が記載されるのでスキップ
    (& $objdumpTool $HeaderOptions --wide $InputFile) | Select-Object -Skip 3 | Add-Content $OutputFile
}

# 各セクションに対して逆アセンブルを実行していく
# ここでオプション文字列が長くなりすぎると objdump がエラーになってしまうので一定値を超えたらコマンドを実行して追記していく
# たぶん 32KB が限界
$cmdOptionParams = @()
$cmdOptionString = ""
foreach ($i in $targetSections)
{
    if ($i -eq "")
    {
        continue
    }

    $cmdOptionParams += '-j'
    $cmdOptionParams += $i
    $cmdOptionString += "-j, $i "
    if ($cmdOptionString.Length -gt 30000)
    {
        # ファイル名が記載される先頭はスキップ
        (& $objdumpTool -D --demangle $cmdOptionParams $InputFile) | Select-Object -Skip 3 | Add-Content $OutputFile
        if (! $DisableFullContentsDump)
        {
            (& $objdumpTool -s $cmdOptionParams $InputFile) | Select-Object -Skip 3 | Add-Content $OutputFile2
        }
        $cmdOptionParams = @()
        $cmdOptionString = ""
    }
}

if ($cmdOptionString)
{
    (& $objdumpTool -D --demangle $cmdOptionParams $InputFile) | Select-Object -Skip 3 | Add-Content $OutputFile
    if (! $DisableFullContentsDump)
    {
        (& $objdumpTool -s $cmdOptionParams $InputFile) | Select-Object -Skip 3 | Add-Content $OutputFile2
    }
}


# 逆アセンブルしたとき、__preinit_array_end と _start が同じアドレスに
# 定義されていることで出力結果が意図せず変わってしまうことの対処として
# 特定の attribute を持つシンボル名を置き換える
if ( $EnableSymbolReplace )
{
    Write-Host "Symbol replacement: start"

    # nm のパスを取得
    $nmTool = "$toolChainsRoot\$ryndaVersion\nx\aarch64\bin\aarch64-nintendo-nx-elf-nm.exe"

    # nm コマンドで定義済シンボルをアドレスでソートしつつ取得
    $symbols = (& $nmTool -C -n --defined-only $InputFile) | Get-Unique

    # 配列に変換
    $symbolTable = @()
    foreach ($l in $symbols)
    {
        $a = ($l -split "\s+")
        $symbolTable += @{
            address = $a[0];
            attribute = $a[1];
            name = $a[2];
        }
    }

    # 逆アセンブル結果のファイル内容を取得
    $content = (Get-Content $OutputFile)

    # 逆アセンブル結果からシンボル情報を抽出
    $symbolsInDisasm = Make-SymbolTable $content

    foreach ($s in $symbolTable)
    {
        # 特定のシンボルタイプが置き換え対象
        # 現在は "A" "a" の絶対値シンボルのみ
        if (($s.attribute -eq "A"))
        {
            $isSymbolInDisasm = $false

            # 逆アセンブル結果内にシンボルが定義されていたら置き換えない
            foreach ($t in $symbolsInDisasm)
            {
                if ($t.name.Contains($s.name))
                {
                    $isSymbolInDisasm = $true
                    break
                }
            }

            if (! $isSymbolInDisasm)
            {
                # 同じアドレスの別名シンボルがあったら置換する
                foreach ($t in $symbolsInDisasm)
                {
                    if (($s.address -eq $t.address) -and !($t.name.Contains($s.name)))
                    {
                        "{0}: {1} --> {2}" -f $s.address,$s.name,$t.name
                        $content = $content -replace $s.name, $t.name
                        break
                    }
                }
            }
        }
    }

    # 置換された内容を出力ファイルに書き戻す
    $content | Set-Content -Encoding UTF8 $OutputFile

    Write-Host "Symbol replacement: done"
}

exit 0
