﻿#
#    COMMAND [-InputFile <inputFile> | -InputDir <directory>] [-Lang <lang>] [-OverWrite] [-Detail]
#
#       -InputFile  <inputFile>       入力ファイル名。
#                                     この指定か、-InputDir 指定のどちらかを行う。
#
#       -InputDir   <directory>       XML があるディレクトリ。
#                                     ここ以下の .xml が対象となる。
#                                     この指定か、-InputFile 指定のどちらかを行う。
#
#       -Lang <lang>                  変換する言語。en を指定すると _ja => _en とする。
#                                     ja を指定すると _en => _ja とする。
#                                     省略時は en 指定と同じ動作となる。
#
#       -Overwrite                    元ファイルに出力を上書きする。
#                                     指定がない場合、.output を付けたファイル名となる。
#
#       -Detail                       動作の情報を詳細に表示する。省略可で、省略時は詳細を表示しない。
#
param
(
    [String]$InputFile = "",
    [String]$InputDir = "",
    [String]$Lang = "en",
    [switch]$Overwrite = $false,
    [switch]$Detail = $false,
    [switch]$Help = $false
)

$enMark = "\"  #dummy"


# ヘルプ
if ( $Help )
{
    Write-Host @"

    コマンド名
        Convert-FileNameReference.ps1

    説明
        XML ファイルの中の filename="xxxx_ja.jpg" のようなファイル指定を変換します。
        言語指定によって xxx_ja.jpg の部分を xxx_en.jpg に変換したり、その逆が可能です。
        実際は、<ri:attachment ri:filename="****"> という部分を探します。

    書式
        Convert-FileNameReference [-InputFile <inputFile> | -InputDir <inputDir>] [-Lang en|ja]
                                  [-Overwrite] [-Detail] [-Help]

    オプション
        -InputFile <inputFile>  処理のの対象となるファイルを指定します。
                                .xml の拡張子を持つファイルである必要があります。
                                次の -InputDir と同時に指定はできません。どちらかを指定する必要はあります。

        -InputDir <inputDir>    処理の対象となるファイルが格納されたディレクトリを指定します。
                                このディレクトリ以下にある、.xml の拡張子を持つファイルが処理対象となります。
                                前の -InputFile と同時に指定はできません。どちらかを指定する必要はあります。

        -Lang <lang>            処理の言語を指定します。
                                en と ja が指定でき、省略すると en と見なします。
                                en の場合、ファイル拡張子前に _ja が付くファイル名を探し、_en に変換します。
                                ja の場合、ファイル拡張子前に _en が付くファイル名を探し、_ja に変換します。

        -Overwrite              元ファイルに上書きします。
                                指定がない場合は 元ファイル.output というファイルに書き込みます。

        -Detail                 指定すると処理の詳細を表示します。

        -Help                   指定するとこのヘルプを表示します。変換処理は行いません。

    注意
        入力ファイルは UTF-8 であることを想定しています。
        出力も UTF-8 で行います。

"@
    exit 0
}

# ファイル・フォルダ両方の指定がされている
if ( $InputFile -ne "" -and $InputDir -ne "" )
{
    Write-Host "*** Cannot specify input file and xml directory at the same time."
    exit 1
}

# ファイル・フォルダ両方の指定がない
if ( $InputFile -eq "" -and $InputDir -eq "" )
{
    Write-Host "*** Not specified file or directry."
    exit 1
}

# ファイルパスが存在しない・InputFile にフォルダを指定した
if ( $InputFile -ne "" )
{
    if ( -not ( Test-Path -path $InputFile ) )
    {
        Write-Host "*** File not found: $InputFile"
        exit 1
    }
    if ( (Get-Item $InputFile).PSIsContainer )
    {
        Write-Host "*** -InputFile can get only file."
        exit 1
    }
}

# フォルダパスが存在しない・InputDir にファイルを指定した
if ( $InputDir -ne "" )
{
    if ( -not ( Test-Path -path $InputDir ) )
    {
        Write-Host "*** Directory not found: $InputDir"
        exit 1
    }
    if ( -not (Get-Item $InputDir).PSIsContainer )
    {
        Write-Host "*** -InputDir can get only directory."
        exit 1
    }
}

# 言語(全部小文字にしておく)
$Lang = $Lang.ToLower()

#----------------------------------------------------------------
#
function Convert-File( $fileName )
{
    #---- .xml でない場合は戻る
    if ( -not ( [System.IO.Path]::GetExtension( $fileName ) -eq ".xml" ) )
    {
        if ( $Detail )
        {
            Write-Host "*** Skip    $fileName (not XML file)"
        }
        return $false
    }

    if ( $Detail )
    {
        Write-Host "*** Convert $fileName"
    }

    # 出力する文字列
    [string[]]$dest = @()


    #---- ファイル読み込み
    Get-Content -Encoding UTF8 $fileName | foreach `
    {
        if ( $_.Contains("<ri:attachment") )
        {
            $c = $_
            [string]$outStr = ""
            While( $true )
            {
                if ( $c -cmatch "(.*?<ri:attachment ri:filename=`")([^`"]*)(.*)" )
                {
                    $outStr += $matches[1]
                    $c = $matches[3]
                    $imgName = $matches[2]
                    #Write-Host $imgName

                    if ( $Lang -eq "en" )
                    {
                        # ja-->en
                        if ( $imgName -cmatch "(.*?)_ja(\.[a-zA-Z]*)$" )
                        {
                           if ( $Detail )
                           {
                               Write-Host "Found: " $imgName
                           }
                           $imgName = $matches[1] + "_en" + $matches[2]
                        }
                    }
                    else
                    {
                        # en-->ja
                        if ( $imgName -cmatch "(.*?)_en(\.[a-zA-Z]*)$" )
                        {
                           if ( $Detail )
                           {
                               Write-Host "Found: " $imgName
                           }
                           $imgName = $matches[1] + "_ja" + $matches[2]
                        }
                    }
                    $outStr += $imgName
                }
                else
                {
                    $outStr += $c
                    $dest += $outStr
                    break
                }
            }
        }
        else
        {
            $dest += $_
        }
    }

    #---- 上書きか別ファイルか
    if ( $Overwrite -eq $true )
    {
        $OutputFile = $fileName
    }
    else
    {
        $OutputFile = $fileName + ".output"
    }

    #---- 出力
    New-Item -force -ItemType file $OutputFile | Out-Null
    ($dest -join "`r`n") | Out-File -Encoding UTF8 $OutputFile -Append

    return $true
}

#================================================================
#---- ファイル一つに対して処理
if ( $InputFile -ne "" )
{
   Convert-File $InputFile
}
#---- 指定フォルダのファイルに対して処理
else
{
    Get-ChildItem $InputDir | Foreach-Object `
    {
        if ( -not $_.PSIsContainer )
        {
            $result = Convert-File ($InputDir + $enMark + $_)
            if ( $Detail -and $result -eq $true )
            {
                Write-Host "*** Done."
            }
        }
    }
}
