﻿#
#    COMMAND [-HtmlDir <directory>] -TagFile <tagfile> [-OutputHtml <html>] [-HtmlPrefix <prefix>] [-Detail]
#
#       -HtmlDir <directory>          HTML があるディレクトリ。
#                                     ここ以下の html が対象となる。
#                                     省略時はカレントディレクトリとなる。
#
#       -TagFile <tagfile>            APIリファレンスのタグファイルjsの位置。
#
#       -OutputHtml <html>            結果を出力するなら指定する。指定が無ければ html は作成しない。
#
#       -HtmlPrefix <prefix>          結果出力された html から、リンクを張る時に加えるパス。
#                                     省略時は -HtmlDir で指定したディレクトリ
#                                     "..\..\Documents\Api\Pages" のような形で。(最後に \ をつけない)
#
#       -Detail                       動作の情報を詳細に表示する。省略化で、省略時は詳細を表示しない。
#
#      (例) COMMAND -HtmlDir "..\..\Pages" -TagFile "..\..\..\apiReference.js" -OutputHtml "resule.html" -HtmlPrefix "..\..\Documents\Api\Pages"
#
param
(
    [String]$HtmlDir = "",
    [String]$TagFile = "",
    [String]$OutputHtml = "",
    [String]$HtmlPrefix = "",
    [switch]$Detail = $false
)

$table = @{}
[string[]]$htmlLine = @()
$foundCount = 0
$enMark = "\"  #dummy"

if ( $HtmlDir -eq "" )
{
    $HtmlDir = "." + $enMark
}

[switch]$IsOutputHtml = $false
if ( $OutputHtml -ne "" )
{
    $IsOutputHtml = $true
}

#----------------------------------------------------------------
#  各ファイルのチェックを行う
#
function Check-EachFile( $fileName )
{
    #---- .html でない場合は戻る
    if ( -not ( [System.IO.Path]::GetExtension($fileName) -eq ".html" ) )
    {
        if ( $Detail )
        {
            Write-Host "*** Skip $fileName ..."
        }
        return
    }

    if ( $Detail )
    {
        Write-Host "*** Check $fileName ..."
    }

    #---- ファイル読み込み
    $num = 0
    $count = 0
    $isTitleFound = $false
    $titleString = "???"
    Get-Content -Encoding UTF8 ($HtmlDir + $enMark + $fileName) | foreach `
    {
        $num ++
        if ( ($_ -match "span") )
        {
            $c = $_
            while( $true )
            {
                if ( $c -match "(.*?)<span class=`"(ApiLink_.*?)`">(.*?)</span>(.*)" )
                {
                    #---- 存在するかチェック
                    if ( ! $table.ContainsKey( $matches[2] ) )
                    {
                        #---- 存在しなかった
                        Write-Host ("*** {0}:{1}: Not found link {2}" -f $fileName, $num, $matches[3])
                        if ( $IsOutputHtml )
                        {
                            if ( $HtmlPrefix -ne "" )
                            {
                                $filePath = $HtmlPrefix + $enMark + $fileName
                            }
                            else
                            {
                                $filePath = $HtmlDir + $enMark + $fileName
                            }
                            if ( $count -eq 0 )
                            {
                                 $script:foundCount ++
                                 $script:htmlLine += ("<h2>({0}) <a href=`"{1}`">{1}</a> {2}</h2>" -f $foundCount, $filePath, $titleString)
                            }
                            $script:htmlLine += ("<h3>({0}) <b><font color=`"green`">Line {3}</font>: <font color=`"red`">{4}</font></b></h3>" `
                            -f ($count + 1), $fileName, $filePath, $num, $matches[3])
                            $script:htmlLine += ("<pre>{0}</pre>" -f $_)
                        }
                        $count ++
                    }

                    $c = $matches[4]
                }
                else
                {
                    break
                }
            }
        }
        else
        {
            #---- <title></title> があれば覚えておく
            if ( (! $isTitleFound) -and ($_ -match "<title>(.*?)</title>") )
            {
                $isTitleFound = $true
                $titleString = $matches[1]
            }
        }
    }
}

#----------------------------------------------------------------
# 指定タグファイルからテーブルを作成する
#
function Create-TagTable( $tagFile )
{
    Get-Content -Encoding UTF8 $tagFile | foreach `
    {
        if ( $_ -match "^[ ]*SetUrl\( preUrl, '(.*?)'" )
        {
            if ( ! $table.ContainsKey( $matches[1] ) )
            {
                $table.Add( $matches[1], 0 )
            }
        }
    }

    if ( $Detail )
    {
        Write-Host ("*** TagFile length: {0}" -f $table.Count)
    }
}

#================================================================
#  メイン

#---- タグファイル読み込み
if ( $tagFile -eq "" -or $tagFile -eq $null )
{
    Write-Host "*** Not specified tagfile."
    exit 1
}
if ( (Test-Path -path "$tagFile") -eq $false  )
{
    Write-Host "*** Not found $tagFile."
    exit 1
}
Create-TagTable $tagFile

#---- 指定フォルダのそれぞれのファイルに対して処理
Get-ChildItem $HtmlDir | Foreach-Object `
{
    if ( -not $_.PSIsContainer )
    {
        Check-EachFile $_
    }
}

#---- 必要なら HTML に結果出力
if ( $IsOutputHtml )
{
    New-Item $OutputHtml -itemType File -Force | Out-Null

    #---- ヘッダ
@"
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
<title>API 自動リンクチェック結果</title>
<style type="text/css">
<!--
span{ color:#300000; }
h1{ background:#c0ffa0; color:#303060; font-size:18pt; width:95%; padding:10px; margin-left:20px; }
h2{ background:#e0ffc0; font-size:16pt; width:80%; padding:8px; margin-left:20px; }
h3{ font-size:12pt; width:80%; padding:8px; margin-left:30px; }
pre{ background:#f0f0ff; font-size:10pt; width:60%; padding:8px; margin-left:50px; word-wrap: break-word; }
-->
</style>
</head>
<body>
<h1> API リファレンス自動リンクチェック結果 </h1>

"@  | Out-File -Encoding UTF8 $OutputHtml -Append

    #---- いくつ見つかったかの表示
    if ( $foundCount -ne 0 )
    {
        ("<p style=`"margin-left:30pt; color:blue; font-weight:bold;`">Found {0} file(s).</p>" -f $foundCount)  | Out-File -Encoding UTF8 $OutputHtml -Append
    }
    else
    {
        "<p style=`"margin-left:30pt; color:blue; font-weight:bold;`">Not found.</p>" | Out-File -Encoding UTF8 $OutputHtml -Append
    }

    #---- 結果
    ($htmlLine -join "`r`n") | Out-File -Encoding UTF8 $OutputHtml -Append

    #---- フッタ
@"
<hr>
</body>
</html>
"@  | Out-File -Encoding UTF8 $OutputHtml -Append
}

#---- 完了
if ( $Detail -eq $true )
{
    Write-Host "*** Done."
}
