﻿// 文字コード:UTF-8
/// @file
#pragma once

#include <lib/Math.hpp>
#include <lib/ShortString.hpp>

//------------------------------------------------------------------------------
namespace lib {
    struct Vector2Pod;
    struct Vector3Pod;
}

//------------------------------------------------------------------------------
namespace lib {

/// @addtogroup LIB-Math
//@{
/// 浮動小数型4次元ベクトル構造体。
/// @details メモリ配置は32bit浮動小数が先頭からx,y,z,wの順番に並んでいます。
struct Vector4Pod
{
    /// @name 変数
    //@{
    float x; ///< x。
    float y; ///< y。
    float z; ///< z。
    float w; ///< w。
    //@}

    /// @name 定数取得
    //@{
    static const Vector4Pod Zero();     ///< 0ベクトル。
    static const Vector4Pod One();      ///< X,Y,Z,Wが全て1のベクトル。
    static const Vector4Pod Min();      ///< 最小値のベクトル。
    static const Vector4Pod Max();      ///< 最大値のベクトル。
    static const Vector4Pod UnitX();    ///< X単位ベクトル。
    static const Vector4Pod UnitY();    ///< Y単位ベクトル。
    static const Vector4Pod UnitZ();    ///< Z単位ベクトル。
    static const Vector4Pod UnitW();    ///< W単位ベクトル。
    static const Vector4Pod NegUnitX(); ///< -X単位ベクトル。
    static const Vector4Pod NegUnitY(); ///< -Y単位ベクトル。
    static const Vector4Pod NegUnitZ(); ///< -Z単位ベクトル。
    static const Vector4Pod NegUnitW(); ///< -W単位ベクトル。
    //@}

    /// @name 変換
    //@{
    const Vector2Pod toXY() const;
    const Vector2Pod toXX() const;
    const Vector2Pod toYY() const;
    const Vector2Pod toZZ() const;
    const Vector2Pod toWW() const;
    const Vector3Pod toXYZ() const;
    const Vector3Pod toXXX() const;
    const Vector3Pod toYYY() const;
    const Vector3Pod toZZZ() const;
    const Vector3Pod toWWW() const;
    const Vector4Pod toXYZ0() const;
    const Vector4Pod toXY0W() const;
    const Vector4Pod toXY00() const;
    const Vector4Pod toX0ZW() const;
    const Vector4Pod toX0Z0() const;
    const Vector4Pod toX00W() const;
    const Vector4Pod toX000() const;
    const Vector4Pod to0YZ0() const;
    const Vector4Pod to0Y0W() const;
    const Vector4Pod to0Y00() const;
    const Vector4Pod to00ZW() const;
    const Vector4Pod to00Z0() const;
    const Vector4Pod to000W() const;
    const Vector4Pod toXXXX() const;
    const Vector4Pod toYYYY() const;
    const Vector4Pod toZZZZ() const;
    const Vector4Pod toWWWW() const;
    //@}

    /// @name 等価比較
    //@{
    /// 許容誤差を考慮した等価比較。
    bool equals(const Vector4Pod& aVec) const;
    /// 許容誤差を許容しない等価比較。
    bool equalsStrict(const Vector4Pod& aVec) const;
    //@}

    /// @name 四則演算
    //@{
    /// 各要素に値を加算した結果を取得する。
    const Vector4Pod add(float aVal) const;
    /// 各要素から値を減算した結果を取得する。
    const Vector4Pod sub(float aVal) const;
    /// 各要素に値をかけた結果を取得する。
    const Vector4Pod mul(float aVal) const;
    /// 各要素から値をわった結果を取得する。
    const Vector4Pod div(float aVal) const;
    /// 対応する値同士を加算した結果を取得する。
    const Vector4Pod add(const Vector4Pod& aVal) const;
    /// 対応する値に対して減算した結果を取得する。
    const Vector4Pod sub(const Vector4Pod& aVal) const;
    /// 対応する値同士をかけた結果を取得する。
    const Vector4Pod mul(const Vector4Pod& aVal) const;
    /// 対応する値に対して割り算をした結果を取得する。
    const Vector4Pod div(const Vector4Pod& aVal) const;
    /// 各要素に値を加算する。
    void addAssign(float aVal);
    /// 各要素から値を減算する。
    void subAssign(float aVal);
    /// 各要素に値をかける。
    void mulAssign(float aVal);
    /// 各要素から値をわる。
    void divAssign(float aVal);
    /// 対応する値同士を加算する。
    void addAssign(const Vector4Pod& aVal);
    /// 対応する値に対して減算する。
    void subAssign(const Vector4Pod& aVal);
    /// 対応する値同士をかける。
    void mulAssign(const Vector4Pod& aVal);
    /// 対応する値に対して割り算をする。
    void divAssign(const Vector4Pod& aVal);
    //@}

    /// @name 演算子オーバーロード
    //@{
    const Vector4Pod operator+(float aVal) const; ///< add()。
    const Vector4Pod operator-(float aVal) const; ///< sub()。
    const Vector4Pod operator*(float aVal) const; ///< mul()。
    const Vector4Pod operator/(float aVal) const; ///< div()。
    const Vector4Pod operator+(const Vector4Pod& aVal) const; ///< add()。
    const Vector4Pod operator-(const Vector4Pod& aVal) const; ///< sub()。
    const Vector4Pod operator*(const Vector4Pod& aVal) const; ///< mul()。
    const Vector4Pod operator/(const Vector4Pod& aVal) const; ///< div()。
    const Vector4Pod operator-() const; ///< neg()。
    Vector4Pod& operator+=(float aVal); ///< addAssign()。
    Vector4Pod& operator-=(float aVal); ///< subAssign()。
    Vector4Pod& operator*=(float aVal); ///< mulAssign()。
    Vector4Pod& operator/=(float aVal); ///< divAssign()。
    Vector4Pod& operator+=(const Vector4Pod& aVal); ///< addAssign()。
    Vector4Pod& operator-=(const Vector4Pod& aVal); ///< subAssign()。
    Vector4Pod& operator*=(const Vector4Pod& aVal); ///< mulAssign()。
    Vector4Pod& operator/=(const Vector4Pod& aVal); ///< divAssign()。
    //@}

    /// @name 選択
    //@{
    /// 指定のベクトルとの各要素の最小値を選択したベクトルを取得する。
    const Vector4Pod min(const Vector4Pod& aVal) const;
    /// 指定のベクトルとの各要素の最大値を選択したベクトルを取得する。
    const Vector4Pod max(const Vector4Pod& aVal) const;
    //@}

    /// @name クランプ
    //@{
    /// aMin以上aMax以下になるようにクランプする。
    const Vector4Pod clamp(const Vector4Pod& aMin, const Vector4Pod& aMax) const;
    /// clamp(Zero(), Max()) を取得する。
    const Vector4Pod clampPositive() const;
    /// clamp(Min(), Zero()) を取得する。
    const Vector4Pod clampNegative() const;
    //@}

    /// @name 符号操作
    //@{
    /// 各要素を正に置き換えたベクトルを取得。
    const Vector4Pod abs() const;
    /// 符号を反転させた結果を取得する。
    const Vector4Pod neg() const;
    //@}

    /// @name 長さ
    //@{
    /// 長さの2乗を取得する。
    float squareLength() const;
    /// 長さを取得する。
    float length() const;
    /// あるベクトルとの距離を取得する。
    float distance(const Vector4Pod& aVal) const;
    /// 長さが0か。許容誤差を許す。
    bool isZero() const;
    /// 長さが0か。許容誤差を許さない。
    bool isZeroStrict() const;
    /// 正規化済みか。許容誤差を許す。
    //@}

    /// @name 正規化
    //@{
    bool isUnit() const;
    /// 正規化したベクトルを取得する。
    /// @details 長さ0のベクトルで正規化するとエラーになります。
    const Vector4Pod unit() const;
    /// 正規化する。
    /// @details 長さ0のベクトルで正規化するとエラーになります。
    void unitAssign();
    //@}

    /// @name 内積
    //@{
    /// あるベクトルとの内積を取得する。
    float dot(const Vector4Pod& aVal) const;
    //@}

    /// @name ユーティリティ
    //@{
    /// x,y,z,w形式の文字列に変換。
    const ::lib::ShortString toShortString() const;
    //@}
};
//@}

/// Vector4Pod のエイリアス。
typedef Vector4Pod Vector4fPod;

} // namespace
// EOF
