﻿// 文字コード:UTF-8
/// @file
#include <lib/Vector2iPod.hpp>

#include <limits>
#include <lib/Math.hpp>
#include <lib/debug/Assert.hpp>

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::Zero()
{
    Vector2iPod obj = {};
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::One()
{
    Vector2iPod obj = {1, 1};
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::Min()
{
    Vector2iPod obj = {
        ::std::numeric_limits<int>::min(),
        ::std::numeric_limits<int>::min()
    };
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::Max()
{
    Vector2iPod obj = {
        ::std::numeric_limits<int>::max(),
        ::std::numeric_limits<int>::max()
    };
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::UnitX()
{
    Vector2iPod obj = {1, 0};
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::UnitY()
{
    Vector2iPod obj = {0, 1};
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::NegUnitX()
{
    Vector2iPod obj = {-1, 0};
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::NegUnitY()
{
    Vector2iPod obj = {0, -1};
    return obj;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::toX0() const
{
    return Vector2i(x, 0);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::toXX() const
{
    return Vector2i(x, x);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::toYX() const
{
    return Vector2i(y, x);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::toYY() const
{
    return Vector2i(y, y);
}

//------------------------------------------------------------------------------
const Vector2 Vector2iPod::toXYf() const
{
    return Vector2(float(x), float(y));
}

//------------------------------------------------------------------------------
void Vector2iPod::set(int aX, int aY)
{
    this->x = aX;
    this->y = aY;
}

//------------------------------------------------------------------------------
bool Vector2iPod::equals(const Vector2iPod& aVec) const
{
    return x == aVec.x && y == aVec.y;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::add(const int aVal) const
{
    Vector2i vec(*this);
    vec.addAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::sub(const int aVal) const
{
    Vector2i vec(*this);
    vec.subAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::mul(const int aVal) const
{
    Vector2i vec(*this);
    vec.mulAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::div(const int aVal) const
{
    Vector2i vec(*this);
    vec.divAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::add(const Vector2iPod& aVal) const
{
    Vector2i vec(*this);
    vec.addAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::sub(const Vector2iPod& aVal) const
{
    Vector2i vec(*this);
    vec.subAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::mul(const Vector2iPod& aVal) const
{
    Vector2i vec(*this);
    vec.mulAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::div(const Vector2iPod& aVal) const
{
    Vector2i vec(*this);
    vec.divAssign(aVal);
    return vec;
}

//------------------------------------------------------------------------------
void Vector2iPod::addAssign(const int aVal)
{
    x += aVal;
    y += aVal;
}

//------------------------------------------------------------------------------
void Vector2iPod::subAssign(const int aVal)
{
    x -= aVal;
    y -= aVal;
}

//------------------------------------------------------------------------------
void Vector2iPod::mulAssign(const int aVal)
{
    x *= aVal;
    y *= aVal;
}

//------------------------------------------------------------------------------
void Vector2iPod::divAssign(const int aVal)
{
    if (aVal == 0) {
        SYS_ASSERT_NOT_REACHED();
        return;
    }
    x /= aVal;
    y /= aVal;
}

//------------------------------------------------------------------------------
void Vector2iPod::addAssign(const Vector2iPod& aVal)
{
    x += aVal.x;
    y += aVal.y;
}

//------------------------------------------------------------------------------
void Vector2iPod::subAssign(const Vector2iPod& aVal)
{
    x -= aVal.x;
    y -= aVal.y;
}

//------------------------------------------------------------------------------
void Vector2iPod::mulAssign(const Vector2iPod& aVal)
{
    x *= aVal.x;
    y *= aVal.y;
}

//------------------------------------------------------------------------------
void Vector2iPod::divAssign(const Vector2iPod& aVal)
{
    x /= aVal.x;
    y /= aVal.y;
}

//------------------------------------------------------------------------------
bool Vector2iPod::operator==(const Vector2iPod& aRhs) const
{
    return equals(aRhs);
}

//------------------------------------------------------------------------------
bool Vector2iPod::operator!=(const Vector2iPod& aRhs) const
{
    return !equals(aRhs);
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator+=(const int aVal)
{
    addAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator-=(const int aVal)
{
    subAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator*=(const int aVal)
{
    mulAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator/=(const int aVal)
{
    divAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator+=(const Vector2iPod& aVal)
{
    addAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator-=(const Vector2iPod& aVal)
{
    subAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator*=(const Vector2iPod& aVal)
{
    mulAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
Vector2iPod& Vector2iPod::operator/=(const Vector2iPod& aVal)
{
    divAssign(aVal);
    return *this;
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator+(const int aVal) const
{
    return add(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator-(const int aVal) const
{
    return sub(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator*(const int aVal) const
{
    return mul(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator/(const int aVal) const
{
    return div(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator+(const Vector2iPod& aVal) const
{
    return add(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator-(const Vector2iPod& aVal) const
{
    return sub(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator*(const Vector2iPod& aVal) const
{
    return mul(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator/(const Vector2iPod& aVal) const
{
    return div(aVal);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::operator-() const
{
    return neg();
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::min(const Vector2iPod& aVal) const
{
    return Vector2i(
        Math::Min(x, aVal.x),
        Math::Min(y, aVal.y)
        );
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::max(const Vector2iPod& aVal) const
{
    return Vector2i(
        Math::Max(x, aVal.x),
        Math::Max(y, aVal.y)
        );
}

//------------------------------------------------------------------------------
void Vector2iPod::clamp(const Vector2iPod& aMin, const Vector2iPod& aMax)
{
    SYS_ASSERT(aMin.x <= aMax.x && aMin.y <= aMax.y);
    *this = max(aMin).min(aMax);
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::getClamped(
    const Vector2iPod& aMin,
    const Vector2iPod& aMax
    ) const
{
    Vector2iPod r = *this;
    r.clamp(aMin, aMax);
    return r;
}


//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::getClampedPositive() const
{
    return max(Zero());
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::getClampedNegative() const
{
    return min(Zero());
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::abs() const
{
    return Vector2i(
        Math::Abs(x),
        Math::Abs(y)
        );
}

//------------------------------------------------------------------------------
const Vector2iPod Vector2iPod::neg() const
{
    Vector2i vec(*this);
    vec.mulAssign(-1);
    return vec;
}

//------------------------------------------------------------------------------
bool Vector2iPod::isPositive() const
{
    return 0 <= x && 0 <= y;
}

//------------------------------------------------------------------------------
bool Vector2iPod::isZero() const
{
    return x == 0 && y == 0;
}

//------------------------------------------------------------------------------
const ::lib::ShortString Vector2iPod::toShortString() const
{
    return ::lib::ShortString::FromFormat(
        "%d,%d",
        x,
        y
        );
}

} // namespace
// EOF
