﻿// 文字コード:UTF-8
/// @file
#include <lib/Vector2Pod.hpp>

#include <limits>
#include <lib/Vector3.hpp>
#include <lib/Vector4.hpp>
#include <lib/Unused.hpp>

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::Zero()
{
    return Vector2Pod::Create(0.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::One()
{
    return Vector2Pod::Create(1.0f, 1.0f);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::Min()
{
    const float val = -::std::numeric_limits<float>::infinity();
    return Vector2Pod::Create(val, val);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::Max()
{
    const float val = ::std::numeric_limits<float>::infinity();
    return Vector2Pod::Create(val, val);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::UnitX()
{
    return Vector2Pod::Create(1.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::UnitY()
{
    return Vector2Pod::Create(0.0f, 1.0f);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::NegUnitX()
{
    return Vector2Pod::Create(-1.0f, 0.0f);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::NegUnitY()
{
    return Vector2Pod::Create(0.0f, -1.0f);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::toX0() const
{
    return Vector2(x, 0);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::to0Y() const
{
    return Vector2(0, y);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::toXX() const
{
    return Vector2(x, x);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::toYX() const
{
    return Vector2(y, x);
}

//------------------------------------------------------------------------------
const Vector2Pod Vector2Pod::toYY() const
{
    return Vector2(y, y);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector2Pod::toXY0() const
{
    return Vector3(x, y, 0);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector2Pod::toX0Y() const
{
    return Vector3(x, 0, y);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector2Pod::toXXX() const
{
    return Vector3(x, x, x);
}

//------------------------------------------------------------------------------
const Vector3Pod Vector2Pod::toYYY() const
{
    return Vector3(y, y, y);
}

//------------------------------------------------------------------------------
const Vector4Pod Vector2Pod::toXY00() const
{
    return Vector4(x, y, 0, 0);
}

//------------------------------------------------------------------------------
const Vector4Pod Vector2Pod::toXXXX() const
{
    return Vector4(x, x, x, x);
}

//------------------------------------------------------------------------------
const Vector4Pod Vector2Pod::toYYYY() const
{
    return Vector4(y, y, y, y);
}

//------------------------------------------------------------------------------
// 長さ

//------------------------------------------------------------------------------
/**
 * 指定ベクトルの大きさがlimitを超えているか調べ、超過分をカットする
 *
 * @return 超過していた場合trueを返す
 */
bool Vector2Pod::limit(float aLimit)
{
    float length = this->length();

    if (length > aLimit) {
        *this *= aLimit / length;
        return true;
    }
    return false;
}

//------------------------------------------------------------------------------
/**
 * 正規化する。
 *
 * ０ベクトルの場合はアサート失敗。
 *
 * @return 正規化前のベクトルの長さ。
 */
float Vector2Pod::unitAssign()
{
    float len = length();
    if (len == 0.0f) {
        SYS_ASSERT_NOT_REACHED();
        return len; // fail safe code
    }
    operator/=(len);
    // 誤差で1.0fより大きな値になった時のための補正。
    if (1.0f <= x) {
        x = 1.0f; y = 0.0f;
    } else if (x <= -1.0f) {
        x = -1.0f; y = 0.0f;
    } else if (1.0f <= y) {
        y = 1.0f; x = 0.0f;
    } else if (y <= -1.0f) {
        y = -1.0f; x = 0.0f;
    }
    return len;
}

//------------------------------------------------------------------------------
/**
 * ベクトルの長さを指定した長さにする
 *
 * @param aTarget 目標の長さ
 * @return 変化する前のベクトルの大きさ
 */
float Vector2Pod::setLength(float aTarget)
{
    float length = this->length();

    if (length == 0.0f) {
        *this = Zero();
    } else {
        *this *= aTarget / length;
    }

    return length;
}

//------------------------------------------------------------------------------
// 方向・角度

//------------------------------------------------------------------------------
/**
 * ２ベクトルのなす角のcos。
 */
float Vector2Pod::cos(const Vector2Pod& aRhs) const
{
    float denominator = length() * aRhs.length();
    if (denominator == 0.0f) {
        SYS_ASSERT_NOT_REACHED();
        return 0.0f; // fail safe code
    }
    float result = dot(aRhs) / denominator;
    if (1.0f < result) {
        return 1.0f;
    }
    if (result < -1.0f) {
        return -1.0f;
    }
    return result;
}

//------------------------------------------------------------------------------
/**
 * ２ベクトルのなす角（符号付き）。
 *
 * @param aRhs 対象となるもう一つのベクトル。
 * @return なす角（ラジアン）。
 */
float Vector2Pod::signedAngle(const Vector2Pod& aRhs) const
{
    float magFactor = this->length() * aRhs.length();

    // どちらかのベクトルの長さが0の時のなす角は0度とする。
    if (magFactor < LIB_MATH_EPSILON) {
        return 0.0f;
    }

    float cosAngle = this->dot(aRhs) / magFactor;

    // 誤差で、cosの値が１より大きかったら、なす角は0度。
    if (cosAngle > 1.0f) {
        return 0.0f;
    }

    // 誤差で、cosの値が-１より小さかったら、なす角は180度。
    if (cosAngle < -1.0f) {
        return LIB_MATH_PI;
    }

    // 一般の場合外積を用いて、半時計回りかどうかを判定。

    // 時計回りの時
    if (this->cross(aRhs) < 0.0f) {
        return -::lib::Math::ArcCosRad(cosAngle);
    } else {
        // 半時計回りの時 (v1とv2が同一直線上の時もこちら)
        return ::lib::Math::ArcCosRad(cosAngle);
    }
}

//------------------------------------------------------------------------------
/**
 * 指定角度だけ回転させる。
 *
 * 反時計周りを正方向とする
 *
 * @param aDegree 回転角度（1周を360度とする角度）
 */
void Vector2Pod::rotateDeg(float aDegree) {
    rotateRad(LIB_MATH_DEG_TO_RAD(aDegree));
}

//------------------------------------------------------------------------------
/**
 * 指定角度だけ回転させる。
 *
 * 反時計周りを正方向とする
 *
 * @param aRadian 回転角度（ラジアン）
 */
void Vector2Pod::rotateRad(float aRadian) {
    float sinFactor = ::lib::Math::SinRad(aRadian);
    float cosFactor = ::lib::Math::CosRad(aRadian);
    Vector2Pod result = Create(this->x * cosFactor - this->y * sinFactor,
                   this->x * sinFactor + this->y * cosFactor);
    this->set(result.x, result.y);
}

//------------------------------------------------------------------------------
/**
 * 別のベクトルに射影する。
 *
 * srcベクトルのbase上への射影ベクトルを求める。
 * 「ゲームプログラミングのための3Dグラフィックス数学」p8参照。
 *
 * @param aRhs 射影の基準となるベクトル。
 *            生成されるベクトルはこのベクトルに平行なものになる。
 */
void Vector2Pod::project(const Vector2Pod& aRhs) {
    float l = aRhs.length();
    if (l == 0.0f) {
        SYS_ASSERT_NOT_REACHED();
        return;
    }

    {
        // 1. srcベクトルのbase上での長さを求める。
        // 長さは次の式で求めれる。src・base / |base| (・は内積、||はベクトルの長さ)
        // 長さと言っても、srcとbaseベクトルのなす角が90以上なら、負の値になったりする。
        float length = this->dot(aRhs) / l;

        // 2. ひとまずbaseベクトルを正規化したベクトルを設定する。
        Vector2Pod result = aRhs.unit() * length;

        // 設定
        this->set(result.x, result.y);
    }
}

//------------------------------------------------------------------------------
/**
 * ある面に対して反射させる。
 *
 * @param aRhs 反射面の法線。
 */
void Vector2Pod::reflect(const Vector2Pod& aRhs) {
    reflect(aRhs, 1.0f);
}

//------------------------------------------------------------------------------
/**
 * ある面に対して反射させる。
 *
 * @param aRhs 反射面の法線。
 * @param aRatio 反射率
 */
void Vector2Pod::reflect(const Vector2Pod& aRhs, const float aRatio) {
    Vector2Pod normal(aRhs);
    if (!normal.isUnit()) {
        normal.unitAssign();
    }

    {
        float d = -(1.0f + aRatio) * this->dot(normal);
        operator+=(normal * d);
    }
}

//------------------------------------------------------------------------------
void Vector2Pod::reflect1(const Vector2Pod& aRhs)
{
    reflect(aRhs);
}

//------------------------------------------------------------------------------
void Vector2Pod::reflect2(const Vector2Pod& aRhs, const float aRatio)
{
    reflect(aRhs, aRatio);
}

//------------------------------------------------------------------------------
/**
 * 各軸それぞれ倍率が記されたベクトルを用いてスケーリングを行う
 *
 * @param aScl スケーリングベクトル
 */
void Vector2Pod::scale(const Vector2Pod& aScl)
{
    x *= aScl.x;
    y *= aScl.y;
}

//------------------------------------------------------------------------------
/**
 * 各軸それぞれ倍率が記されたベクトルを用いてスケーリングを行ったベクトルを得る
 *
 * @param aScl スケーリングベクトル
 * @return スケーリング後のベクトル
 */
Vector2Pod Vector2Pod::getScaled(const Vector2Pod& aScl) const
{
    Vector2Pod result(*this);
    result.scale(aScl);
    return result;
}

//------------------------------------------------------------------------------
void Vector2Pod::floor()
{
    set(::lib::Math::Floor(x), ::lib::Math::Floor(y));
}

//------------------------------------------------------------------------------
Vector2Pod Vector2Pod::getFloored() const
{
    Vector2 result(*this);
    result.floor();
    return result;
}

//------------------------------------------------------------------------------
Vector2iPod Vector2Pod::getFlooredVector2i() const
{
    Vector2 result = getFloored();
    return Vector2i(static_cast<int>(result.x), static_cast<int>(result.y));
}

//------------------------------------------------------------------------------
void Vector2Pod::round()
{
    set(::lib::Math::Floor(x + 0.5f), ::lib::Math::Floor(y + 0.5f));
}

//------------------------------------------------------------------------------
Vector2Pod Vector2Pod::getRounded() const
{
    Vector2 result(*this);
    result.round();
    return result;
}

//------------------------------------------------------------------------------
Vector2iPod Vector2Pod::getRoundedVector2i() const
{
    Vector2 result = getRounded();
    return Vector2i(static_cast<int>(result.x), static_cast<int>(result.y));
}

//------------------------------------------------------------------------------
// デバッグ

//------------------------------------------------------------------------------
/**
 * 文字列表現を得る。
 */
const ::lib::ShortString Vector2Pod::toShortString() const
{
    return LIB_SPRINTF("(%f, %f)", x, y);
}

} // namespace
// EOF
