﻿// 文字コード:UTF-8
/// @file
#include "lib/StringConvert.hpp"

namespace {

// 不正な文字はこの文字に置換する
const char16_t fReplacementCharacter = 0xfffd;

// 上位・下位サロゲートの上限下限
const char16_t fHighSurrogateStart   = 0xd800;
const char16_t fHighSurrogateEnd     = 0xdbff;

const char16_t fLowSurrogateStart    = 0xdc00;
const char16_t fLowSurrogateEnd      = 0xdfff;

} // namespace

namespace lib {

//------------------------------------------------------------------------------
// ・移植性、拡張性、独自エラー処理を考え、SDK関数は利用せずに実装すること
// ・UTF16のエンディアンは処理系に準拠するようにしている
// ・途中で途切れていた場合は、途切れていたUTF8文字は無かったものとなる
// ・正しいUTF8かどうかの確認処理は無い、必要があれば実装すること
void StringConvert::ConvertUtf8ToUtf16(char16_t* aDst, int aDstCap, const char* aSrc)
{
    SYS_ASSERT_POINTER(aDst);
    SYS_ASSERT(0 < aDstCap);
    SYS_ASSERT_POINTER(aSrc);
    int di = 0;
    int si = 0;
    while (1) {
        // 変換先が終端になった場合
        if (di == aDstCap - 1) {
            break;
        }
        uint8_t c0 = uint8_t(aSrc[si++]);
        // 変換元の現在位置の1文字目による分岐
        if (c0 == 0) {
            // 終端の場合
            {
                break;
            }
        } else if ((c0 & 0x80) == 0x00) {
            // 1バイト文字の場合
            {
                aDst[di++] = (c0 & 0x7f);
            }
        } else if ((c0 & 0xe0) == 0xc0) {
            // 2バイト文字の場合
            {
                uint8_t c1 = uint8_t(aSrc[si++]);
                if (c1 == 0) {
                    break;
                }
                aDst[di++] =
                    ((c0 & 0x3f) << 6) |
                    ((c1 & 0x3f) << 0);
            }
        } else  if ((c0 & 0xf0) == 0xe0) {
            // 3バイト文字の場合
            {
                uint8_t c1 = uint8_t(aSrc[si++]);
                if (c1 == 0) {
                    break;
                }
                uint8_t c2 = uint8_t(aSrc[si++]);
                if (c2 == 0) {
                    break;
                }
                aDst[di++] =
                    ((c0 & 0x0f) << 12) |
                    ((c1 & 0x3f) <<  6) |
                    ((c2 & 0x3f) <<  0);
            }
        } else if ((0x80 <= c0 && c0 <= 0xbf) || c0 == 0xfe || c0 == 0xff) {
            // 1バイト目が不正だった場合、1バイト文字とみなしてREPLACEMENT CHARACTERに置換
            {
                aDst[di++] = fReplacementCharacter;
            }
        } else {
            // 4バイト以上の文字の場合
            uint8_t c1 = uint8_t(aSrc[si++]);
            if (c1 == 0) {
                break;
            }
            uint8_t c2 = uint8_t(aSrc[si++]);
            if (c2 == 0) {
                break;
            }
            uint8_t c3 = uint8_t(aSrc[si++]);
            if (c3 == 0) {
                break;
            }
            if ((c0 & 0xf8) == 0xf0) {
                // 4バイト文字の場合
                {
                    uint32_t c =
                        ((c0 & 0x07) << 18) |
                        ((c1 & 0x3f) << 12) |
                        ((c2 & 0x3f) <<  6) |
                        ((c3 & 0x3f) <<  0);
                    c -= 0x00010000;
                    char16_t highSurrogateCand = char16_t((c >> 10) + 0xd800);
                    char16_t lowSurrogateCand = char16_t((c & 0x3ffUL) + 0xdc00);

                    if (
                        fHighSurrogateStart <= highSurrogateCand &&
                        highSurrogateCand   <= fHighSurrogateEnd &&
                        fLowSurrogateStart  <= lowSurrogateCand &&
                        lowSurrogateCand    <= fLowSurrogateEnd
                    ) {
                        // サロゲートペアならサロゲートペアに変換
                        if (di >= aDstCap - 2) {
                            break;
                        }
                        aDst[di++] = highSurrogateCand;
                        aDst[di++] = lowSurrogateCand;
                    } else {
                        // サロゲートペアでないならREPLACEMENT CHARACTERに置換
                        aDst[di++] = fReplacementCharacter;
                    }
                }
            } else if ((c0 & 0xfc) == 0xf8) {
                // 5バイト文字の場合
                {
                    uint8_t c4 = uint8_t(aSrc[si++]);
                    if (c4 == 0) {
                        break;
                    }
                    aDst[di++] = fReplacementCharacter;
                }
            } else if ((c0 & 0xfe) == 0xfc) {
                // 6バイト文字の場合
                {
                    uint8_t c4 = uint8_t(aSrc[si++]);
                    if (c4 == 0) {
                        break;
                    }
                    uint8_t c5 = uint8_t(aSrc[si++]);
                    if (c5 == 0) {
                        break;
                    }
                    aDst[di++] = fReplacementCharacter;
                }
            } else {
                // もしいずれにも該当しなかった場合、REPLACEMENT CHARACTERに置換
                {
                    aDst[di++] = fReplacementCharacter;
                }
            }
        }
    }
    SYS_ASSERT(di < aDstCap);
    // ヌル終端を付加する
    aDst[di] = 0;
}

//------------------------------------------------------------------------------
// ・移植性、拡張性、独自エラー処理を考え、SDK関数は利用せずに実装すること
// ・UTF16のエンディアンは処理系に準拠するようにしている
// ・サロゲードペアには対応していない、必要があれば対応すること
void StringConvert::ConvertUtf16ToUtf8(char* aDst, int aDstCap, const char16_t* aSrc)
{
    SYS_ASSERT_POINTER(aDst);
    SYS_ASSERT(0 < aDstCap);
    SYS_ASSERT_POINTER(aSrc);
    int si = 0;
    int di = 0;
    while (1) {
        // 変換先が終端になった場合
        if (di == aDstCap - 1) {
            break;
        }
        char16_t c = aSrc[si++];
        // 変換元の現在位置の1文字目による分岐
        if (c == 0) {
            break;
        } else if (c <= 0x007f) {
            // 1バイト文字の場合
            {
                aDst[di++] = (c & 0x007f);
            }
        } else if (c <= 0x07ff) {
            // 2バイト文字の場合
            {
                if (di + 1 >= aDstCap - 1) {
                    break;
                }
                aDst[di++] = static_cast<char>(((c & 0x07c0) >>  6) | 0xc0);
                aDst[di++] = ((c & 0x003f) >>  0) | 0x80;
            }
        } else {
            // 3バイト文字の場合
            {
                if (di + 2 >= aDstCap - 1) {
                    break;
                }
                aDst[di++] = static_cast<char>(((c & 0xf000) >> 12) | 0xe0);
                aDst[di++] = static_cast<char>(((c & 0x0fc0) >>  6) | 0x80);
                aDst[di++] = ((c & 0x003f) >>  0) | 0x80;
            }
        }
    }
    SYS_ASSERT(di < aDstCap);
    // ヌル終端を付加する
    aDst[di] = 0;
}

} // namespace
// EOF
