﻿// 文字コード:UTF-8
/// @file

#include "lib/String.hpp"

#include <stdarg.h>
#include <string>
#include <cstring>
#include "lib/Math.hpp"
#include "lib/CString.hpp"
#include "lib/StringConvert.hpp"
#include "lib/StringUtil.hpp"
#include "lib/U16CString.hpp"

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
String String::GetEmpty()
{
    return String(1, "");
}

//------------------------------------------------------------------------------
String String::FromUtf16(const char16_t* aStr)
{
    SYS_ASSERT_POINTER(aStr);
    String string(CString::GetMaxUtf8CapacityFromUtf16Length(::lib::U16CString::GetLength(aStr)));
    string.setFromUtf16AsPossible(aStr);
    return string;
}

//------------------------------------------------------------------------------
String String::FromUtf16(const U16String& aStr)
{
    SYS_ASSERT(aStr.capacity() > 0);
    String string(CString::GetMaxUtf8CapacityFromUtf16Length(aStr.capacity() - 1));
    string.setFromUtf16AsPossible(aStr);
    return string;
}

//------------------------------------------------------------------------------
String String::FromFormat(int aCap, const char* aStr, ...)
{
    ::std::va_list vaList = { 0 };
    va_start(vaList, aStr);
    String string = FromFormatV(aCap, aStr, vaList);
    va_end(vaList);
    return string; //lint !e438
}

//------------------------------------------------------------------------------
String String::FromFormatV(int aCap, const char* aStr, ::std::va_list aVaList)
{
    String string(aCap);
    string.formatV(aStr, aVaList);
    return string;
}

//------------------------------------------------------------------------------
String String::FromFormatAsPossible(int aCap, const char* aStr, ...)
{
    ::std::va_list vaList = {};
    va_start(vaList, aStr);
    String string = FromFormatVAsPossible(aCap, aStr, vaList);
    va_end(vaList);
    return string;
}

//------------------------------------------------------------------------------
String String::FromFormatVAsPossible(int aCap, const char* aStr, ::std::va_list aVaList)
{
    String string(aCap);
    string.formatVAsPossible(aStr, aVaList);
    return string;
}

//------------------------------------------------------------------------------
String::String()
: mBuf(0)
, mCap(0)
{}

//------------------------------------------------------------------------------
String::String(int aCap)
: mBuf(0)
, mCap(aCap)
{
    SYS_ASSERT(0 < aCap);
    mBuf = new char[mCap];
    mBuf[0] = 0;
}

//------------------------------------------------------------------------------
String::String(const char* aStr)
: mBuf(0)
, mCap(0)
{
    reset(aStr);
}

//------------------------------------------------------------------------------
String::String(int aCap, const char* aStr)
: mBuf(0)
, mCap(0)
{
    SYS_ASSERT(0 < aCap);
    reset(aCap, aStr);
}

//------------------------------------------------------------------------------
String::String(const String& aStr)
: mBuf(0)
, mCap(0)
{
    reset(aStr);
}

//------------------------------------------------------------------------------
String::~String()
{
    reset();
}

//------------------------------------------------------------------------------
void String::clear()
{
    if (mBuf) {
        SYS_ASSERT(mCap > 0);
        mBuf[0] = 0;
    }
}

//------------------------------------------------------------------------------
void String::reset()
{
    if (mBuf) {
        delete[] mBuf;
        mBuf = 0;
        mCap = 0;
    }
}

//------------------------------------------------------------------------------
void String::reset(int aCap)
{
    reset();
    SYS_ASSERT(0 < aCap);
    mCap = aCap;
    mBuf = new char[mCap];
    ::std::memset(mBuf, 0, mCap);
}

//------------------------------------------------------------------------------
void String::reset(const char* aStr)
{
    int cap = static_cast<int>(::std::strlen(aStr) + 1);
    reset(cap);
    set(aStr);
}

//------------------------------------------------------------------------------
void String::reset(int aCap, const char* aStr)
{
    reset(aCap);
    set(aStr);
}

//------------------------------------------------------------------------------
void String::reset(const String& aStr)
{
    reset();
    if (aStr.isValid()) {
        mCap = aStr.mCap;
        mBuf = new char[mCap];
        ::std::memcpy(mBuf, aStr.buffer(), mCap);
    }
}

//------------------------------------------------------------------------------
bool String::isEmpty() const
{
    return ((mCap == 0) || (mBuf[0] == 0));
}

//------------------------------------------------------------------------------
bool String::operator==(const char* aStr) const
{
    return equals(aStr, false);
}

//------------------------------------------------------------------------------
bool String::equals(const char* aStr, bool aWillIgnoresCase) const
{
    SYS_ASSERT_POINTER(aStr);
    if (!isValid()) {
        return false;
    }
    int len = getLength();
    int strLen = CString::GetLength(aStr, len + 1); // len + 1 以上は調べる必要がない
    if (len != strLen) {
        return false;
    }

    for (int i = 0; i < strLen; ++i) {
        if (!Equals(mBuf[i], aStr[i], aWillIgnoresCase)) {
            return false;
        }
    }
    return(true);
}

//------------------------------------------------------------------------------
bool String::startsWith(const char* aStr, bool aWillIgnoresCase) const
{
    if (!isValid()) {
        return false;
    }
    int len = getLength();
    int strLen = CString::GetLength(aStr, len + 1); // len + 1 以上は調べる必要がない
    if (strLen > len) {
        return false;
    }
    return getSubstringByLength(0, strLen).equals(aStr, aWillIgnoresCase);
}

//------------------------------------------------------------------------------
bool String::endsWith(const char* aStr, bool aWillIgnoresCase) const
{
    if (!isValid()) {
        return false;
    }
    int len = getLength();
    int strLen = CString::GetLength(aStr, len + 1); // len + 1 以上は調べる必要がない
    if (strLen > len) {
        return false;
    }
    return getSubstringByLength(len - strLen, strLen).equals(aStr, aWillIgnoresCase);
}

//------------------------------------------------------------------------------
int String::getIndexOf(const char* aStr, int aStartIdx, bool aWillIgnoresCase) const
{
    if (!isValid()) {
        return -1;
    }
    SYS_ASSERT(0 <= aStartIdx);

    const int strlen = getLength();
    const int patternLen = CString::GetLength(aStr, strlen + 1);
    if (strlen == 0 || patternLen == 0) {
        return -1;
    }

    const String substr = getSubstring(aStartIdx);
    // 必要ならば大文字を小文字へ変換
    const String targetStr  = aWillIgnoresCase ? substr.toLowerCase() : substr;
    const String patternstr = aWillIgnoresCase ? String(aStr).toLowerCase() : String(aStr);

    const char* target = targetStr.buffer();
    const char* pattern = patternstr.buffer();
    // パターンマッチした位置を取得する
    const char* matchedAddr = ::std::strstr(target, pattern);
    if (matchedAddr) {
        const uint64_t matchedAddrOffset = uintptr_t(matchedAddr) - uintptr_t(target);
        const int matchedIndex = static_cast<int32_t>(matchedAddrOffset / sizeof(char));
        return aStartIdx + matchedIndex;
    } else {
        // 見つからなかった
        return -1;
    }
}

//------------------------------------------------------------------------------
int String::getIndexOf(char aChar, int aStartIdx, bool aWillIgnoresCase) const
{
    if (!isValid()) {
        return -1;
    }
    SYS_ASSERT(0 <= aStartIdx);

    const int strlen = getLength();
    if (strlen == 0 || strlen <= aStartIdx) {
        return -1;
    }

    const String substr = getSubstring(aStartIdx);
    // 必要ならば大文字を小文字へ変換
    const String targetStr  = aWillIgnoresCase ? substr.toLowerCase() : substr;
    const char patternChar = static_cast<const char>(aWillIgnoresCase ? ::std::tolower(aChar) : aChar);

    const char* target = targetStr.buffer();
    int pattern = static_cast<int>(patternChar);
    // パターンマッチした位置を取得する
    const char* matchedAddr = ::std::strchr(target, pattern);
    if (matchedAddr) {
        const uint64_t matchedAddrOffset = uintptr_t(matchedAddr) - uintptr_t(target);
        const int matchedIndex = static_cast<int32_t>(matchedAddrOffset / sizeof(char));
        return aStartIdx + matchedIndex;
    }
    else {
        // 見つからなかった
        return -1;
    }
}

//------------------------------------------------------------------------------
int String::getLastIndexOf(const char* aStr, int aStartIdx, bool aWillIgnoresCase) const
{
    if (!isValid()) {
        return -1;
    }
    SYS_ASSERT(-1 <= aStartIdx);
    const int strLen = getLength();
    const int patternStrLen = CString::GetLength(aStr, strLen + 1);
    if (strLen == 0 || patternStrLen == 0) {
        return -1;
    }
    // 「aStartIdx == -1」が指定された際は、文字列全体を対象にする
    int startIdx = (aStartIdx == -1) ? strLen : aStartIdx;
    const String substr = getSubstringByIndex(0, startIdx);
    // 必要ならば大文字を小文字へ変換
    const String targetStr  = aWillIgnoresCase ? substr.toLowerCase() : substr;
    int targetStrLen = startIdx;
    const char* targetStrBuf = targetStr.buffer();
    const String patternStr = aWillIgnoresCase ? String(aStr).toLowerCase() : String(aStr);
    const char* patternStrBuf = patternStr.buffer();

    for (int i = targetStrLen - patternStrLen; i >= 0; --i) {
        if (memcmp(&targetStrBuf[i], patternStrBuf, patternStrLen) == 0) {
            return i;
        }
    }
    return -1;
}

//------------------------------------------------------------------------------
String String::getSubstringByLength(int aStartIdx, int aLen) const
{
    SYS_ASSERT(isValid());
    SYS_ASSERT_MSG(0 <= aLen, "aLen: %d\n", aLen);
    SYS_ASSERT_MSG(aStartIdx + aLen <= getLength(),
        "aStartIdx:%d, aLen:%d, getLength():%d\n",
        aStartIdx, aLen, getLength()
        );
    String str(aLen + 1);
    StringUtil::GetSubstringByLength(str.mBuf, mBuf, aStartIdx, aLen);
    return str;
}

//------------------------------------------------------------------------------
void String::set(const String& aStr)
{
    // 空Stringから空Stringへのset呼び出しは代入済み扱いにする
    if (!isValid() && !aStr.isValid()) {
        return;
    }

    SYS_ASSERT(isValid());
    if (aStr.isValid()) {
        const auto len = aStr.getLength();
        SYS_ASSERT_LESS(len, mCap);
        ::std::memcpy(mBuf, aStr.mBuf, len);
        mBuf[len] = 0;
    } else {
        clear();
    }
}

//------------------------------------------------------------------------------
String& String::operator=(const String& aStr)
{
    set(aStr);
    return *this;
}

//------------------------------------------------------------------------------
void String::setAsPossible(const String& aStr)
{
    SYS_ASSERT(isValid());
    SYS_ASSERT(aStr.isValid());
    int minCap = Math::Min(mCap, aStr.mCap);
    ::std::memcpy(mBuf, aStr.buffer(), minCap);
    if (minCap > 0) {
        mBuf[minCap - 1] = 0;
    }
}

//------------------------------------------------------------------------------
void String::set(const char* aStr)
{
    SYS_ASSERT_POINTER(aStr);
    int reqCap = CString::GetLength(aStr, mCap) + 1;
    SYS_ASSERT(reqCap <= mCap);
    ::std::memcpy(mBuf, aStr, reqCap);
}

//------------------------------------------------------------------------------
String& String::operator=(const char* aStr)
{
    set(aStr);
    return *this;
}

//------------------------------------------------------------------------------
void String::setAsPossible(const char* aStr)
{
    SYS_ASSERT_POINTER(aStr);
    int reqCap = CString::GetLength(aStr, mCap) + 1;
    int minCap = Math::Min(reqCap, mCap);
    ::std::memcpy(mBuf, aStr, minCap);
    if (minCap > 0) {
        mBuf[minCap - 1] = 0;
    }
}

//------------------------------------------------------------------------------
void String::setFromUtf16(const char16_t* aStr)
{
    // 最悪値でアサート
    SYS_ASSERT(CString::GetMaxUtf8CapacityFromUtf16Length(U16CString::GetLength(aStr)) <= mCap);
    setFromUtf16AsPossible(aStr);
}

//------------------------------------------------------------------------------
void String::setFromUtf16(const U16String& aStr)
{
    // 最悪値でアサート
    SYS_ASSERT(CString::GetMaxUtf8CapacityFromUtf16Length(aStr.capacity() - 1) <= mCap);
    setFromUtf16AsPossible(aStr);
}

//------------------------------------------------------------------------------
void String::setFromUtf16AsPossible(const char16_t* aStr)
{
    SYS_ASSERT_POINTER(aStr);
    SYS_ASSERT_POINTER(mBuf);
    StringConvert::ConvertUtf16ToUtf8(mBuf, capacity(), aStr);
}

//------------------------------------------------------------------------------
void String::setFromUtf16AsPossible(const U16String& aStr)
{
    setFromUtf16AsPossible(aStr.buffer());
}

//------------------------------------------------------------------------------
void String::limit(int aLen)
{
    SYS_ASSERT(aLen < mCap);
    mBuf[aLen] = 0;
}

//------------------------------------------------------------------------------
void String::replaceAll(const char* aRemoveStr, const char* aInsertStr)
{
    const int len = getLength();
    const int removeLen = CString::GetLength(aRemoveStr, len + 1); // len + 1以上は調べる必要がない
    const String insertString(aInsertStr);

    // aRemoveStrが空なら、何もせずに抜ける
    if (removeLen <= 0) {
        return;
    }

    // 逆順チェック(入れ替えが発生してもシンプル)
    for (int i = len - removeLen; 0 <= i; --i) {
        String tmp(removeLen + 1);
        tmp.setAsPossible(&mBuf[i]);

        if (::std::strcmp(aRemoveStr, tmp.buffer()) == 0) {
            // 一致した。差し替える
            String headString(i + 1);
            headString.setAsPossible(mBuf);
            String tailString(getLength() - i - removeLen + 1);
            tailString.setAsPossible(&mBuf[i + removeLen]);

            *this = headString + insertString + tailString;
            //
            i -= removeLen - 1;
        }
    }
}

//------------------------------------------------------------------------------
void String::replaceWord(char aRemoveWord, char aInsertWord)
{
    const int len = getLength();
    for (int i = 0; i < len; ++i) {
        if (mBuf[i] == aRemoveWord) {
            // aRemoveWord を aInsertWord に変換
            mBuf[i] = aInsertWord;
        }
    }
}

//------------------------------------------------------------------------------
String String::toLowerCase() const
{
    String lowerStr(mBuf);
    const int len = lowerStr.getLength();
    for (int i = 0; i < len; ++i) {
        uint8_t c = mBuf[i];
        if ((c & 0x80) == 0) {
            lowerStr.at(i) = static_cast<char>(::std::tolower(c));
        }
    }
    return lowerStr;
}

//------------------------------------------------------------------------------
bool String::isCorrectUtf8() const
{
    int charRestByteCount = 0;
    for (int i = 0; i < mCap; ++i) {
        uint8_t c = mBuf[i];
        if (c == 0) {
            // 終端
            {
                // 継続バイトが不完全に残っている場合は正しくない
                if (charRestByteCount > 0) {
                    return false;
                }
            }
            break;
        } else if (c & 0x80) {
            // 開始バイト、もしくは、継続バイト
            {
                if (c & 0x40) {
                    // 開始バイト「11xxxxxx」
                    {
                        if ((c & 0x20) == 0) {
                            // 2バイト文字「110xxxxx」
                            charRestByteCount = 1;
                        } else if ((c & 0x10) == 0) {
                            // 3バイト文字「1110xxxx」
                            charRestByteCount = 2;
                        } else if ((c & 0x08) == 0) {
                            // 4バイト文字「11110xxx」
                            charRestByteCount = 3;
                        } else if ((c & 0x04) == 0) {
                            // 5バイト文字「111110xx」
                            charRestByteCount = 4;
                        } else if ((c & 0x02) == 0) {
                            // 6バイト文字「1111110x」
                            charRestByteCount = 5;
                        } else {
                            return false;
                        }
                    }
                } else {
                    // 継続バイト「10xxxxxx」
                    {
                        charRestByteCount--;
                        if (charRestByteCount < 0) {
                            return false;
                        }
                    }
                }
            }
        } else {
            // 1バイト文字
            {
                if (charRestByteCount != 0) {
                    return false;
                }
            }
        }
    }
    return true;
}

//------------------------------------------------------------------------------
void String::correctTailAsUtf8IfPossible()
{
    int charStartIdx = -1;
    int charRestByteCount = 0;
    for (int i = 0; i < mCap; ++i) {
        uint8_t c = mBuf[i];
        if (c == 0) {
            // 終端
            {
                // 継続バイトが不完全に残っていたら開始バイトまで消す
                if (charRestByteCount > 0) {
                    SYS_ASSERT(charStartIdx != -1);
                    mBuf[charStartIdx] = 0;
                }
                break;
            }
        } else if (c & 0x80) {
            // 開始バイト、もしくは、継続バイト
            {
                // 開始バイト「11xxxxxx」
                if (c & 0x40) {
                    charStartIdx = i;
                    if ((c & 0x20) == 0) {
                        // 2バイト文字「110xxxxx」
                        charRestByteCount = 1;
                    } else if ((c & 0x10) == 0) {
                        // 3バイト文字「1110xxxx」
                        charRestByteCount = 2;
                    } else if ((c & 0x08) == 0) {
                        // 4バイト文字「11110xxx」
                        charRestByteCount = 3;
                    } else if ((c & 0x04) == 0) {
                        // 5バイト文字「111110xx」
                        charRestByteCount = 4;
                    } else if ((c & 0x02) == 0) {
                        // 6バイト文字「1111110x」
                        charRestByteCount = 5;
                    } else {
                        return;
                    }
                } else {
                    // 継続バイト「10xxxxxx」
                    {
                        charRestByteCount--;
                        if (charRestByteCount < 0) {
                            return;
                        }
                    }
                }
            }
        } else {
            // 1バイト文字
            {
                if (charRestByteCount != 0) {
                    return;
                }
            }
        }
    }
}

//------------------------------------------------------------------------------
int String::getLength() const
{
    SYS_ASSERT_POINTER(mBuf);
    return CString::GetLength(mBuf, mCap);
}

//------------------------------------------------------------------------------
bool String::contains(char aCh) const
{
    for (int i = 0; i < mCap; ++i) {
        if (mBuf[i] == 0) {
            break;
        }
        if (mBuf[i] == aCh) {
            return true;
        }
    }
    return false;
}

//------------------------------------------------------------------------------
bool String::contains(const char* aStr) const
{
    return (getIndexOf(aStr) != -1);
}

//------------------------------------------------------------------------------
bool String::containsOnlyAscii() const {
    for (int i = 0; i < mCap; ++i) {
        if (mBuf[i] == 0) {
            break;
        }
        // UTF-8のマルチバイト文字（⇔ASCII文字）は0x80～0xffで構成されている
        if ((mBuf[i] & 0x80) != 0) {
            return false;
        }
    }
    return true;
}

//------------------------------------------------------------------------------
String& String::operator+=(const String& aStr)
{
    SYS_ASSERT(getLength() + aStr.getLength() < mCap);
    ::std::strcat(mBuf, aStr.mBuf);
    return *this;
}

//------------------------------------------------------------------------------
String String::operator+(const String& aStr) const
{
    String string(capacity() + aStr.capacity());
    string.set(*this);
    string += aStr;
    return string;
}

//------------------------------------------------------------------------------
void String::format(const char* aStr, ...)
{
    ::std::va_list vaList = { 0 };
    va_start(vaList, aStr);
    formatV(aStr, vaList);
    va_end(vaList);
} //lint !e438

//------------------------------------------------------------------------------
// 「format」をオーバーロードすると事故が起きるので「formatV」にしている
// ・オーバーロードではVC2013で、「%s」を1つ指定する時に「va_list」版が呼ばれてしまう問題が起きる
//   ・VC2013では「va_list」が「char*」の「typedef」になっている為
void String::formatV(const char* aStr, va_list aVaList)
{
    CString::FormatV(mBuf, mCap, aStr, aVaList);
}

//------------------------------------------------------------------------------
void String::formatAsPossible(const char* aStr, ...)
{
    ::std::va_list vaList = { 0 };
    va_start(vaList, aStr);
    formatVAsPossible(aStr, vaList);
    va_end(vaList);
} //lint !e438

//------------------------------------------------------------------------------
void String::formatVAsPossible(const char* aStr, va_list aVaList)
{
    CString::FormatVAsPossible(mBuf, mCap, aStr, aVaList);
}

//------------------------------------------------------------------------------
void String::strip()
{
    const char space(' ');
    const int length = getLength();
    // 先頭からの空白の数
    int startIdx = 0;
    {
        for (int i=0 ; i < length ; ++i) {
            if (space == at(i)) {
                ++startIdx;
            } else {
                break;
            }
        }
    }
    // strip後の文字長さ
    int strippedLength = length - startIdx;

    // 末尾からの空白の数
    {
        for (int i=length-1 ; i > startIdx ; --i) {
            if (space == at(i)) {
                --strippedLength;
            } else {
                break;
            }
        }
    }
    if (strippedLength < length) {
        SYS_ASSERT(0 <= strippedLength && strippedLength < length);
        setAsPossible(getSubstringByLength(startIdx, strippedLength));
    }
}

} // namespace
// EOF
