﻿// 文字コード:UTF-8
/// @file
#pragma once

#include "lib/DefaultDeleter.hpp"
#include "lib/LinkedPtrKind.hpp"

//------------------------------------------------------------------------------
namespace lib{

/// リンク方式のスマートポインタ、管理領域が別管理されないため断片化しない
template<class TType, class TDeleter = DefaultDeleter<TType>>
class LinkedPtr
{
public:
    typedef LinkedPtr<TType, TDeleter> This;
    typedef LinkedPtrKind::EnumType Kind;
    /// 空の状態で生成
    LinkedPtr() : mPtr(0), mKind(LinkedPtrKind::Invalid) , mNext(0) {}
    /// 生ポインタから生成
    explicit LinkedPtr(TType* aPtr, Kind aKind);
    /// 生ポインタから生成
    explicit LinkedPtr(TType* aPtr);
    /// リンクドポインタから生成
    LinkedPtr(const This& aPtr);
    /// リンクドポインタから生成
    LinkedPtr(const This& aPtr, Kind aKind);
    /// 破棄、インスタンスをロックしているリンクドポインタが1つも無ければインスタンスがデリートされる
    ~LinkedPtr() { reset(); }
    /// 有効なポインタか？
    bool isValid() const { return mPtr != nullptr; }
    /// 種類を取得
    Kind kind() const { return mKind; }
    /// 生ポインタの取得
    TType* get() { return mPtr; }
    /// 生ポインタの取得
    const TType* get() const { return mPtr; }
    /// 生ポインタから設定
    void reset(TType* aPtr, Kind aKind = LinkedPtrKind::Strong);
    /// リンクドポインタから設定
    void reset(const This& aPtr, Kind aKind);
    /// リンクドポインタから設定
    void reset(const This& aPtr);
    /// リンクドポインタから設定
    This& operator=(const This& aPtr) { reset(aPtr); return *this; }
    /// 破棄、インスタンスをロックしているリンクドポインタが1つも無ければインスタンスがデリートされる
    void reset();
    /// ポインタの管理を譲る
    TType& pass();
    /// インスタンスを取得
    TType& operator*() { SYS_ASSERT_POINTER(mPtr); return *mPtr; }
    /// 前項のコンスト版
    const TType& operator*() const { SYS_ASSERT_POINTER(mPtr); return *mPtr; }
    /// インスタンスを使用
    TType* operator->() { SYS_ASSERT_POINTER(mPtr); return mPtr; }
    /// 前項のコンスト版
    const TType* operator->() const { SYS_ASSERT_POINTER(mPtr); return mPtr; }
private:
    static void(*DeleteFunc_)(TType*);
    TType* mPtr;
    Kind mKind;
    This* mNext;
};

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
LinkedPtr<TType, TDeleter>::LinkedPtr(TType* aPtr, Kind aKind)
: mPtr(0)
, mKind(LinkedPtrKind::Invalid)
, mNext(0)
{
    reset(aPtr, aKind);
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
LinkedPtr<TType, TDeleter>::LinkedPtr(TType* aPtr)
: mPtr(0)
, mKind(LinkedPtrKind::Invalid)
, mNext(0)
{
    reset(aPtr, LinkedPtrKind::Strong);
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
LinkedPtr<TType, TDeleter>::LinkedPtr(const This& aPtr)
: mPtr(0)
, mKind(LinkedPtrKind::Invalid)
, mNext(0)
{
    reset(aPtr);
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
LinkedPtr<TType, TDeleter>::LinkedPtr(const This& aPtr, Kind aKind)
: mPtr(0)
, mKind(LinkedPtrKind::Invalid)
, mNext(0)
{
    reset(aPtr, aPtr.kind());
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
void LinkedPtr<TType, TDeleter>::reset(TType* aPtr, Kind aKind)
{
    DeleteFunc_ = TDeleter::Delete;
    reset();
    mPtr = aPtr;
    // 生ポインタからは弱参照リンクドポインタは生成できない
    SYS_ASSERT(aKind != LinkedPtrKind::Weak);
    mKind = aKind;
    mNext = this;
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
void LinkedPtr<TType, TDeleter>::reset(const This& aPtr, Kind aKind)
{
    reset();
    if (aPtr.mPtr) {
        mPtr = aPtr.mPtr;
        // リンクドポインタからは自分自身リンクドポインタは生成できない
        SYS_ASSERT(aKind != LinkedPtrKind::Self);
        // 自身リンクドポインタからは弱参照リンクドポインタしか生成できない
        SYS_ASSERT(aPtr.kind() != LinkedPtrKind::Self || aKind == LinkedPtrKind::Weak);
        mKind = aKind;
        mNext = aPtr.mNext;
        // 代入元はコンストだが性質上どうしてもリンクを変更する必要がある
        const_cast<This&>(aPtr).mNext = this;
    }
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
void LinkedPtr<TType, TDeleter>::reset(const This& aPtr)
{
    Kind kind = aPtr.kind();
    // 自分自身を指すリンクドポインタからは弱参照しか生成できない
    if (aPtr.kind() == LinkedPtrKind::Self) {
        kind = LinkedPtrKind::Weak;
    }
    reset(aPtr, kind);
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
void LinkedPtr<TType, TDeleter>::reset()
{
    if (!mPtr) {
        return;
    }
    bool hasOtherStrongLinkedPtr = false;
    // 自分以外に強参照(自身リンクドポインタも含む)のリンクドポインタが存在するか？
    for (This* lp = mNext; lp != this; lp = lp->mNext) {
        if (lp->kind() == LinkedPtrKind::Strong || lp->kind() == LinkedPtrKind::Self) {
            hasOtherStrongLinkedPtr = true;
            break;
        }
    }
    // 自身リンクドポインタの場合には、他に強参照リンクドポインタが生成されていてはいけない
    SYS_ASSERT(mKind != LinkedPtrKind::Self || !hasOtherStrongLinkedPtr);
    if (hasOtherStrongLinkedPtr) {
        // 強参照が残っている場合は、インスタンスの削除は行わない
        {
            // 自分だけをリンクから外し
            {
                This* front = mNext;
                SYS_ASSERT_POINTER(front);
                while (front->mNext != this) {
                    front = front->mNext;
                }
                front->mNext = mNext;
            }
            // 自分だけ情報を消去
            {
                mPtr = 0;
                mKind = LinkedPtrKind::Invalid;
                mNext = 0;
            }
        }
    } else {
        // 他に強参照が無ければ、インスタンスの削除実行
        {
            // 弱参照の破棄でインスタンスの削除が実行されることはない
            SYS_ASSERT(mKind != LinkedPtrKind::Weak);
            // 自分自身の場合は他で削除が実行されるので、削除しない
            if (mKind != LinkedPtrKind::Self) {
                // 対象クラスがデストラクト時に不完全型であっても、デストラクタが走るようにする
                DeleteFunc_(mPtr);
            }
            // 自身と、同一リンクの全リンクドポインタの情報を消去
            for (This* lp = this; lp;) {
                This* n = lp->mNext;
                lp->mPtr = 0;
                lp->mKind = LinkedPtrKind::Invalid;
                lp->mNext = 0;
                lp = n;
            }
        }
    }
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
TType& LinkedPtr<TType, TDeleter>::pass()
{
    SYS_ASSERT_POINTER(mPtr);
    // 実装が不完全、弱参照があった場合に弱参照を破棄して続行することができない)
    {}
    // 自分以外に監視者がいない事を確認
    SYS_ASSERT(mNext == this);
    TType* ptr = mPtr;
    mPtr = 0;
    mKind = LinkedPtrKind::Invalid;
    mNext = 0;
    return(*ptr);
}

//------------------------------------------------------------------------------
template<class TType, class TDeleter>
void (*LinkedPtr<TType, TDeleter>::DeleteFunc_)(TType*) = 0;

} // namespace
// EOF
