﻿// 文字コード:UTF-8
/// @file
#include "lib/IntRect.hpp"

#include "lib/Math.hpp"
#include "lib/Rect.hpp"

//------------------------------------------------------------------------------
namespace lib {

//------------------------------------------------------------------------------
IntRect::IntRect()
: mPosition()
, mSize()
{
}

//------------------------------------------------------------------------------
IntRect::IntRect(const ::lib::Vector2i& aPosition, const ::lib::Vector2i& aSize)
: mPosition(aPosition)
, mSize(aSize)
{
}

//------------------------------------------------------------------------------
IntRect::IntRect(int aPosX, int aPosY, int aWidth, int aHeight)
: mPosition(aPosX, aPosY)
, mSize(aWidth, aHeight)
{
}

//------------------------------------------------------------------------------
bool IntRect::operator==(const IntRect& aRect) const
{
    return mPosition == aRect.position() && mSize == aRect.size();
}

//------------------------------------------------------------------------------
void IntRect::set(const ::lib::Vector2i& aPosition, const ::lib::Vector2i& aSize)
{
    mPosition = aPosition;
    mSize = aSize;
}

//------------------------------------------------------------------------------
Rect IntRect::toRect() const
{
    return Rect(mPosition.toVector2(), mSize.toVector2());
}

//------------------------------------------------------------------------------
int IntRect::getManhattanDistance(const IntRect& aRect) const
{
    auto getMinDistance = [] (int aPos0, int aSize0, int aPos1, int aSize1) {
        int d0 = aPos1 - (aPos0 + aSize0);
        int d1 = (aPos1 + aSize1) - aPos0;
        if (d0 * d1 < 0) {
            return 0;
        } else {
            return ::lib::Math::Min(::lib::Math::Abs(d0), ::lib::Math::Abs(d1));
        }
    };
    return
        getMinDistance(mPosition.x, mSize.x, aRect.position().x, aRect.size().x) +
        getMinDistance(mPosition.y, mSize.y, aRect.position().y, aRect.size().y);
}

//------------------------------------------------------------------------------
bool IntRect::isEmpty() const
{
    return (size().x <= 0) || (size().y <= 0);
}

//------------------------------------------------------------------------------
int IntRect::getArea() const
{
    return mSize.x * mSize.y;
}

//------------------------------------------------------------------------------
bool IntRect::contains(const ::lib::Vector2i& aPos) const
{
    return
        position().x <= aPos.x &&
        position().x + size().x > aPos.x &&
        position().y <= aPos.y &&
        position().y + size().y >= aPos.y;
}

//------------------------------------------------------------------------------
bool IntRect::contains(const IntRect& aRect) const
{
    return
        position().x <= aRect.position().x &&
        position().x + size().x >= aRect.position().x + aRect.size().x &&
        position().y <= aRect.position().y &&
        position().y + size().y >= aRect.position().y + aRect.size().y;
}

//------------------------------------------------------------------------------
bool IntRect::intersects(const IntRect& aRect) const
{
    return
        position().x + size().x > aRect.position().x &&
        position().x < aRect.position().x + aRect.size().x &&
        position().y + size().y > aRect.position().y &&
        position().y < aRect.position().y + aRect.size().y;
}

//------------------------------------------------------------------------------
bool IntRect::intersectsWithBound(const IntRect& aRect) const
{
    return
        position().x + size().x >= aRect.position().x &&
        position().x <= aRect.position().x + aRect.size().x &&
        position().y + size().y >= aRect.position().y &&
        position().y <= aRect.position().y + aRect.size().y;
}

//------------------------------------------------------------------------------
void IntRect::intersect(const IntRect& aRect)
{
    if (aRect.position().x > position().x) {
        size().x -= aRect.position().x - position().x;
        position().x = aRect.position().x;
    }
    if (aRect.position().y > position().y) {
        size().y -= aRect.position().y - position().y;
        position().y = aRect.position().y;
    }
    if (aRect.position().x + aRect.size().x < position().x + size().x) {
        size().x -= (position().x + size().x) - (aRect.position().x + aRect.size().x);
    }
    if (aRect.position().y + aRect.size().y < position().y + size().y) {
        size().y -= (position().y + size().y) - (aRect.position().y + aRect.size().y);
    }
}

//------------------------------------------------------------------------------
IntRect IntRect::getIntersected(const IntRect& aRect) const
{
    IntRect r = *this;
    r.intersect(aRect);
    return r;
}

//------------------------------------------------------------------------------
void IntRect::unite(const IntRect& aRect)
{
    if (aRect.isEmpty()) {
        return;
    }
    if (isEmpty()) {
        *this = aRect;
        return;
    }
    if (aRect.position().x < position().x) {
        size().x += position().x - aRect.position().x;
        position().x = aRect.position().x;
    }
    if (aRect.position().y < position().y) {
        size().y += position().y - aRect.position().y;
        position().y = aRect.position().y;
    }
    if (aRect.position().x + aRect.size().x > position().x + size().x) {
        size().x += (aRect.position().x + aRect.size().x) - (position().x + size().x);
    }
    if (aRect.position().y + aRect.size().y > position().y + size().y) {
        size().y += (aRect.position().y + aRect.size().y) - (position().y + size().y);
    }
}

//------------------------------------------------------------------------------
IntRect IntRect::getUnited(const IntRect& aRect) const
{
    IntRect r = *this;
    r.unite(aRect);
    return r;
}

//------------------------------------------------------------------------------
void IntRect::clamp(const IntRect& aRect)
{
    ::lib::Vector2i maxPos = aRect.getOppositePosition() - size();
    position().x = ::lib::Math::LimitMinMax(position().x, aRect.position().x, maxPos.x);
    position().y = ::lib::Math::LimitMinMax(position().y, aRect.position().y, maxPos.y);
}

//------------------------------------------------------------------------------
IntRect IntRect::getClamped(const IntRect& aRect) const
{
    IntRect r = *this;
    r.clamp(aRect);
    return r;
}

//------------------------------------------------------------------------------
void IntRect::inflate(int aLen)
{
    inflate(::lib::Vector2i(aLen, aLen));
}

//------------------------------------------------------------------------------
IntRect IntRect::getInflated(int aLen) const
{
    IntRect r = *this;
    r.inflate(aLen);
    return r;
}

//------------------------------------------------------------------------------
void IntRect::inflate(const ::lib::Vector2i& aSize)
{
    position().x -= aSize.x;
    position().y -= aSize.y;
    size().x += 2 * aSize.x;
    size().y += 2 * aSize.y;
}

//------------------------------------------------------------------------------
IntRect IntRect::getInflated(const ::lib::Vector2i& aSize) const
{
    IntRect r = *this;
    r.inflate(aSize);
    return r;
}

//------------------------------------------------------------------------------
::lib::Vector2i IntRect::getOppositePosition() const
{
    return ::lib::Vector2i(mPosition + mSize);
}

//------------------------------------------------------------------------------
::lib::Vector2 IntRect::getCenterPosition() const
{
    return ::lib::Vector2(
        position().x + 0.5f * size().x,
        position().y + 0.5f * size().y);
}

//------------------------------------------------------------------------------
::lib::Vector2i IntRect::getNearestPosition(const ::lib::Vector2i& aPos) const
{
    SYS_ASSERT(!isEmpty());
    // 最大点は含まないようにしている点に注意
    // ・整数の矩形ではこれが一般的な為
    return ::lib::Vector2i(
        Math::LimitMinMax(aPos.x, mPosition.x, mPosition.x + mSize.x - 1),
        Math::LimitMinMax(aPos.y, mPosition.y, mPosition.y + mSize.y - 1)
        );
}

#if DEBUG_IS_ENABLED
//------------------------------------------------------------------------------
String IntRect::toString() const
{
    return String::FromFormat(128, "((%d, %d), (%d, %d))",
        position().x, position().y,
        size().x, size().y);
}

#endif

} // namespace
// EOF
