﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/systm.h>
#include <sys/queue.h>
#include <sys/lock.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

SLIST_HEAD(sq_slist, sq_entry);

typedef struct sq_entry {
    SLIST_ENTRY(sq_entry) tqe;
    nn::os::ConditionVariableType event;
    nn::os::MutexType* mutex;
    int wokenup;
    void* chan;
} sq_entry_t;

typedef struct sq_list {
    struct sq_slist   sq_queue;
    nn::os::MutexType sq_mtx;
} sq_queue_t;

#define SC_TABLESIZE      256
#define SC_ENTRIES        128
#define SC_MASK           (SC_TABLESIZE - 1)
#define SC_SHIFT          8
#define SC_HASH(wc)       ((((uintptr_t)(wc) >> SC_SHIFT) ^ (uintptr_t)(wc)) & SC_MASK)

static  struct sq_slist   sq_free;
static  sq_queue_t        sq_queues[SC_TABLESIZE];
static  sq_entry_t        sq_entries[SC_ENTRIES];
static  nn::os::MutexType sq_free_mtx;

int _sleep(
    void *chan, struct lock_object *lock, int priority,
    const char *wmesg, sbintime_t sbt, sbintime_t pr, int flags)
{
    int rval = 0;
    sq_queue_t* sq = &sq_queues[SC_HASH(chan)];
    sq_entry_t* entry;
    nn::os::MutexType dummy_mutex;

    nn::os::LockMutex(&sq_free_mtx);
    entry = SLIST_PULL_HEAD(&sq_free, sq_entry, tqe);
    nn::os::UnlockMutex(&sq_free_mtx);

    NN_ABORT_UNLESS(entry != NULL);

    entry->chan    = chan;
    entry->wokenup = false;
    if(lock != NULL){
      entry->mutex = (nn::os::MutexType*)(((struct mtx*)lock)->siglo_ptr);
    }else{
      nn::os::InitializeMutex(&dummy_mutex, false, 0);
      nn::os::LockMutex(&dummy_mutex);
      entry->mutex = &dummy_mutex;
    }

    NN_ABORT_UNLESS(entry->mutex != NULL);

    nn::os::LockMutex(&sq->sq_mtx);
    SLIST_INSERT_HEAD(&sq->sq_queue, entry, tqe);
    nn::os::UnlockMutex(&sq->sq_mtx);

    if (sbt != 0) {
        nn::os::TimedWaitConditionVariable(
                    &entry->event,
                    entry->mutex,
                    nn::os::ConvertToTimeSpan(nn::os::Tick(sbintime_to_cputicks(sbt) * cputicks_per_hz))
        );
        rval = entry->wokenup ? 0 : EWOULDBLOCK;
    } else {
        nn::os::WaitConditionVariable(&entry->event, entry->mutex);
    }

    if(lock == NULL){
      nn::os::FinalizeMutex(&dummy_mutex);
    }

    nn::os::LockMutex(&sq->sq_mtx);
    SLIST_REMOVE(&sq->sq_queue, entry, sq_entry, tqe);
    nn::os::UnlockMutex(&sq->sq_mtx);

    nn::os::LockMutex(&sq_free_mtx);
    SLIST_INSERT_HEAD(&sq_free, entry, tqe);
    nn::os::UnlockMutex(&sq_free_mtx);

    return rval;
}

void wakeup(void* chan)
{
    sq_queue_t* sq = &sq_queues[SC_HASH(chan)];
    sq_entry_t* entry;
    nn::os::LockMutex(&sq->sq_mtx);
    SLIST_FOREACH(entry, &sq->sq_queue, tqe) {
        if (entry->chan == chan) {
            entry->wokenup = true;
            nn::os::BroadcastConditionVariable(&entry->event);
        }
    }
    nn::os::UnlockMutex(&sq->sq_mtx);
    return;
}

void wakeup_one(void* chan)
{
    sq_queue_t* sq = &sq_queues[SC_HASH(chan)];
    sq_entry_t* entry;
    nn::os::LockMutex(&sq->sq_mtx);
    SLIST_FOREACH(entry, &sq->sq_queue, tqe) {
        if (entry->chan == chan) {
            entry->wokenup = true;
            nn::os::SignalConditionVariable(&entry->event);
        }
    }
    nn::os::UnlockMutex(&sq->sq_mtx);
    return;
}

static void sleep_sysinit(void *dummy)
{
    int i;

    memset((void*)sq_queues, 0x0, sizeof(sq_queues));
    memset((void*)sq_entries, 0x0, sizeof(sq_entries));

    /* allocate SC_TABLESIZE queues, each with it's own mutex */
    for (i = 0; i < SC_TABLESIZE; i++) {
        SLIST_INIT(&sq_queues[i].sq_queue);
        nn::os::InitializeMutex(&sq_queues[i].sq_mtx, false, 0);
    }

    nn::os::InitializeMutex(&sq_free_mtx, false, 0);
    SLIST_INIT(&sq_free);

    /* allocate SC_ENTRIES channels, put on free queue */
    for (i = 0; i < SC_ENTRIES; i++) {
        nn::os::InitializeConditionVariable(&sq_entries[i].event);
        SLIST_INSERT_HEAD(&sq_free, &sq_entries[i], tqe);
    }
}

SYSINIT(sleep_sysinit, SI_SUB_PORTINIT, SI_ORDER_FIRST, sleep_sysinit, NULL);

}
