﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>

static MALLOC_DEFINE(M_MUTEX, "imutex", "Mutex");

static void lock_spin(struct lock_object *lock, uintptr_t how)
{
    mtx_lock((struct mtx *)lock);
}

static uintptr_t unlock_spin(struct lock_object *lock)
{
    mtx_unlock((struct mtx*)lock);
    return 0;
}

struct lock_class lock_class_mtx_sleep = {
    .lc_name     = "spin mutex",
    .lc_flags    = LC_SPINLOCK | LC_RECURSABLE,
    .lc_assert   = NULL,
    .lc_ddb_show = NULL,
    .lc_lock     = lock_spin,
    .lc_owner    = NULL,
    .lc_unlock   = unlock_spin
};

struct lock_class lock_class_mtx_spin = {
    .lc_name     = "spin mutex",
    .lc_flags    = LC_SPINLOCK | LC_RECURSABLE,
    .lc_assert   = NULL,
    .lc_ddb_show = NULL,
    .lc_lock     = lock_spin,
    .lc_owner    = NULL,
    .lc_unlock   = unlock_spin
};

void mtx_init_l(struct mtx *m, const char *name, const char *type, int opts)
{
    struct nn::os::MutexType* siglo_mutex;
    struct lock_class *l_class;
    siglo_mutex = (nn::os::MutexType*)malloc(sizeof(nn::os::MutexType), M_MUTEX, M_ZERO | M_WAITOK);
    NN_ABORT_UNLESS(siglo_mutex != NULL);
    nn::os::InitializeMutex(siglo_mutex, false, 0);
    m->siglo_ptr   = (void*)siglo_mutex;
    m->mtx_lock    = MTX_UNOWNED;
    m->mtx_recurse = 0;
    l_class = (opts & MTX_SPIN) ? &lock_class_mtx_spin : &lock_class_mtx_sleep;
    lock_init(&m->lock_object, l_class, name, type, 0);
}

void mtx_destroy_l(struct mtx *m)
{
    nn::os::MutexType* siglo_mutex = (nn::os::MutexType*)m->siglo_ptr;
    NN_ABORT_UNLESS(siglo_mutex != NULL);
    nn::os::FinalizeMutex(siglo_mutex);
    free(siglo_mutex, M_MUTEX);
}

void mtx_sysinit(void *arg)
{
    struct mtx_args *margs = (struct mtx_args*)arg;
    mtx_init((struct mtx*)margs->ma_mtx, margs->ma_desc, NULL, margs->ma_opts);
    return;
}

int  mtx_trylock_l(struct mtx *m)
{
    nn::os::MutexType* siglo_mutex = (nn::os::MutexType*)m->siglo_ptr;
    NN_ABORT_UNLESS(siglo_mutex != NULL);
    return nn::os::TryLockMutex(siglo_mutex);
}

void mtx_lock_l(struct mtx *m)
{
    nn::os::MutexType* siglo_mutex = (nn::os::MutexType*)m->siglo_ptr;
    NN_ABORT_UNLESS(siglo_mutex != NULL);
    LockMutex(siglo_mutex);
    return;
}

void mtx_lock_flags_l(struct mtx *m, int opt)
{
    mtx_lock(m);
    return;
}

void mtx_unlock_l(struct mtx *m)
{
    nn::os::MutexType* siglo_mutex = (nn::os::MutexType*)m->siglo_ptr;
    NN_ABORT_UNLESS(siglo_mutex != NULL);
    UnlockMutex(siglo_mutex);
    return;
}

void mtx_assert_l(struct mtx *m, int what)
{
    if (what & (MA_RECURSED | MA_NOTRECURSED))
    {
        panic("%s: MA_RECURSED and MA_NOTRECURSED not supported", __FUNCTION__);
    }

    switch (what)
    {
        case MA_OWNED:
            NN_ABORT_UNLESS(nn::os::IsMutexLockedByCurrentThread((nn::os::MutexType*)m->siglo_ptr));
            break;
        case MA_NOTOWNED:
            NN_ABORT_UNLESS(!nn::os::IsMutexLockedByCurrentThread((nn::os::MutexType*)m->siglo_ptr));
            break;
        default:
            panic("%s: Invalid 'what' option.", __FUNCTION__);
    }
    return;
}

int  mtx_owned_l(struct mtx *m)
{
    nn::os::MutexType* siglo_mutex = (nn::os::MutexType*)m->siglo_ptr;
    NN_ABORT_UNLESS(siglo_mutex != NULL);
    return (siglo_mutex->_ownerThread == nn::os::GetCurrentThread());
}

struct mtx Giant;
MTX_SYSINIT(mtx_Giant, &Giant, "mtx_Giant", MTX_DEF);

}
