﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

extern "C" {

#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/filedesc.h>
#include <sys/kernel.h>
#include <sys/socket.h>
#include <sys/malloc.h>
#include <siglo/idtab.h>
#include <sys/sysproto.h>
#include <siglo/client.h>

int maxfiles = maxfilesperproc;
static MALLOC_DEFINE(M_FILE, "ifile", "file descriptor allocations");

static int
badfo_readwrite(struct file *fp, struct uio *uio, struct ucred *active_cred,
    int flags, struct thread *td)
{
    return (EBADF);
}

static int
badfo_truncate(struct file *fp, off_t length, struct ucred *active_cred, struct thread *td)
{
    return (EINVAL);
}

static int
badfo_ioctl(struct file *fp, u_long com, void *data, struct ucred *active_cred, struct thread *td)
{
    return (EBADF);
}

static int
badfo_poll(struct file *fp, int events, struct ucred *active_cred, struct thread *td)
{
    return (0);
}

static int
badfo_kqfilter(struct file *fp, struct knote *kn)
{
    return (EBADF);
}

static int
badfo_stat(struct file *fp, struct stat *sb, struct ucred *active_cred, struct thread *td)
{
    return (EBADF);
}

static int
badfo_close(struct file *fp, struct thread *td)
{
    return (EBADF);
}

static int
badfo_chmod(struct file *fp, mode_t mode, struct ucred *active_cred, struct thread *td)
{
    return (EBADF);
}

static int
badfo_chown(struct file *fp, uid_t uid, gid_t gid, struct ucred *active_cred, struct thread *td)
{
    return (EBADF);
}

static int
badfo_sendfile(struct file *fp, int sockfd, struct uio *hdr_uio,
    struct uio *trl_uio, off_t offset, size_t nbytes, off_t *sent, int flags,
    int kflags, struct thread *td)
{
    return (EBADF);
}

struct fileops badfileops = {
    .fo_read     = badfo_readwrite,
    .fo_write    = badfo_readwrite,
    .fo_truncate = badfo_truncate,
    .fo_ioctl    = badfo_ioctl,
    .fo_poll     = badfo_poll,
    .fo_kqfilter = badfo_kqfilter,
    .fo_stat     = badfo_stat,
    .fo_close    = badfo_close,
    .fo_chmod    = badfo_chmod,
    .fo_chown    = badfo_chown,
    .fo_sendfile = badfo_sendfile,
};

int falloc(struct thread *td, struct file **resultfp, int *resultfd, int flags)
{
    // 1. Reserve file descriptor id
    // 2. Allocate file struct
    // 3. Initialize ref count on file struct
    // 4. Bump up ref counter, it'll keep close from destroying this file struct
    //    while we are operating on it. Ref will be dropped once this call
    //    returns to the user.
    int fd;
    int rval;
    struct file* fp;
    struct filedesc* fdp = curproc->p_fd;
    FILEDESC_XLOCK(fdp);
    if (!client_check_is_valid_from_cred(td->td_ucred))
    {
        rval = EINVAL;
    }
    else if ((fd = idalloc(fdp)) >= 0) {
        fp = (struct file*)malloc(sizeof(struct file), M_FILE, M_ZERO);
        if (fp == NULL) {
            idfree(fdp, fd);
            FILEDESC_XUNLOCK(fdp);
            return ENOMEM;
        }
        fp->f_ops  = &badfileops;
        fp->f_data = NULL;
        fp->f_type = 0;
        fp->f_flag = 0;
        fp->f_cred = td->td_ucred;
        fdp->fd_ofiles[fd].fde_file = fp;
        refcount_init(&(fp)->f_count, 1);
        *resultfp = fp;
        *resultfd = fd;
        fhold(fp);
        rval = 0;
    } else {
        rval = EMFILE;
    }
    FILEDESC_XUNLOCK(fdp);
    return rval;
}

void fdclose(struct filedesc *fdp, struct file *fp, int fd, struct thread *td)
{
    bool dropFd = false;

    FILEDESC_XLOCK(fdp);
    if (fdp->fd_ofiles[fd].fde_file == fp) {
        fdp->fd_ofiles[fd].fde_file = NULL;
        idfree(fdp, fd);
        dropFd = true;
    }
    FILEDESC_XUNLOCK(fdp);

    if(dropFd){
        fdrop(fp, td);
    }
}

void finit(struct file *fp, u_int flag, short type, void *data, struct fileops *ops)
{
    fp->f_data = data;
    fp->f_flag = flag;
    fp->f_type = type;
    fp->f_ops  = ops;
    return;
}

void fhold(struct file *fp)
{
    struct thread *td = curthread;
    refcount_acquire(&(fp)->f_count);
    if((td != NULL) && (fp->f_cred != td->td_ucred)){
        refcount_acquire(&fp->f_cred->cr_cross_proc_ref);
    }
}

int fdrop(struct file *fp, struct thread *td)
{
    int error = 0;

    if(fp->f_cred != td->td_ucred){
        refcount_release(&fp->f_cred->cr_cross_proc_ref);
    }

    if (refcount_release(&(fp)->f_count) == 1) {
        critical_enter();
        if (fp->f_ops != &badfileops) {
             struct file tmp_fp = *fp;
             fp->f_ops  = &badfileops;
             fp->f_data = NULL;
             fp->f_type = 0;
             fp->f_flag = 0;
             fp->f_cred = NULL;
             critical_exit();
             fo_close(&tmp_fp, td);
        } else {
             critical_exit();
        }

        // noone else is holding reference to this structure, ok to free.
        free(fp, M_FILE);
    }
    return error;
}

int fget(struct thread *td, int fd, cap_rights_t *rightsp, struct file **fpp)
{
    int error;
    struct file *fp;
    struct filedesc* fdp = curproc->p_fd;
    if (fd < 0 || fd > maxfilesperproc) {
        return EBADF;
    }
    FILEDESC_SLOCK(fdp);
    fp = fdp->fd_ofiles[fd].fde_file;
    if (fp != NULL) {
        if (fp->f_type == DTYPE_VNODE) {
            curthread->td_fpop = fp;
        }
        *fpp = fp;
        fhold(fp);
        error = 0;
    } else {
        error = EBADF;
    }
    FILEDESC_SUNLOCK(fdp);
    return error;
}

int fdshutdown_all(struct filedesc* fdp, int* pcount, int forced)
{
    int count = 0;
    int fd;
    struct file *fp;
    int error = 0;
    FILEDESC_SLOCK(fdp);
    for (fd = 0; fd < maxfilesperproc; fd++) {
        if (get_field(fdp->fd_map, fd)
            && ((fp = fdp->fd_ofiles[fd].fde_file) != NULL)) {
            sys_shutdown_all_helper(fp, &count, SHUT_RDWR, forced);
        }
    }
    FILEDESC_SUNLOCK(fdp);
    *pcount = count;
    return error;
}

int fget_unlocked(
        struct filedesc *fdp, int fd, cap_rights_t *rightsp,
        int needfcntl, struct file **fpp, cap_rights_t *haverightsp)
{
    return fget(NULL, fd, rightsp, fpp);
}

int fget_write(struct thread *td, int fd, cap_rights_t *rightsp, struct file **fpp)
{
    return fget(td, fd, rightsp, fpp);
}

int fget_read(struct thread *td, int fd, cap_rights_t *rightsp, struct file **fpp)
{
    return fget(td, fd, rightsp, fpp);
}

// 1. Reserve file descriptor id
// 2. Copy file struct pointer from other process
// 3. Increase ref counter on original file struct
int fdup(struct proc *owner, int fd, int* fd_out)
{
    struct file *fp = NULL;
    struct filedesc* fdp = curproc->p_fd;
    int    error;
    int    newfd;
    if (owner == curproc) {
        return EINVAL;
    }
    if (fd < 0 || fd > maxfilesperproc) {
        return EBADF;
    }
    FILEDESC_XLOCK(fdp);
    FILEDESC_SLOCK(owner->p_fd);
    if ((fp = owner->p_fd->fd_ofiles[fd].fde_file) != NULL) {
        if ((newfd = idalloc(fdp)) >= 0) {
            fdp->fd_ofiles[newfd].fde_file = fp;
            fhold(fp);
            *fd_out = newfd;
            error = 0;
        } else {
            error = EMFILE;
        }
    } else {
        error = EBADF;
    }
    FILEDESC_SUNLOCK(owner->p_fd);
    FILEDESC_XUNLOCK(fdp);
    return error;
}

}

