﻿/*-
 * Copyright (c) 2007 Attilio Rao <attilio@freebsd.org>
 * Copyright (c) 2001 Jason Evans <jasone@freebsd.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice(s), this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified other than the possible
 *    addition of one or more copyright notices.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice(s), this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * $FreeBSD$
 */

#ifndef    _SYS_SX_H_
#define    _SYS_SX_H_

#include <sys/_lock.h>
#include <sys/_sx.h>

#ifdef    _KERNEL

#include <sys/pcpu.h>

/*
 * In general, the sx locks and rwlocks use very similar algorithms.
 * The main difference in the implementations is how threads are
 * blocked when a lock is unavailable.  For this, sx locks use sleep
 * queues which do not support priority propagation, and rwlocks use
 * turnstiles which do.
 *
 * The sx_lock field consists of several fields.  The low bit
 * indicates if the lock is locked with a shared or exclusive lock.  A
 * value of 0 indicates an exclusive lock, and a value of 1 indicates
 * a shared lock.  Bit 1 is a boolean indicating if there are any
 * threads waiting for a shared lock.  Bit 2 is a boolean indicating
 * if there are any threads waiting for an exclusive lock.  Bit 3 is a
 * boolean indicating if an exclusive lock is recursively held.  The
 * rest of the variable's definition is dependent on the value of the
 * first bit.  For an exclusive lock, it is a pointer to the thread
 * holding the lock, similar to the mtx_lock field of mutexes.  For
 * shared locks, it is a count of read locks that are held.
 *
 * When the lock is not locked by any thread, it is encoded as a
 * shared lock with zero waiters.
 */

/*
 * Function prototipes.  Routines that start with an underscore are not part
 * of the public interface and are wrappered with a macro.
 */
void sx_sysinit(void *arg);
void sx_init(struct sx *sx, const char *desc);
void sx_init_flags(struct sx *sx, const char *description, int opts);
void sx_destroy(struct sx *sx);
int  sx_try_slock(struct sx *sx);
int  sx_try_xlock(struct sx *sx);
void sx_slock(struct sx *sx);
void sx_xlock(struct sx *sx);
void sx_sunlock(struct sx *sx);
void sx_xunlock(struct sx *sx);
int  sx_slock_sig(struct sx *sx);
int  sx_xlock_sig(struct sx *sx);
#define sx_assert(sx, what)    (void)(0)

struct sx_args {
    struct sx     *sa_sx;
    const char    *sa_desc;
    int            sa_flags;
};

#define SX_SYSINIT_FLAGS(name, sxa, desc, flags)                 \
    static struct sx_args name##_args = {                           \
        (sxa),                                                      \
        (desc),                                                     \
        (flags)                                                     \
    };                                                              \
    SYSINIT(name##_sx_sysinit, SI_SUB_LOCK, SI_ORDER_MIDDLE,        \
        sx_sysinit, &name##_args);                                  \
    SYSUNINIT(name##_sx_sysuninit, SI_SUB_LOCK, SI_ORDER_MIDDLE,    \
        sx_destroy, (sxa))

#define SX_SYSINIT(name, sxa, desc)    SX_SYSINIT_FLAGS(name, sxa, desc, 0)


#define sx_sleep(chan, sx, pri, wmesg, timo)                        \
        _sleep((chan), &(sx)->lock_object, (pri), (wmesg),          \
            tick_sbt * (timo), 0,  C_HARDCLOCK)


/*
 * Options passed to sx_init_flags().
 */
#define SX_DUPOK            0x01
#define SX_NOPROFILE        0x02
#define SX_NOWITNESS        0x04
#define SX_QUIET            0x08
#define SX_NOADAPTIVE       0x10
#define SX_RECURSE          0x20
#define SX_INTERRUPTIBLE    0x40

#if defined(INVARIANTS) || defined(INVARIANT_SUPPORT)
#define SA_LOCKED           LA_LOCKED
#define SA_SLOCKED          LA_SLOCKED
#define SA_XLOCKED          LA_XLOCKED
#define SA_UNLOCKED         LA_UNLOCKED
#define SA_RECURSED         LA_RECURSED
#define SA_NOTRECURSED      LA_NOTRECURSED

/* Backwards compatability. */
#define SX_LOCKED           LA_LOCKED
#define SX_SLOCKED          LA_SLOCKED
#define SX_XLOCKED          LA_XLOCKED
#define SX_UNLOCKED         LA_UNLOCKED
#define SX_RECURSED         LA_RECURSED
#define SX_NOTRECURSED      LA_NOTRECURSED
#endif

#endif /* _KERNEL */

#endif /* !_SYS_SX_H_ */
