// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//
// This example shows how Compute can be used to create a procedural field of
// grass on a generated terrain model which is then rendered with GX2.
// Because we share buffers, the data we produce is consumed in-place by
// rendering, thus eliminating the API overhead of creating and submitting the
// vertices from the host.
//
// All geometry is generated on the GPU, and outputted into a shared buffer.
// A page of grass is computed on the surface of the terrain as bezier patches,
// and flow noise is applied to the angle of the blades to simulate wind.
// Multiple instances of grass are rendered at jittered offsets to add more
// grass coverage without having to compute new pages.
//
// Finally, a physically based sky shader (via GX2) is applied to the
// background to provide an environment for the grass.
//
////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef M_PI
#define M_PI        3.14159265358979323846264338327950288   /* pi */
#endif
#define DEG_TO_RAD             ((float)(M_PI / 180.0))
#define RADIANS(x)             (radians((x)))

#define mix lerp

////////////////////////////////////////////////////////////////////////////////////////////////////

// OpenGLの gl_NumWorkGroups および gl_WorkGroupSizeに相当するセマンティクスは
// HLSLには存在しません。
// それらに相当する値をシェーダ内で定義するか、cbuffer経由で入力する必要があります。
// 以下は、一時的な実装として、スレッドサイズ（gl_NumWorkGroups * gl_WorkGroupSize相当）をグローバル変数で定義しています。
static const uint3 GlobalThreadSize = { 128, 128, 1 };

static const int P_MASK = 255;
static const int P_SIZE = 256;

static const int G_MASK = 15;
static const int G_SIZE = 16;
static const int G_VECSIZE = 4;

cbuffer fixed_terrain : register( b0 )
{
    int4 P[512];
    float4  G[64];
};

cbuffer variable_terrain : register( b1 )
{
    float4 camera_position;
    float4 camera_rotation;
    float4 camera_view;
    float4 camera_left;
    int2 grid_resolution;
    float2 clip_range;
    float camera_fov;
    int vertex_count;
    int vertex_output_offset;
    int normal_output_offset;
    int texcoords_output_offset;
};

//buffer attribute_output {
//    float4 output_data[4096];
//};
RWByteAddressBuffer attribute_output : register( u0 );

void storeAttributeOutputData( int index, float4 value )
{
    attribute_output.Store( index * 16 + 0, asuint( value.x ) );
    attribute_output.Store( index * 16 + 4, asuint( value.y ) );
    attribute_output.Store( index * 16 + 8, asuint( value.z ) );
    attribute_output.Store( index * 16 + 12, asuint( value.w ) );
}


////////////////////////////////////////////////////////////////////////////////////////////////////

int mod(int x, int a)
{
    int n = (x / a);
    int v = x - n * a;
    if (v < 0)
        v += a;
    return v;
}

// smooth is a GLSL built-in
float smoothie(float t)
{
    return t * t * t * (t * (t * 6.0f - 15.0f) + 10.0f);
}

float4 normalized(float4 v)
{
    float d = sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
    d = d > 0.0f ? d : 1.0f;

    float4 result = float4(v.x, v.y, v.z, 0.0f);
    result /= d;
    result.w = 1.0f;
    return result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

int lattice1d(int i)
{
    return P[i].x;   // Bug 9397, should just be int
}

int lattice2d(int2 i)
{
    return P[i.x + P[i.y].x].x;   // Bug 9397, should just be int
}

int lattice3d(int4 i)
{
    return P[i.x + P[i.y + P[i.z].x].x].x;   // Bug 9397, should just be int
}

////////////////////////////////////////////////////////////////////////////////////////////////////

float gradient1d(int i, float v)
{
    int index = (lattice1d(i) & G_MASK) * G_VECSIZE;
    float g = G[index + 0].x;
    return (v * g);
}

float gradient2d(int2 i, float2 v)
{
    int index = (lattice2d(i) & G_MASK) * G_VECSIZE;
    float2 g = float2(G[index + 0].x, G[index + 1].x);
    return dot(v, g);
}

float gradient3d(int4 i, float4 v)
{
    int index = (lattice3d(i) & G_MASK) * G_VECSIZE;
    float4 g = float4(G[index + 0].x, G[index + 1].x, G[index + 2].x, 1.0f);
    return dot(v, g);
}

float2 rotated_gradient2d(int2 i, float2 v, float2 r)
{
    int index = (lattice2d(i) & G_MASK) * G_VECSIZE;
    float2 g = float2(G[index + 0].x, G[index + 1].x);
    g.x = r.x * g.x - r.y * g.y;
    g.y = r.y * g.x + r.x * g.y;
    return g;
}

float dot_rotated_gradient2d(int2 i, float2 v, float2 r)
{
    int index = (lattice2d(i) & G_MASK) * G_VECSIZE;
    float2 g = float2(G[index + 0].x, G[index + 1].x);
    g.x = r.x * g.x - r.y * g.y;
    g.y = r.y * g.x + r.x * g.y;
    return dot(v, g);
}

////////////////////////////////////////////////////////////////////////////////////////////////////


// Unsigned cell noise 1d  (+0.0f -> +1.0f)
float CellNoise1dfu(float position)
{
    float p = position;
    float pf = floor(p);
    int ip = int(pf);
    float fp = p - pf;
    ip &= P_MASK;

    return (lattice1d(ip) * (1.0f / (P_SIZE - 1)));
}

// Signed cell noise 1d (-1.0 -> +1.0f)
float CellNoise1dfs(float position)
{
    return 2.0f * CellNoise1dfu(position) - 1.0f;
}

// Unsigned cell noise 2d  (+0.0f -> +1.0f)
float CellNoise2dfu(float2 position)
{
    float2 p = position;
    float2 pf = floor(p);
    int2 ip = int2(int(pf.x), int(pf.y));
    float2 fp = p - pf;
    ip &= P_MASK;

    return (lattice2d(ip) * (1.0f / (P_SIZE - 1)));
}

// Signed cell noise 2d (-1.0 -> +1.0f)
float CellNoise2dfs(float2 position)
{
    return 2.0f * CellNoise2dfu(position) - 1.0f;
}

// Unsigned cell noise 3d (+0.0f -> +1.0f)
float CellNoise3dfu(float4 position)
{
    float4 p = position;
    float4 pf = floor(p);
    int4 ip = int4(int(pf.x), int(pf.y), int(pf.z), 0);
    float4 fp = p - pf;
    ip &= P_MASK;

    return (lattice3d(ip) * (1.0f / (P_SIZE - 1)));
}

// Signed cell noise 2d (-1.0 -> +1.0f)
float CellNoise3dfs(float4 position)
{
    return 2.0f * CellNoise3dfu(position) - 1.0f;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

// Signed gradient noise 1d (-1.0 -> +1.0f)
float GradientNoise1dfs(float position)
{
    float p = position;
    float pf = floor(p);
    int ip = int(pf);
    float fp = p - pf;
    ip &= P_MASK;

    float n0 = gradient1d(ip + 0, fp - 0.0f);
    float n1 = gradient1d(ip + 1, fp - 1.0f);

    float n = mix(n0, n1, smoothie(fp));
    return n * (1.0f / 0.7f);
}

// Unsigned Gradient Noise 1d
float GradientNoise1dfu(float position)
{
    return (0.5f - 0.5f * GradientNoise1dfs(position));
}

// Signed gradient noise 2d (-1.0 -> +1.0f)
float GradientNoise2dfs(float2 position)
{
    float2 p = position;
    float2 pf = floor(p);
    int2 ip = int2(int(pf.x), int(pf.y));
    float2 fp = p - pf;
    ip &= P_MASK;

    const int2 I00 = int2(0, 0);
    const int2 I01 = int2(0, 1);
    const int2 I10 = int2(1, 0);
    const int2 I11 = int2(1, 1);

    const float2 F00 = float2(0.0f, 0.0f);
    const float2 F01 = float2(0.0f, 1.0f);
    const float2 F10 = float2(1.0f, 0.0f);
    const float2 F11 = float2(1.0f, 1.0f);

    float n00 = gradient2d(ip + I00, fp - F00);
    float n10 = gradient2d(ip + I10, fp - F10);
    float n01 = gradient2d(ip + I01, fp - F01);
    float n11 = gradient2d(ip + I11, fp - F11);

    float2 n0001 = float2(n00, n01);
    float2 n1011 = float2(n10, n11);

    float2 n2 = mix(n0001, n1011, smoothie(fp.x));
    float n = mix(n2.x, n2.y, smoothie(fp.y));
    return n * (1.0f / 0.7f);
}

// Unsigned Gradient Noise 2d
float GradientNoise2dfu(float2 position)
{
    return (0.5f - 0.5f * GradientNoise2dfs(position));
}

// Signed gradient noise 3d (-1.0 -> +1.0f)
float GradientNoise3dfs(float4 position)
{
    float4 p = position;
    float4 pf = floor(p);
    int4 ip = int4(int(pf.x), int(pf.y), int(pf.z), 0);
    float4 fp = p - pf;
    ip &= P_MASK;

    int4 I000 = int4(0, 0, 0, 0);
    int4 I001 = int4(0, 0, 1, 0);
    int4 I010 = int4(0, 1, 0, 0);
    int4 I011 = int4(0, 1, 1, 0);
    int4 I100 = int4(1, 0, 0, 0);
    int4 I101 = int4(1, 0, 1, 0);
    int4 I110 = int4(1, 1, 0, 0);
    int4 I111 = int4(1, 1, 1, 0);

    float4 F000 = float4(0.0f, 0.0f, 0.0f, 0.0f);
    float4 F001 = float4(0.0f, 0.0f, 1.0f, 0.0f);
    float4 F010 = float4(0.0f, 1.0f, 0.0f, 0.0f);
    float4 F011 = float4(0.0f, 1.0f, 1.0f, 0.0f);
    float4 F100 = float4(1.0f, 0.0f, 0.0f, 0.0f);
    float4 F101 = float4(1.0f, 0.0f, 1.0f, 0.0f);
    float4 F110 = float4(1.0f, 1.0f, 0.0f, 0.0f);
    float4 F111 = float4(1.0f, 1.0f, 1.0f, 0.0f);

    float n000 = gradient3d(ip + I000, fp - F000);
    float n001 = gradient3d(ip + I001, fp - F001);

    float n010 = gradient3d(ip + I010, fp - F010);
    float n011 = gradient3d(ip + I011, fp - F011);

    float n100 = gradient3d(ip + I100, fp - F100);
    float n101 = gradient3d(ip + I101, fp - F101);

    float n110 = gradient3d(ip + I110, fp - F110);
    float n111 = gradient3d(ip + I111, fp - F111);

    float4 n40 = float4(n000, n001, n010, n011);
    float4 n41 = float4(n100, n101, n110, n111);

    float4 n4 = mix(n40, n41, smoothie(fp.x));
    float2 n2 = mix(n4.xy, n4.zw, smoothie(fp.y));
    float n = mix(n2.x, n2.y, smoothie(fp.z));
    return n * (1.0f / 0.7f);
}

// Unsigned Gradient Noise 3d
float GradientNoise3dfu(float4 position)
{
    return (0.5f - 0.5f * GradientNoise3dfs(position));
}

////////////////////////////////////////////////////////////////////////////////////////////////////

float RotatedGradientNoise2dfs(float2 position, float angle)
{
    float2 p = position;
    float2 pf = floor(p);
    int2 ip = int2(int(pf.x), int(pf.y));
    float2 fp = p - pf;
    ip &= P_MASK;

    float r = radians(angle);
    float2 rg = float2(sin(r), cos(r));

    const int2 I00 = int2(0, 0);
    const int2 I01 = int2(0, 1);
    const int2 I10 = int2(1, 0);
    const int2 I11 = int2(1, 1);

    const float2 F00 = float2(0.0f, 0.0f);
    const float2 F01 = float2(0.0f, 1.0f);
    const float2 F10 = float2(1.0f, 0.0f);
    const float2 F11 = float2(1.0f, 1.0f);

    float n00 = dot_rotated_gradient2d(ip + I00, fp - F00, rg);
    float n10 = dot_rotated_gradient2d(ip + I10, fp - F10, rg);
    float n01 = dot_rotated_gradient2d(ip + I01, fp - F01, rg);
    float n11 = dot_rotated_gradient2d(ip + I11, fp - F11, rg);

    float2 n0001 = float2(n00, n01);
    float2 n1011 = float2(n10, n11);

    float2 n2 = mix(n0001, n1011, smoothie(fp.x));
    float n = mix(n2.x, n2.y, smoothie(fp.y));
    return n * (1.0f / 0.7f);
}

float4
RotatedSimplexNoise2dfs(float2 position, float angle)
{
    float2 p = position;
    float r = radians(angle);
    float2 rg = float2(sin(r), cos(r));

    const float F2 = 0.366025403f; // 0.5*(sqrt(3.0)-1.0)
    const float G2 = 0.211324865f; // (3.0-Math.sqrt(3.0))/6.0
    const float G22 = 2.0f * G2;

    const float2 FF = float2(F2, F2);
    const float2 GG = float2(G2, G2);
    const float2 GG2 = float2(G22, G22);

   	const float2 F00 = float2(0.0f, 0.0f);
   	const float2 F01 = float2(0.0f, 1.0f);
   	const float2 F10 = float2(1.0f, 0.0f);
   	const float2 F11 = float2(1.0f, 1.0f);

    const int2 I00 = int2(0, 0);
    const int2 I01 = int2(0, 1);
    const int2 I10 = int2(1, 0);
    const int2 I11 = int2(1, 1);

    float s = (p.x + p.y) * F2;
    float2 ps = float2(p.x + s, p.y + s);
    float2 pf = floor(ps);
    int2 ip = int2(int(pf.x), int(pf.y));
    ip &= int2(P_MASK, P_MASK);

    float t = (pf.x + pf.y) * G2;
    float2 tt = float2(t, t);
    float2 tf = pf - tt;
    float2 fp = p - tf;

    float2 p0 = fp;
    int2 i1 = (p0.x > p0.y) ? (I10) : (I01);
    float2 f1 = (p0.x > p0.y) ? (F10) : (F01);

    float2 p1 = p0 - f1 + GG;
    float2 p2 = p0 - F11 + GG2;

    float t0 = 0.5f - p0.x * p0.x - p0.y * p0.y;
    float t1 = 0.5f - p1.x * p1.x - p1.y * p1.y;
    float t2 = 0.5f - p2.x * p2.x - p2.y * p2.y;

    float2 g0 = F00;
    float2 g1 = F00;
    float2 g2 = F00;

    float n0 = 0.0f;
    float n1 = 0.0f;
    float n2 = 0.0f;

    float t20 = 0.0f;
    float t40 = 0.0f;
    float t21 = 0.0f;
    float t41 = 0.0f;
    float t22 = 0.0f;
    float t42 = 0.0f;

    if (t0 >= 0.0f)
    {
        g0 = rotated_gradient2d(ip + I00, p0, rg);
        t20 = t0 * t0;
        t40 = t20 * t20;
        n0 = t40 * dot(p0, g0);
    }

    if (t1 >= 0.0f)
    {
        g1 = rotated_gradient2d(ip + i1, p1, rg);
        t21 = t1 * t1;
        t41 = t21 * t21;
        n1 = t41 * dot(p1, g1);
    }

    if (t2 >= 0.0f)
    {
        g2 = rotated_gradient2d(ip + I11, p2, rg);
        t22 = t2 * t2;
        t42 = t22 * t22;
        n2 = t42 * dot(p2, g2);
    }

    float noise = 40.0f * (n0 + n1 + n2);

    float2 dn = p0 * t20 * t0 * dot(p0, g0);
    dn += p1 * t21 * t1 * dot(p1, g1);
    dn += p2 * t22 * t2 * dot(p2, g2);
    dn *= -8.0f;
    dn += t40 * g0 + t41 * g1 + t42 * g2;
    dn *= 40.0f;

    return float4(noise, dn.x, dn.y, 1.0f);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

float2
GetGridCoordinates(int index, int2 size)
{
    float2 coord;
    coord.x = index % size.x;
    index /= size.x;
    coord.y = index % size.y;
    return coord;
}

float4
ComputeRadialGridPosition(
    float2 uv, // Coordinates in range [0, 1]
    float2 vt, // Coordinates between [0, grid_resolution.xy]
    float2 ve, // Coordinates equal to grid_resolution
    float4 camera_position, // Position of Camera
    float4 camera_view) // Viewing vector of camera
{
    float extend = 360.0 - camera_fov;
    float angle = RADIANS(camera_fov + extend) * uv.y;

    float a0 = 0.0f; //clip_range.x;
    float a1 = clip_range.y - clip_range.x;

    float x2 = vt.x * vt.x;
    float fr = a0 + a1 * x2;
    float fx = fr * cos(angle);
    float fy = fr * sin(angle);

    // Compute z offset of blade from camera
    float4 view = float4(0.0f, 0.0f, ve.y + fy, 1.0f);

    // Compute x offset of blade from camera
    float4 left = float4(ve.x + fx, 0.0f, 0.0f, 1.0f);

    // Compute x/z position of blade
    float4 position = view + left;
    position.y = 0.0f;
    position.w = 1.0f;
    return position;
}

//void ComputeTerrainKernel()
[numthreads(8, 8, 1)]
void main( uint3 dispatchThreadId : SV_DispatchThreadID )
{
    int tx = int(dispatchThreadId.x); // get_global_id(0);
    int ty = int(dispatchThreadId.y); // get_global_id(1);
    int sx = int(GlobalThreadSize.x); // get_global_size(0);
    int sy = int(GlobalThreadSize.y); // get_global_size(1);
    int index = ty * sx + tx;
    if(index > vertex_count)
        return;

    float frequency = 0.0025f;
    float amplitude = 70.00f;
    float phase = 1.0f;
    float lacunarity = 2.0345f;
    float increment = 1.0f;
    float octaves = 1.0f;
    float roughness = 1.00f;

    float ir = float(index) / float(vertex_count);
    int2 di = int2(tx, ty);
    float2 vt = GetGridCoordinates(index, grid_resolution);
    float2 vs = float2(1.0f / float(grid_resolution.x), 1.0f / float(grid_resolution.y));
    float2 ve = float2(float(grid_resolution.x), float(grid_resolution.y));
    float2 uv = vt * vs;

    // From Grass
    float4 lcl_camera_position = camera_position;
    float4 lcl_camera_view = camera_view;
    lcl_camera_position.y = 0.0f;
    lcl_camera_position -= (lcl_camera_position - floor(lcl_camera_position * 0.1f));
    lcl_camera_view -= (lcl_camera_view - floor(lcl_camera_view));
    float4 position = ComputeRadialGridPosition(uv, vt, ve, lcl_camera_position, lcl_camera_view);
    // End From Grass

    float4 bias = float4(phase, 0.0f, phase, 0.0f);
    float4 sampleVar = position + bias;
    float4 noise = RotatedSimplexNoise2dfs(sampleVar.xz * frequency, 35.0f);
    float displacement = noise.x;

    float4 normal = float4(0.0f, 1.0f, 0.0f, 1.0f);
    float4 vertex = sampleVar + (amplitude * displacement * normal);
    vertex.w = 1.0f;

    normal = float4(noise.y, 1.0f, noise.z, 1.0f);
    normal = normalize(normal);

    storeAttributeOutputData( vertex_output_offset + index, vertex );
    // We currently render the terrain in a solid color
//    output_data[normal_output_offset + index] = normal;
//    output_data[texcoords_output_offset + index].xy = uv;
}

