﻿//
// File:       mesh_renderer.cpp
//
// Version:    <1.0>
//
// Disclaimer: IMPORTANT:  This Apple software is supplied to you by Apple Inc. ("Apple")
//             in consideration of your agreement to the following terms, and your use,
//             installation, modification or redistribution of this Apple software
//             constitutes acceptance of these terms.  If you do not agree with these
//             terms, please do not use, install, modify or redistribute this Apple
//             software.
//
//             In consideration of your agreement to abide by the following terms, and
//             subject to these terms, Apple grants you a personal, non - exclusive
//             license, under Apple's copyrights in this original Apple software ( the
//             "Apple Software" ), to use, reproduce, modify and redistribute the Apple
//             Software, with or without modifications, in source and / or binary forms;
//             provided that if you redistribute the Apple Software in its entirety and
//             without modifications, you must retain this notice and the following text
//             and disclaimers in all such redistributions of the Apple Software. Neither
//             the name, trademarks, service marks or logos of Apple Inc. may be used to
//             endorse or promote products derived from the Apple Software without specific
//             prior written permission from Apple.  Except as expressly stated in this
//             notice, no other rights or licenses, express or implied, are granted by
//             Apple herein, including but not limited to any patent rights that may be
//             infringed by your derivative works or by other works in which the Apple
//             Software may be incorporated.
//
//             The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
//             WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
//             WARRANTIES OF NON - INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A
//             PARTICULAR PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION
//             ALONE OR IN COMBINATION WITH YOUR PRODUCTS.
//
//             IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
//             CONSEQUENTIAL DAMAGES ( INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
//             SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
//             INTERRUPTION ) ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION
//             AND / OR DISTRIBUTION OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER
//             UNDER THEORY OF CONTRACT, TORT ( INCLUDING NEGLIGENCE ), STRICT LIABILITY OR
//             OTHERWISE, EVEN IF APPLE HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Copyright ( C ) 2008 Apple Inc. All Rights Reserved.
//
////////////////////////////////////////////////////////////////////////////////////////////////////

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/gx2.h>
#define MTX_USE_PS
#include <cafe/mat.h>
#endif

#include "compute_types.h"
#include "mesh_renderer.h"
#include "camera.h"

#define odd(x) (x%2)

static const int VERTEX_ATTRIB_BUFFER_INDEX = 0;
static const int COLOR_ATTRIB_BUFFER_INDEX = 1;

struct VUniforms
{
    float16 m_afProjMtx;
};

// nn::gfx doesn't address the Big-Endian Cafe, which is as it should be
// but we need the GPU to override Cafe's BE Vertex Buffer default
#ifdef CAFE
ALIGNVAR(GX2_SHADER_ALIGNMENT)
static u32 TerrainRenderFetchShaderData[] =
{
    0x02000000, 0x00148001,
    0x00000000, 0x0000008a,
    0x01a0003c, 0x0510cd28,
    0x00000800, 0x00000000,
    0x21a2003f, 0x0110cd28,
    0x00000a00, 0x00000000,
    0x21a2003f, 0x0210cd28,
    0x10000a00, 0x00000000,
    0x21a2003f, 0x0310cd28,
    0x20000a00, 0x00000000,
    0x21a2003f, 0x0410cd28,
    0x30000a00, 0x00000000,
    0x01a1003c, 0x0010cd28,
    0x00000a00, 0x00000000,
};

static GX2FetchShader TerrainRendererFetchShader =
{
    GX2_FETCH_SHADER_TESSELATION_NONE,
    { 0 },
    112,
    TerrainRenderFetchShaderData,
    6,
    0,
    { 0, 0 }
};

ALIGNVAR(GX2_SHADER_ALIGNMENT)
static u32 GrassRenderFetchShaderData[] =
{
    0x02000000, 0x00148001,
    0x00000000, 0x0000008a,
    0x01a0003c, 0x0510cd28,
    0x00000800, 0x00000000,
    0x21a2003f, 0x0110cd28,
    0x00000a00, 0x00000000,
    0x21a2003f, 0x0210cd28,
    0x10000a00, 0x00000000,
    0x21a2003f, 0x0310cd28,
    0x20000a00, 0x00000000,
    0x21a2003f, 0x0410cd28,
    0x30000a00, 0x00000000,
    0x01a1003c, 0x0010cd28,
    0x00000800, 0x00000000,
};

static GX2FetchShader GrassRendererFetchShader =
{
    GX2_FETCH_SHADER_TESSELATION_NONE,
    {0},
    112,
    GrassRenderFetchShaderData,
    6,
    0,
    {0,0}
};
#endif

MeshRenderer::MeshRenderer() :

m_uiVertexBytes( 0 ),
m_uiVertexCount( 0 ),
m_uiVertexComponents( 0 ),
m_uiModelViewBytes( 0 ),
m_uiModelViewCount( 0 ),
m_uiModelViewComponents( 0 ),
m_uiColorBytes( 0 ),
m_uiColorCount( 0 ),
m_uiColorComponents( 0 ),
m_auiIndexData( 0 ),
m_uiIndexBytes( 0 ),
m_uiIndexCount( 0 ),
m_iVUniformsLoc( 0 ),
m_afSolidColorData( 0 ),
m_bColor( false ),
m_bSolidColor( false)
{
    // EMPTY!
}

MeshRenderer::~MeshRenderer()
{
    m_uiVertexComponents = 0;
    m_uiVertexCount = 0;

    m_uiModelViewComponents = 0;
    m_uiModelViewCount = 0;

    m_uiColorComponents = 0;
    m_uiColorCount = 0;

    m_uiIndexCount = 0;
}

void MeshRenderer::finalize()
{
    m_Pipeline.Finalize();
    if ( m_uiIndexCount )
    {
        m_IndexBuffer.Finalize();
    }
    m_ModelViewBuffer.Finalize();
    if ( m_bSolidColor )
    {
        m_SolidColorData.Finalize();
    }
    m_VertexUniformBuffer.Finalize();
}

void MeshRenderer::setSolidColor( float r, float g, float b )
{
    m_uiColorComponents = 4;
    m_uiColorCount = 1;

    m_SolidColorData.Initialize( sizeof( float4 ), NULL, nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read, 0 );
    m_afSolidColorData = m_SolidColorData.Map< float >();

    m_afSolidColorData[ 0 ] = r;
    m_afSolidColorData[ 1 ] = g;
    m_afSolidColorData[ 2 ] = b;
    m_afSolidColorData[ 3 ] = 1.0;
    m_SolidColorData.Unmap();

    m_bSolidColor = true;
}

void MeshRenderer::setVertexData(
    nn::gfx::GpuAddress afData,
    uint uiComponents,
    uint uiCount)
{
    m_uiVertexComponents = uiComponents;
    m_uiVertexCount = uiCount;
    m_uiVertexBytes = m_uiVertexComponents * m_uiVertexCount * sizeof(float);

    m_afVertexData = afData;
}

void MeshRenderer::setModelViewData(
    uint uiComponents,
    uint uiCount)
{
    m_uiModelViewComponents = uiComponents;
    m_uiModelViewCount = uiCount;
    m_uiModelViewBytes = m_uiModelViewComponents * m_uiModelViewCount * sizeof(float);

    m_ModelViewBuffer.Initialize( m_uiModelViewBytes, NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );
}

void MeshRenderer::setColorData(
    nn::gfx::GpuAddress afData,
    uint uiComponents,
    uint uiCount)
{
    m_uiColorComponents = uiComponents;
    m_uiColorCount = uiCount;
    m_uiColorBytes = m_uiColorComponents * m_uiColorCount * sizeof(float);

    m_afColorData = afData;

    m_bColor = true;
}

const nn::gfx::AttributeFormat compTable[ ] =
{
    nn::gfx::AttributeFormat_32_Float,      // inaccessible
    nn::gfx::AttributeFormat_32_Float,
    nn::gfx::AttributeFormat_32_32_Float,
    nn::gfx::AttributeFormat_32_32_32_Float,
    nn::gfx::AttributeFormat_32_32_32_32_Float,
};

void MeshRenderer::setup( nn::gfx::PrimitiveTopology topology, const char* pFilename )
{
    m_Topology = topology;

    m_Pipeline.SetDefaults();
    DEMOGfxLoadShadersFromFile( &m_Pipeline.shaders, 0, pFilename );

    if (!m_uiVertexCount)
    {
        return;
    }

    if (m_bColor || m_bSolidColor)
    {
        if (m_bColor)
        {
            // Interleaved
            DEMOGfxInitShaderVertexBuffer( &m_Pipeline.shaders, VERTEX_ATTRIB_BUFFER_INDEX, ( m_uiVertexComponents + m_uiColorComponents ) * sizeof( float ), 0 );
            DEMOGfxInitShaderAttribute( &m_Pipeline.shaders, "a_position", VERTEX_ATTRIB_BUFFER_INDEX, 0, compTable[ m_uiVertexComponents ] );
            DEMOGfxInitShaderAttribute(&m_Pipeline.shaders, "a_color", VERTEX_ATTRIB_BUFFER_INDEX, m_uiVertexComponents * sizeof( float ), compTable[ m_uiColorComponents ]  );
        } else {
            DEMOGfxInitShaderVertexBuffer( &m_Pipeline.shaders, VERTEX_ATTRIB_BUFFER_INDEX, m_uiVertexComponents * sizeof( float ), 0 );
            DEMOGfxInitShaderAttribute( &m_Pipeline.shaders, "a_position", VERTEX_ATTRIB_BUFFER_INDEX, 0, compTable[ m_uiVertexComponents ] );

            DEMOGfxInitShaderVertexBuffer( &m_Pipeline.shaders, COLOR_ATTRIB_BUFFER_INDEX, 0, 0 );
            DEMOGfxInitShaderAttribute(&m_Pipeline.shaders, "a_color", COLOR_ATTRIB_BUFFER_INDEX, 0, compTable[ m_uiColorComponents ]  );
        }
    }

    // Uniform buffers
    m_VertexUniformBuffer.Initialize( sizeof( struct VUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );
    m_iVUniformsLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_VUniforms" );
    DEMOAssert( m_iVUniformsLoc != 0xffffffff && "u_VUniforms location is invalid." );
    m_iMVUniformsLoc = m_Pipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_InstancedMV" );
    DEMOAssert( m_iMVUniformsLoc != 0xffffffff && "u_InstancedMV location is invalid." );

    // Set Color Mask
    m_Pipeline.blendTargetStateCount = 1;
    m_Pipeline.blendTargetStateInfoArray[ 0 ].SetDefault();
    m_Pipeline.blendTargetStateInfoArray[ 0 ].SetBlendEnabled( false );
    m_Pipeline.blendTargetStateInfoArray[ 0 ].SetChannelMask( nn::gfx::ChannelMask_All );

    // Color targets
    m_Pipeline.colorTargetStateCount = 1;
    m_Pipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    m_Pipeline.colorTargetStateInfoArray[ 0 ].SetFormat( DEMOColorBufferInfo.GetImageFormat() );

    m_Pipeline.depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_LessEqual );

    m_Pipeline.rasterizerStateInfo.SetMultisampleEnabled( DEMOColorBufferInfo.GetMultisampleCount() > 1 );
    m_Pipeline.rasterizerStateInfo.EditMultisampleStateInfo().SetSampleCount( DEMOColorBufferInfo.GetMultisampleCount() );

    m_Pipeline.Initialize( &DEMODevice );
}

bool
MeshRenderer::createGridIndexBuffer(
    uint uiCountX, uint uiCountY, bool bCloseStrip)
{
    uint y;

    m_uiIndexCount =  2 * (bCloseStrip ? uiCountY : (uiCountY - 1)) * uiCountX;
    if (!m_uiIndexCount)
    {
        return false;
    }
    m_uiIndexBytes = m_uiIndexCount * sizeof(uint);

    m_IndexBuffer.Initialize( m_uiIndexBytes, NULL, nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read, 0 );
    m_auiIndexData = m_IndexBuffer.Map< uint >();

    int i = 0;
    for ( y = 0; y < uiCountY - 1; y++ )
    {
        if ( odd( y ) )
        {
            for ( int x = uiCountX - 1; x >= 0; x-- )
            {
                m_auiIndexData[ i++ ] = ( uiCountX * ( y + 1 ) + x );
                m_auiIndexData[ i++ ] = ( uiCountX * y + x );
            }
        }
        else
        {
            for ( uint x = 0; x < uiCountX; x++ )
            {
                m_auiIndexData[ i++ ] = ( uiCountX * y + x );
                m_auiIndexData[ i++ ] = ( uiCountX * ( y + 1 ) + x );
            }
        }
    }

    // Repeat the initial row
    if ( bCloseStrip )
    {
        if ( odd( y ) )
        {
            for ( int x = uiCountX - 1; x >= 0; x-- )
            {
                m_auiIndexData[ i++ ] = ( uiCountX * ( ( y + 1 ) % uiCountY ) + x );
                m_auiIndexData[ i++ ] = ( uiCountX * ( y % uiCountY ) + x );
            }
        }
        else
        {
            for ( uint x = 0; x < uiCountX; x++ )
            {
                m_auiIndexData[ i++ ] = ( uiCountX * ( y % uiCountY ) + x );
                m_auiIndexData[ i++ ] = ( uiCountX * ( ( y + 1 ) % uiCountY ) + x );
            }
        }
    }
    m_IndexBuffer.Unmap();

    return true;
}

bool MeshRenderer::render( int numInstances, int baseInstance )
{
    dglPushMatrix();

    // Setup constant uniform projection matrix
    struct VUniforms* pVUniforms = m_VertexUniformBuffer.Map< struct VUniforms >( );
    memcpy( &pVUniforms->m_afProjMtx , g_projMtx, sizeof( float16 ) );
#ifdef CAFE
    GX2EndianSwap( pVUniforms, sizeof( struct VUniforms ) );
#endif
    m_VertexUniformBuffer.Unmap( );

    if (m_bColor)
    {
        // Buffer is interleaved
        DEMOCommandBuffer.SetVertexBuffer( VERTEX_ATTRIB_BUFFER_INDEX, m_afVertexData, ( m_uiVertexComponents + m_uiColorComponents ) * sizeof( float ), m_uiVertexBytes + m_uiColorBytes );
    } else {
        DEMOCommandBuffer.SetVertexBuffer( VERTEX_ATTRIB_BUFFER_INDEX, m_afVertexData, m_uiVertexComponents * sizeof( float ), m_uiVertexBytes );
        DEMOCommandBuffer.SetVertexBuffer( COLOR_ATTRIB_BUFFER_INDEX, m_SolidColorData.gpuAddress, 0, m_SolidColorData.size );
    }
    DEMOCommandBuffer.SetConstantBuffer( m_iVUniformsLoc, nn::gfx::ShaderStage_Vertex, m_VertexUniformBuffer.gpuAddress, m_VertexUniformBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( m_iMVUniformsLoc, nn::gfx::ShaderStage_Vertex, m_ModelViewBuffer.gpuAddress, m_ModelViewBuffer.size );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );
    DEMOCommandBuffer.SetPipeline( &m_Pipeline.pipeline );

    // Draw
    if (m_uiIndexCount)
    {
#ifdef CAFE
        GX2SetFetchShader( &TerrainRendererFetchShader );
#endif
        DEMOCommandBuffer.DrawIndexed( m_Topology, nn::gfx::IndexFormat_Uint32, m_IndexBuffer.gpuAddress, m_uiIndexCount, 0, numInstances, baseInstance );
    }
    else
    {
#ifdef CAFE
        GX2SetFetchShader( &GrassRendererFetchShader );
#endif
        DEMOCommandBuffer.Draw( m_Topology, m_uiVertexCount, 0, numInstances, baseInstance );
    }

#if NN_GFX_IS_TARGET_D3D
    nn::gfx::GpuAddress nullAddress;
    DEMOCommandBuffer.SetVertexBuffer( VERTEX_ATTRIB_BUFFER_INDEX, nullAddress, 0, 0 );
#endif

    dglPopMatrix();
    return true;
}
