﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h" /* Needs to be included before 'pthread.h' for '_GNU_SOURCE' */
/* The header above includes other headers this file needs:
     #include <stdio.h>

*/
#include <pthread.h>
#include <stdint.h>
#include <inttypes.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
  #include "ntd-tests/ntd-tests.h"
#ifdef __cplusplus
}
#endif /* __cplusplus */

#define MS_TO_NS(x)	((x)*1000)

const char *testcase;

typedef struct thread_params {
    int thread_index;
    int iterations;
    long ns_predelay;
    long ns_delay;
    long ns_postdelay;
    const char *message;
} thread_params;

thread_params	params[] = {
    {	0,	10,	MS_TO_NS(1),	MS_TO_NS(10),	MS_TO_NS(10),	"AAAAAAAAAAAAAAAAAAAA" },
    {	1,	15,	MS_TO_NS(2),	MS_TO_NS( 8),	MS_TO_NS( 8),	"BBBBBBBBBBBBBBBBBBBB" },
    {	2,	20,	MS_TO_NS(3),	MS_TO_NS( 6),	MS_TO_NS( 6),	"CCCCCCCCCCCCCCCCCCCC" },
    {	3,	25,	MS_TO_NS(4),	MS_TO_NS( 4),	MS_TO_NS( 4),	"DDDDDDDDDDDDDDDDDDDD" },
    {	4,	30,	MS_TO_NS(5),	MS_TO_NS( 2),	MS_TO_NS( 2),	"EEEEEEEEEEEEEEEEEEEE" },
    {	5,	35,	MS_TO_NS(6),	MS_TO_NS( 1),	MS_TO_NS( 1),	"FFFFFFFFFFFFFFFFFFFF" }
};

#define NUM_THREADS (sizeof(params) / sizeof(thread_params))
#define MAX_ITERATIONS 50

typedef struct held_lock {
    uint64_t	acquired;
    uint64_t	released;
} held_lock;

held_lock	locked_data[NUM_THREADS][MAX_ITERATIONS];

uint64_t	base_clock = 0;

uint64_t ns_clock(void)
{
    struct timespec tp;
    uint64_t ns;
    clock_gettime(CLOCK_REALTIME, &tp);
    ns = ((uint64_t) tp.tv_sec) * 1000000000 + (uint64_t) tp.tv_nsec;
    return ns - base_clock;
}

static uintptr_t  okStatus        = 34;

static void *threadfunc(void *arg)
{
    thread_params *parm = (thread_params *)arg;
    int i;
    struct timespec predelay;
    struct timespec delay;
    struct timespec postdelay;
    struct timespec sleepleft;
    int t = parm->thread_index;
    int iterations = parm->iterations;

    if (iterations > MAX_ITERATIONS) iterations = MAX_ITERATIONS;

    predelay.tv_sec = 0;
    predelay.tv_nsec = parm->ns_predelay;
    delay.tv_sec = 0;
    delay.tv_nsec = parm->ns_delay;
    postdelay.tv_sec = 0;
    postdelay.tv_nsec = parm->ns_postdelay;

    for (i = 0; i < iterations; i++) {
        nanosleep(&predelay, &sleepleft);
        flockfile(stdout);
        locked_data[t][i].acquired = ns_clock();
        TESTWARN("%02d:%02d-%s\n", t, i, parm->message);
        nanosleep(&delay, &sleepleft);
        locked_data[t][i].released = ns_clock();
        funlockfile(stdout);
        nanosleep(&postdelay, &sleepleft);
    }
    return (void *)okStatus;
}

int compare_acquired_less(int t1, int i1, int t2, int i2)
{
  return (locked_data[t1][i1].acquired < locked_data[t2][i2].acquired);
}

void check_acquire_release(int t1, int i1)
{
  bool test = ( locked_data[t1][i1].acquired < locked_data[t1][i1].released );
  TESTCASE( test );
  if (!test) {
  	TESTWARN("%s:FAILED: T%dI%d released (%" PRIdMAX ") before acquired (%" PRIdMAX ")\n",
  		testcase,
  		t1,
  		i1,
  		locked_data[t1][i1].released,
  		locked_data[t1][i1].acquired);
  }
}

void print_lockdata(int t1, int i1)
{
  	TESTWARN("%s:FAILED: T%dI%02d (% 9" PRIdMAX "-% 9" PRIdMAX ")\n",
  		testcase, t1, i1,
  		locked_data[t1][i1].acquired,
  		locked_data[t1][i1].released);

}

void check_for_overlap(int t1, int i1, int t2, int i2)
{
  bool test = ( locked_data[t1][i1].released <= locked_data[t2][i2].acquired );
  TESTCASE( test );
  if (!test) {
  	print_lockdata(t1, i1);
  	print_lockdata(t2, i2);
  	TESTWARN("%s:FAILED: T%dI%02d acquired (% 9" PRIdMAX ") before T%dI%02d released (% 9" PRIdMAX ")\n",
  		testcase,
  		t2,
  		i2,
  		locked_data[t2][i2].acquired,
  		t1,
  		i1,
  		locked_data[t1][i1].released);
  }
}

void check_released(int t1, int i1, int thread_iter[NUM_THREADS])
{
  int i;

  check_acquire_release(t1, i1);

  for (i = 0; i < NUM_THREADS; i++) {
  	if (i != t1) {
  		if (thread_iter[i] < params[i].iterations) {
  		  check_for_overlap(t1, i1, i, thread_iter[i]);
  		}
  	}
  }
}

void test_pthread_validate_filelocks(void)
{
  pthread_t             thread[NUM_THREADS];
  int                   rc=0;
  void                 *status;
  int                   i;
  int                   t;
  bool                  test;

  testcase = __FUNCTION__;
  NTD_TEST_GROUP_START("validate_filelocks", 3);

  base_clock = ns_clock();

  for (i = 0; i < NUM_THREADS; i++) {
    rc = pthread_create(&thread[i], NULL, threadfunc, (void *) &params[i]);
  TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);
    if (rc != 0) {
      TESTWARN("%s: failed creating thread %d (result = %d)\n", testcase, i, rc);
      return;
    }
  }

  for (i = 0; i < NUM_THREADS; i++) {
    rc = pthread_join(thread[i], &status);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);
    if (rc != 0) {
      TESTWARN("%s: failed joining thread %d (result = %d)\n", testcase, i, rc);
    }
    test = ( (uintptr_t)(status) == okStatus );
    TESTCASE( test );
    if (!test) {
      TESTWARN("%s: failed thread %d result != okStatus (was %" PRIdPTR ")\n", testcase, i, rc);
      return;
    }
    TESTWARN("%s: thread %i joined\n", testcase, i);
  }

  int lockcheck[NUM_THREADS];

  for (t = 0; t < NUM_THREADS; t++) {
  	lockcheck[t] = 0;
  }

  int done = 0;

  while (done < NUM_THREADS) {
  	int lowest = -1;
  	for (t = 0; t < NUM_THREADS; t++) {
  	  if (lockcheck[t] < params[t].iterations) {
        if (lowest == -1) {
  		  lowest = t;
        } else if (compare_acquired_less(t, lockcheck[t], lowest, lockcheck[lowest])) {
          lowest = t;
        }
      }
    }
 	check_released(lowest, lockcheck[lowest], lockcheck);
 	lockcheck[lowest]++;
 	if (lockcheck[lowest] >= params[lowest].iterations) {
 	  done++;
 	}
  }

  NTD_TEST_GROUP_END("validate_filelocks", 3);
}
