﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
*/
#include <pthread.h>
#include "ntd-tests/ntd-tests.h"

/*
  This example shows the corruption that can result if no
  serialization is done and also shows the use of
  pthread_mutex_lock(). Call it with no parameters
  to use pthread_mutex_lock() to protect the critical section,
  or 1 or more parameters to show data corruption that occurs
  without locking.
  */
#define            LOOPCONSTANT     100000
#define            THREADS          10

#if SUPPORT_STATIC_INITALIZATION
static pthread_mutex_t    mutex = PTHREAD_MUTEX_INITIALIZER;
#else
static pthread_mutex_t    mutex;
#endif
static int                i,j,k,l;
static int                uselock=1;

static void *threadfunc(void *parm)
{
  int   loop = 0;
  int   rc;
  int   lock_failures = 0;
  int   unlock_failures = 0;

  for (loop=0; loop<LOOPCONSTANT; ++loop) {
    if (uselock) {
      rc = pthread_mutex_lock(&mutex);
      if (rc != 0) {
        if (lock_failures == 0) {
            TESTCASE_MESSAGE(rc != 0, "pthread_mutex_lock(%p) returns %d",
                &mutex, rc);
        }
        lock_failures++;
      }
    }
    ++i; ++j; ++k; ++l;
    if (uselock) {
      rc = pthread_mutex_unlock(&mutex);
      if (rc != 0) {
        if (unlock_failures == 0) {
            TESTCASE_MESSAGE(rc != 0, "pthread_mutex_lock(%p) returns %d",
                &mutex, rc);
        }
        unlock_failures++;
      }
    }
  }
  TESTCASE_MESSAGE(lock_failures == 0, "%d failures calling pthread_mutex_lock(%p)",
    lock_failures, &mutex);
  TESTCASE_MESSAGE(unlock_failures == 0, "%d failures calling pthread_mutex_unlock(%p)",
    lock_failures, &mutex);

  return NULL;
}

void test_pthread_mutex_lock_func(int do_lock)
{
  pthread_t             threadid[THREADS];
  int                   rc=0;

  int                   loop=0;
  pthread_attr_t        pta;
  const char *testcase = __FUNCTION__;

  i=j=k=l=0;
#if !SUPPORT_STATIC_INITALIZATION
  rc = pthread_mutex_init(&mutex, NULL);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_init(NULL) returned %d expected 0", rc);
#endif
  TESTWARN("\nEnter Testcase - %s do_lock=%d\n", testcase, do_lock);
  TESTWARN("Give any number of parameters to show data corruption\n");
  if (do_lock != 1) {
    TESTWARN("A parameter was specified, no serialization is being done!\n");
    uselock = 0;
  }

  pthread_attr_init(&pta);
  /* pthread_attr_setdetachstate(&pta, PTHREAD_CREATE_JOINABLE); */

  TESTWARN("Creating %d threads\n", THREADS);
  for (loop=0; loop<THREADS; ++loop) {
    rc = pthread_create(&threadid[loop], &pta, threadfunc, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);
  }

  TESTWARN("Wait for results\n");
  for (loop=0; loop<THREADS; ++loop) {
    rc = pthread_join(threadid[loop], NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);
  }

  TESTWARN("Cleanup and show results\n");
  pthread_attr_destroy(&pta);


  /* The global "mutex" variable gets initialized only once, so if we're doing
     static initialization, then don't call the destroy() function since this
     test can get called multiple times and destroy() might get called twice */
  #if !SUPPORT_STATIC_INITALIZATION
    pthread_mutex_destroy(&mutex);
  #endif

  TESTWARN("\nUsing %d threads and LOOPCONSTANT = %d\n",
         THREADS, LOOPCONSTANT);
  TESTWARN("Values are: (should be %d)\n", THREADS * LOOPCONSTANT);
  TESTWARN("  ==>%d, %d, %d, %d\n", i, j, k, l);

  TESTWARN("%s completed\n", testcase);
}

void test_pthread_mutex_lock()
{
  NTD_TEST_GROUP_START("mutex_lock", 3);
  test_pthread_mutex_lock_func(1);
  test_pthread_mutex_lock_func(0);
  NTD_TEST_GROUP_END("mutex_lock", 3);
}

#if 0
Debug output:

Entering testcase
Give any number of parameters to show data corruption
Creating 10 threads
Wait for results
Cleanup and show results

Using 10 threads and LOOPCONSTANT = 100000
Values are: (should be 1000000)
  ==>1000000, 1000000, 1000000, 1000000
Main completed

Debug output:

(data corruption without locking example)

Entering testcase
Give any number of parameters to show data corruption
A parameter was specified, no serialization is being done!
Creating 10 threads
Wait for results
Cleanup and show results

Using 10 threads and LOOPCONSTANT = 100000
Values are: (should be 1000000)
  ==>883380, 834630, 725131, 931883
Main completed
#endif
