﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
*/
#include <time.h>
#include <sys/time.h>
#include <pthread.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
  #include "ntd-tests/ntd-tests.h"
#ifdef __cplusplus
}
#endif /* __cplusplus */

/* For safe condition variable usage, must use a boolean predicate and  */
/* a mutex with the condition.                                          */
static int                 workToDo = 0;
#if SUPPORT_STATIC_INITALIZATION
static pthread_cond_t      cond  = PTHREAD_COND_INITIALIZER;
static pthread_mutex_t     mutex = PTHREAD_MUTEX_INITIALIZER;
#else
static pthread_cond_t      cond;
static pthread_mutex_t     mutex;
#endif

#define NTHREADS                3
#define WAIT_TIME_SECONDS       1

typedef struct thread_info {
    pthread_t           threadid;
    char                name[16];
} thread_info;
static thread_info      threads[NTHREADS];

static void *threadfunc(void *parm)
{
  int               rc, i;
  struct timespec   ts;
  struct timeval    tp;
  pthread_t         me = pthread_self();
  char              *name = (char *)"Can not find thread";

  for (i = 0; i < NTHREADS; i++) {
      if (threads[i].threadid == me) {
          name = threads[i].name;
          break;
      }
  }

  rc = pthread_mutex_lock(&mutex);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);

  /* Usually worker threads will loop on these operations */
  while (1) {
    rc =  gettimeofday(&tp, NULL);
    TESTCASE_MESSAGE(rc == 0, "gettimeofday() returned %d expected 0", rc);

    /* Convert from timeval to timespec */
    ts.tv_sec  = tp.tv_sec;
    ts.tv_nsec = tp.tv_usec * 1000;
    ts.tv_sec += WAIT_TIME_SECONDS;

    while (!workToDo) {
      TESTWARN("%s blocked\n", name);
      rc = pthread_cond_timedwait(&cond, &mutex, &ts);
      /* If the wait timed out, in this example, the work is complete, and   */
      /* the thread will end.                                                */
      /* In reality, a timeout must be accompanied by some sort of checking  */
      /* to see if the work is REALLY all complete. In the simple example    */
      /* we will just go belly up when we time out.                          */
      if (rc == ETIMEDOUT) {
        TESTWARN("Wait timed out for %s!\n", name);
        rc = pthread_mutex_unlock(&mutex);
        TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);
        pthread_exit(NULL);
      }
      TESTCASE_MESSAGE(rc == 0, "pthread_cond_timedwait() returned %d expected 0", rc);
    }

    TESTWARN("%s consumes work here\n", name);
    workToDo = 0;
  }

  rc = pthread_mutex_unlock(&mutex);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);
  return NULL;
}

void test_pthread_cond_timedwait()
{
  int                   rc=0;
  int                   i;

  NTD_TEST_GROUP_START("cond_timedwait", 3);

#if !SUPPORT_STATIC_INITALIZATION
  rc = pthread_cond_init(&cond, NULL);
  TESTCASE_MESSAGE(rc == 0, "pthread_cond_init() returned %d expected 0", rc);

  rc = pthread_mutex_init(&mutex, NULL);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_init(NULL) returned %d expected 0", rc);
#endif

  TESTWARN("Create %d threads\n", NTHREADS);
  for(i=0; i<NTHREADS; ++i) {
    rc = pthread_create(&(threads[i].threadid), NULL, threadfunc, NULL);
    sprintf(threads[i].name, "thread %d", i);
    TESTCASE_MESSAGE(rc == 0, "pthread_create() returned %d expected 0", rc);
  }

  rc = pthread_mutex_lock(&mutex);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_lock() returned %d expected 0", rc);

  TESTWARN("One work item to give to a thread\n");
  workToDo = 1;
  rc = pthread_cond_signal(&cond);
  TESTCASE_MESSAGE(rc == 0, "pthread_cond_signal() returned %d expected 0", rc);

  rc = pthread_mutex_unlock(&mutex);
  TESTCASE_MESSAGE(rc == 0, "pthread_mutex_unlock() returned %d expected 0", rc);

  TESTWARN("Wait for threads and cleanup\n");
  for (i=0; i<NTHREADS; ++i) {
    rc = pthread_join(threads[i].threadid, NULL);
    TESTCASE_MESSAGE(rc == 0, "pthread_join() returned %d expected 0", rc);
  }

  pthread_cond_destroy(&cond);
  pthread_mutex_destroy(&mutex);

  NTD_TEST_GROUP_END("cond_timedwait", 3);
}

#if 0
Debug output:

Enter Testcase - QP0WTEST/TPCOT0
Create 3 threads
Thread blocked
One work item to give to a thread
Wait for threads and cleanup
Thread consumes work here
Thread blocked
Thread blocked
Thread blocked
Wait timed out!
Wait timed out!
Wait timed out!
Main completed
#endif
