﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "thread_test.h"
/* The header above includes other headers this file needs:
     #include <stdio.h>
*/
#include <pthread.h>
#include <sched.h>

static void validate_scope(pthread_attr_t *a, int set_scope, char *set_scope_str)
{
    int                 rc=0;
    int                 scope;
    char *              scope_str;
    bool                test;

    rc = pthread_attr_getscope(a, &scope);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_getscope() returned %d expected 0", rc);

    if (!rc) {
        if (scope == PTHREAD_SCOPE_PROCESS) {
            scope_str = (char *)"PTHREAD_SCOPE_PROCESS";
#if TEST_UNSUPPORTED_FEATURES
        } else if (scope == PTHREAD_SCOPE_SYSTEM) {
            scope_str = (char *)"PTHREAD_SCOPE_SYSTEM";
#endif
        } else {
            scope_str = (char *)"UNKNOWN_SCOPE";
        }

#if defined(__NX__)
        test = (scope == set_scope);
        /* Linux doesn't support 'PTHREAD_SCOPE_PROCESS', so don't print a
           failure if the test is running on Linux */
        TESTCASE_MESSAGE(scope == set_scope,
            "Failure: expected scope %s but pthread_attr_getscope returned %s",
            set_scope_str, scope_str);
#endif /* __NX__ */
    }
}

void test_pthread_attr_setscope()
{
    int                   rc=0;
    pthread_attr_t        pta;
    bool                  test;

    NTD_TEST_GROUP_START("attr_setscope", 3);

    TESTWARN("Create a thread attributes object\n");
    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);
    if (!rc) {
        validate_scope(&pta, PTHREAD_SCOPE_PROCESS, (char *)"PTHREAD_SCOPE_PROCESS");
    } else {
        TESTWARN("Failed calling pthread_attr_init\n");
    }

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

    TESTWARN("Setting scope to PTHREAD_SCOPE_PROCESS\n");
    rc = pthread_attr_setscope(&pta, PTHREAD_SCOPE_PROCESS);
    /* Linux doesn't support 'PTHREAD_SCOPE_PROCESS', so don't print a
       failure if the test is running on Linux */
    #if defined(__NX__)
      TESTCASE_MESSAGE(rc == 0, "pthread_attr_setscope() returned %d expected 0", rc);
    #endif /* __NX__ */

    if (!rc) {
        validate_scope(&pta, PTHREAD_SCOPE_PROCESS, (char *)"PTHREAD_SCOPE_PROCESS");
    } else {
        TESTWARN("Failed passing %s for scope\n", "PTHREAD_SCOPE_PROCESS");
    }

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

#if TEST_UNSUPPORTED_FEATURES
    TESTWARN("Setting scope to PTHREAD_SCOPE_SYSTEM\n");
    rc = pthread_attr_setscope(&pta, PTHREAD_SCOPE_SYSTEM);
    TESTCASE_MESSAGE(rc == ENOTSUP
        "pthread_attr_setscope(&pta, PTHREAD_SCOPE_SYSTEM) returned %d expected ENOTSUP(%d)",
        rc, ENOTSUP);
#endif

    rc = pthread_attr_init(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_init() returned %d expected 0", rc);

#if TEST_UNSUPPORTED_FEATURES
    TESTWARN("Setting scope to 3\n");
    rc = pthread_attr_setscope(&pta, 3);
    TESTCASE_MESSAGE(rc == EINVAL,
        "pthread_attr_setscope(&pta, 3) returned %d expected EINVAL(%d)",
        rc, EINVAL);
    validate_scope(&pta, 3, (char *)"3");
#endif

    TESTWARN("Destroy thread attributes object\n");
    rc = pthread_attr_destroy(&pta);
    TESTCASE_MESSAGE(rc == 0, "pthread_attr_destroy() returned %d expected 0", rc);

    NTD_TEST_GROUP_END("attr_setscope", 3);
}
