﻿/*---------------------------------------------------------------------------*
  Copyright (C)2015 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "ntd-tests/ntd-tests.h"
#include <string.h>
#include <ctype.h>

// while this test could work on unprintable strings
// limit src and dst to printable strings to make messages easier to read.
#define DOTEST_MEMCCPY(dst, off, src, expect, len, guard, dst_len, expected_result) \
    dotest_memccpy(__LINE__, dst, off, src, expect, len, guard, dst_len, expected_result)

void dotest_memccpy(
    int line,
    char *dst,
    off_t dst_offset,
    const char *src,
    unsigned char guard,
    size_t len,
    const char *expect,
    size_t dst_len,
    char *expected_result)
{
    char *result = memccpy(dst + dst_offset, src, guard, len);
    char *guard_string = "???";
    char guard_string_buffer[10];
    if (isprint(guard)) {
        snprintf(guard_string_buffer, sizeof(guard_string_buffer), "'%c'", guard);
        guard_string = guard_string_buffer;
    } else {
        switch (guard) {
            case '\n':
                guard_string = "'\\n'";
                break;
            case '\t':
                guard_string = "'\\t'";
                break;
            case '\r':
                guard_string = "'\\r'";
                break;
            case '\0':
                guard_string = "'\\0'";
                break;
            default:
                snprintf(guard_string_buffer, sizeof(guard_string_buffer), "'\\%3o'", guard);
                guard_string = guard_string_buffer;
        }
    }
    TESTCASE_WITH_LINE(line, result == expected_result,
        "memccpy(%p, %p, %s, %d) returnd %p expected %p",
        dst, src, guard_string, len, result, expected_result);
    TESTCASE_WITH_LINE(line, memcmp(dst, expect, dst_len) == 0,
        "memccpy(%p (off = %d), %p, %s, %d) dst = '%s' src = '%s' expect = '%s'",
        dst, dst_offset, src, guard_string, len, dst, src, expect);
}

#define DST_SIZE 100
#define INIT_DST_STRING "AaBbCcDdEeFfGgHhIiJjKkLlMmNnOo"
#define SET_DST(pattern) strncpy(dst, pattern, sizeof(dst));
#define SET_SRC(pattern) strncpy(src, pattern, sizeof(src));
#define SET_RESULT(pattern) strncpy(result, pattern, sizeof(result));
#define VALIDATE_SRC(pattern, len) TESTCASE_MESSAGE(memcmp(src, pattern, len) == 0, \
        "memccpy(%p, %p, ...) overwrote the input pointer",  dst, src);

// setup src with sequential characters
void set_seq_chars(char *str, size_t offset, size_t count, unsigned char start, size_t size)
{
    if (offset+count >= size) {
        TESTCASE_MESSAGE(offset+count < size,
            "offset %d + count %d should not be bigger than %d\n",
            offset, count, size);
        return;
    }
    int i;

    for (i = 0; i < count; i++) {
        str[i + offset] = start + (unsigned char) i;
    }
    str[offset + count] = '\0';
}

void ntd_test_memccpy( void )
{
    char dst[DST_SIZE];
    char src[DST_SIZE];
    char result[DST_SIZE];


    NTD_TEST_GROUP_START("memccpy", 3);

    SET_DST(INIT_DST_STRING);
    DOTEST_MEMCCPY(dst, 0, "abc", 'q', 4, "abc", 4, NULL);

    SET_DST("abcdefgh");
    DOTEST_MEMCCPY(dst, 1, "xyz", 'q', 2, "axydefgh", 9, NULL);

    SET_DST("abc");
    DOTEST_MEMCCPY(dst, 0, "xyz", 'q', 0, "abc", 4, NULL);

    SET_SRC("hi there");
    SET_DST("foo");
    DOTEST_MEMCCPY(dst, 0, src, 'q', 9, "hi there", 9, NULL);
    VALIDATE_SRC("hi there", 9);

    SET_SRC("abcdefgh");
    SET_DST("horsefeathers");
    DOTEST_MEMCCPY(dst, 0, src, 'f', 9, "abcdefeathers", 14, &dst[6]);
    VALIDATE_SRC("abcdefgh", 9);

    SET_SRC("abcd");
    SET_DST("bumblebee");
    DOTEST_MEMCCPY(dst, 0, src, 'a', 4, "aumblebee", 10, &dst[1]);
    VALIDATE_SRC("abcd", 5);
    DOTEST_MEMCCPY(dst, 0, src, 'd', 4, "abcdlebee", 10, &dst[4]);
    VALIDATE_SRC("abcd", 5);
    SET_SRC("xyz");
    DOTEST_MEMCCPY(dst, 0, src, 'x', 1, "xbcdlebee", 10, &dst[1]);
    VALIDATE_SRC("xyz", 4);

    // RYNDA-501
    SET_SRC("0123456789\nabcdef\nABC");
    SET_DST(INIT_DST_STRING);
    DOTEST_MEMCCPY(dst, 0, src, '\n', 64, "0123456789\nfGgHhIiJjKkLlMmNnOo", 31, &dst[11]);

    SET_DST(INIT_DST_STRING);
    DOTEST_MEMCCPY(dst, 0, src, '\n', 11, "0123456789\nfGgHhIiJjKkLlMmNnOo", 31, &dst[11]);

    SET_DST(INIT_DST_STRING);
    DOTEST_MEMCCPY(dst, 0, src, '\n', 9, "012345678eFfGgHhIiJjKkLlMmNnOo", 31, NULL);

    SET_DST(INIT_DST_STRING);
    DOTEST_MEMCCPY(dst, 0, src, '\n', 10, "0123456789FfGgHhIiJjKkLlMmNnOo", 31, NULL);

    // musl memccpy is optimized to do a word wise copy if both src and dst have same alignment
    // (src & 3) == (dst & 3)
    // try all different alignments: 0, 1, 2, 3
    // try the matching character being before the first fully aligned word
    // try the matching character being after the last fully aligned word
    // try the matching character being in each possible position in a word in the source string.

    // setup a src string with '0' .. '0' + (sizeof(size_t) * 3 - 2)
    // vary matching character between 0..9
    // vary source length between 0 and 9
    // should cover every combination.

    int align_s;
    int n = (3*sizeof(size_t)) - 2;
    for (align_s = 0; align_s < sizeof(size_t); align_s++) {
        char *s1 = &src[align_s];
        int align_d;

        set_seq_chars(src, align_s, n, '0', sizeof(src));
        for (align_d = 0; align_d < sizeof(size_t); align_d++) {
            char *d1 = &dst[align_d];
            size_t len;

            for (len = 0; len < n; len++) {
                size_t g;
                for (g = 0; g < n; g++) {
                    unsigned char guard = '0' + (unsigned char) g;
                    char *expect_result = &d1[g + 1];
                    if (len < (g + 1)) {
                        expect_result = NULL;
                    }
                    set_seq_chars(dst, align_d, 26, 'A', sizeof(dst));
                    set_seq_chars(result, 0, 26, 'A', sizeof(result));
                    if (len > 0) {
                        size_t copy_len = len;
                        if (g < len) {
                            copy_len = g+1;
                        }
                        memcpy(result, s1, copy_len);
                    }

                    DOTEST_MEMCCPY(d1, 0, s1, guard, len, result, 27, expect_result);
                }
            }
        }
    }


    NTD_TEST_GROUP_END("memccpy", 3);
}
