﻿/*******************************************************************************

  This file contains a series of tests for realloc()

*******************************************************************************/

#include "ntd_extended_test.h"
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <errno.h>


static void test_remember_numbers()
{
    int i; /* Loop counter */
    int array_size; /* Will store size of array 'numbers_to_add' */
    int *numbers = NULL;
    int *realloc_numbers = NULL; /* Return value from realloc() */
    int numbers_to_insert[] = { 37, 8, 1, 2, 3, 9001 };

    array_size = sizeof(numbers_to_insert) / sizeof(numbers_to_insert[0]);

    for (i = 0; i < array_size; ++i)
    {
        realloc_numbers = (int*)realloc(numbers, (i+1) * sizeof(int));

        TESTCASE(realloc_numbers != NULL);
        if (realloc_numbers == NULL) {
            /* The passed-in pointer should be untouched if realloc() fails */
            free(numbers);
            return;
        }

        numbers = realloc_numbers;
        numbers[i] = numbers_to_insert[i];
    }

    /* Make sure all the numbers got inserted */
    for (i = 0; i < array_size; ++i)
    {
        TESTCASE(numbers[i] == numbers_to_insert[i]);
    }

    free(numbers);
}

static void test_realloc_chars()
{
    int i;
    char *str_ret_val;
    char *strcat_string;
    char *char_ptr = NULL;
    char *realloc_ptr = NULL;
    char char_array1[] = { 'a', 'b', 'y', 'x', 'z', '\0' };
    char char_array2[] = { 't', 'a', 'c', 'o', 'c', 'a', 't', '\0' };
    char *joined_strings = "abyxztacocat";
    int array_size1 = sizeof(char_array1);
    int array_size2 = sizeof(char_array2);
    size_t strcat_string_size = array_size1 + array_size2;

    /* Allocate memory for 'strcat_string', then concatenate 'char_array1'
     * and 'char_array2' into it
     */
    strcat_string = (char*)malloc((strcat_string_size) * sizeof(char));
    TESTCASE(strcat_string);
    if (!strcat_string) return;
    str_ret_val = strncpy(strcat_string, char_array1, strcat_string_size);
    TESTCASE(str_ret_val == strcat_string);
    str_ret_val = strncat(strcat_string, char_array2, strcat_string_size);
    TESTCASE(str_ret_val == strcat_string);
    TESTCASE_STRINGS_MATCH(strcat_string, joined_strings);

    /* Basic flow of the test:
       malloc() memory for string1, strncpy() string1,
       strcmp() to make sure string1 copied over, realloc() memory for string2,
       strcat() string2, strcmp() to check for string1+string2 */

    /* Allocate memory to contain the first string */
    char_ptr = (char*)malloc(array_size1);
    TESTCASE(char_ptr);
    if (!char_ptr) return;

    str_ret_val = strncpy(char_ptr, char_array1, array_size1);
    TESTCASE(str_ret_val == char_ptr);
    TESTCASE_STRINGS_MATCH(char_ptr, char_array1);

    /* Re-allocate memory to contain the second string */
    realloc_ptr = (char*)realloc(char_ptr, (array_size1 + array_size2));
    TESTCASE(realloc_ptr);
    if (!realloc_ptr) return;

    char_ptr = realloc_ptr;

    TESTCASE_STRINGS_MATCH(char_ptr, char_array1);

    /* Concatenate the second string into 'char_ptr' so that it contains both
     string1 and strng2 */
    str_ret_val = strncat(char_ptr, char_array2, array_size1 + array_size2);
    TESTCASE(str_ret_val == char_ptr);
    TESTCASE_STRINGS_MATCH(char_ptr, joined_strings);
    TESTCASE_STRINGS_MATCH(char_ptr, strcat_string);

    free(strcat_string);
    free(char_ptr);
}

static void test_realloc_strings()
{
    char **char_arr;
    char **realloc_ptr;
    char *realloc_ptr_string;
    char *str_ret_val;
    char *final_string;
    char *final_concat_string;
    int i, j;
    int middle_index;
    int strings1_num_strings;
    int strings2_num_strings;
    int strings1_size;
    int strings2_size;
    char *strings1[] = { "asdf", "words", "text" };
    char *strings2[] = { "aiajfiasoidjafldjriojzoijaosdijeppajfs", "---+-+---_++==",
                       "*@*(&@(*&@%)*(^@#%)(&^@%#", "a",
                       "xyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzxyzabcxyzxyzxyzxyz",
                       "1", "2", "3", "4", "5", "6", "7", "8", "9", "10" };
    strings1_num_strings = sizeof(strings1) / sizeof(strings1[0]);
    strings2_num_strings = sizeof(strings2) / sizeof(strings2[0]);
    strings1_size = 0;
    strings2_size = 0;
    final_string = "final_string";
    middle_index = (strings1_num_strings + strings2_num_strings) / 2;
    char printChar = '+';

    /* Figure out how many bytes are in each array of strings */
    for (i = 0; i < strings1_num_strings; ++i) {
        strings1_size += strlen(strings1[i]) + 1;
    }
    for (i = 0; i < strings2_num_strings; ++i) {
        strings2_size += strlen(strings2[i]) + 1;
    }

    /* Malloc memory for the array and for each string within the array, and
     * also copy each string from 'strings1' into the array
     */
    char_arr = (char**)malloc(sizeof(char*) * strings1_num_strings);
    TESTCASE(char_arr);
    if (!char_arr) return;

    for (i = 0; i < strings1_num_strings; ++i)
    {
        size_t str_size = strlen(strings1[i]) + 1;
        char_arr[i] = (char*)malloc(str_size);
        TESTCASE(char_arr[i]);
        if (!char_arr[i]) return;
        str_ret_val = strncpy(char_arr[i], strings1[i], str_size);
        TESTCASE(str_ret_val == char_arr[i]);
    }

    /* Make sure all the strings got copied */
    for (i = 0; i < strings1_num_strings; ++i)
    {
        TESTCASE_STRINGS_MATCH(char_arr[i], strings1[i]);
    }

    /* Realloc memory to contain the 2nd array of strings */
    realloc_ptr = (char**)realloc(char_arr, sizeof(char*) * (strings1_num_strings + strings2_num_strings));
    TESTCASE(realloc_ptr);
    if (!realloc_ptr) return;
    char_arr = realloc_ptr;

    /* Make sure the original content remains in-tact after the call to realloc() */
    for (i = 0; i < strings1_num_strings; ++i)
    {
        TESTCASE_STRINGS_MATCH(char_arr[i], strings1[i]);
    }

    /* Allocate memory for all the strings in strings2 and strncpy() them over */
    for (i = strings1_num_strings, j = 0; i < (strings1_num_strings + strings2_num_strings); ++i, ++j)
    {
        size_t str_size = strlen(strings2[j]) + 1;
        char_arr[i] = (char*)malloc(str_size);
        TESTCASE_MESSAGE(char_arr[i] != NULL, "malloc(%d) failed", str_size);
        if (!char_arr[i]) return;
        str_ret_val = strncpy(char_arr[i], strings2[j], str_size);
        TESTCASE(str_ret_val == char_arr[i]);
    }

    /* Make sure all the strings are copied into place */
    for (i = 0; i < strings1_num_strings; ++i)
    {
        TESTCASE_STRINGS_MATCH(char_arr[i], strings1[i]);
    }
    for (i = strings1_num_strings, j = 0; i < (strings1_num_strings + strings2_num_strings); ++i, ++j)
    {
        TESTCASE_STRINGS_MATCH(char_arr[i], strings2[j]);
    }


    /* Now test realloc()-ing one individual string in the array.
     * Choose a string in the middle of the array so we're not
     * hard-coding an index
     */

    /* Build the final test string by mallocing memory for it and concatenating
     * the necessary strings together
     */
    size_t final_string_size = strlen(char_arr[middle_index]) + strlen(final_string) + 1;

    final_concat_string = (char*)malloc(final_string_size);
    TESTCASE_MESSAGE(final_concat_string, "malloc(%d) failed", final_string_size);
    if (!final_concat_string) return;

    str_ret_val = strncpy(final_concat_string, char_arr[middle_index], final_string_size);
    TESTCASE(str_ret_val == final_concat_string);
    str_ret_val = strncat(final_concat_string, final_string, final_string_size);
    TESTCASE(str_ret_val == final_concat_string);

    /* Realloc the string in the array at index 'middle_index' and add more to that string */

    size_t realloc_ptr_size =  strlen(char_arr[middle_index]) + strlen(final_string) + 1;
    realloc_ptr_string =
       (char*)realloc( char_arr[middle_index], realloc_ptr_size);
    TESTCASE_MESSAGE(realloc_ptr_string != NULL, "realloc(%p, %d) = %p",
        char_arr[middle_index], realloc_ptr_size, realloc_ptr_string);
    if (!realloc_ptr_string) return;

    char_arr[middle_index] = realloc_ptr_string;
    str_ret_val = strncat(char_arr[middle_index], final_string, realloc_ptr_size);
    TESTCASE(str_ret_val == char_arr[middle_index]);

    TESTCASE_STRINGS_MATCH(char_arr[middle_index], final_concat_string);

    free(final_concat_string);
    /* Free all the individual strings first, then free the main pointer itself */
    for (i = 0; i < strings1_num_strings + strings2_num_strings; ++i)
    {
        free(char_arr[i]);
    }
    free(char_arr);
}



static void test_null_argument()
{
    char *realloc_ptr;
    char *str_ret_val;
    char *strcpy_string = "strcpy_string";
    size_t realloc_size = strlen(strcpy_string) + 1;
    realloc_ptr = NULL;


    realloc_ptr = (char*)realloc(NULL, realloc_size);
    TESTCASE_MESSAGE(realloc_ptr != NULL, "realloc(NULL, %d) returned %p",
        realloc_size, realloc_ptr);
    if (!realloc_ptr) return;

    str_ret_val = strncpy(realloc_ptr, strcpy_string, realloc_size);
    TESTCASE(str_ret_val == realloc_ptr);

    TESTCASE_STRINGS_MATCH(realloc_ptr, strcpy_string);
    free(realloc_ptr);
}

static void test_calloc_realloc_ints()
{
    int i, j; /* Loop counters */
    int *realloc_ptr; /* Return value from realloc() */
    int *int_ptr;
    int arr_size1; /* Size of 'int_arr1' */
    int arr_size2; /* Size of 'int_arr2' */
    int int_arr1[] = { 1, 2, 3, 1324, 518951, 1381925391, 82829918, 10 };
    int int_arr2[] = { 9, 8, 7, 6, 5, 4 };
    arr_size1 = sizeof(int_arr1) / sizeof(int_arr1[0]);
    arr_size2 = sizeof(int_arr2) / sizeof(int_arr2[0]);

    /* Copy the array of numbers into the allocated memory then check to make
     * sure they were copied in correctly
     */

    int_ptr = (int*)calloc(arr_size1, sizeof(int));
    TESTCASE_MESSAGE(int_ptr != NULL, "calloc(%d, %d) failed", arr_size1, sizeof(int));
    if (!int_ptr) return;

    for (i = 0; i < arr_size1; ++i) {
        int_ptr[i] = int_arr1[i];
    }
    for (i = 0; i < arr_size1; ++i)
    {
        TESTCASE(int_ptr[i] == int_arr1[i]);
    }

    realloc_ptr = (int*)realloc(int_ptr, sizeof(int) * (arr_size1 + arr_size2));
    TESTCASE_MESSAGE(realloc_ptr != NULL,
        "realloc(%p, %d) failed",
        int_ptr, sizeof(int) * (arr_size1 + arr_size2));
    if (!realloc_ptr) return;

    int_ptr = realloc_ptr;
    for (i = arr_size1, j = 0; i < (arr_size1 + arr_size2); ++i, ++j) {
        int_ptr[i] = int_arr2[j];
    }
    for (i = 0; i < arr_size1; ++i)
    {
        TESTCASE(int_ptr[i] == int_arr1[i]);
    }
    for (i = arr_size1, j = 0; i < (arr_size1 + arr_size2); ++i, ++j)
    {
        TESTCASE(int_ptr[i] == int_arr2[j]);
    }
    free(int_ptr);
}



static void test_calloc_realloc_doubles()
{
    int i, j;
    double *realloc_ptr;
    double *dbl_ptr;
    int arr_size1;
    int arr_size2;
    double dbl_arr1[] = { 1.0, 1234358194.123483, 0.000001, 3.14,
                        0.772827, 8283282.283732, 1.111101, 0.101010 };
    double dbl_arr2[] = { 9.999999, 7777.777777, 123.456, 89.0 };
    arr_size1 = sizeof(dbl_arr1) / sizeof(dbl_arr1[0]);
    arr_size2 = sizeof(dbl_arr2) / sizeof(dbl_arr2[0]);

    /* Copy the array of numbers into the allocated memory then check to make
     * sure they were copied in correctly
     */
    dbl_ptr = (double*)calloc(arr_size1, sizeof(double));
    TESTCASE_MESSAGE(dbl_ptr != NULL, "calloc(%d, %d) failed", arr_size1, sizeof(double));
    if (!dbl_ptr) return;

    for (i = 0; i < arr_size1; ++i) dbl_ptr[i] = dbl_arr1[i];

    for (i = 0; i < arr_size1; ++i)
    {
        TESTCASE(dbl_ptr[i] == dbl_arr1[i]);
    }

    realloc_ptr = (double*)realloc(dbl_ptr, sizeof(double) * (arr_size1 + arr_size2));
    TESTCASE_MESSAGE(realloc_ptr != NULL, "realloc(%p, %d) failed",
        dbl_ptr, sizeof(double) * (arr_size1 + arr_size2));
    if (!realloc_ptr) return;

    dbl_ptr = realloc_ptr;
    for (i = arr_size1, j = 0; i < (arr_size1 + arr_size2); ++i, ++j)
        dbl_ptr[i] = dbl_arr2[j];

    for (i = 0; i < arr_size1; ++i)
    {
        TESTCASE(dbl_ptr[i] == dbl_arr1[i]);
    }

    for (i = arr_size1, j = 0; i < (arr_size1 + arr_size2); ++i, ++j)
    {
        TESTCASE(dbl_ptr[i] == dbl_arr2[j]);
    }

    free(dbl_ptr);
}

static void test_realloc_smaller_size()
{
    char *realloc_ptr;
    char *char_ptr;
    char *str_ret_val;
    char *str         = "1234567890";
    char *str_smaller = "12345";
    size_t str_size = strlen(str) + 1;
    size_t str_smaller_size = strlen(str_smaller) + 1;

    char_ptr = (char*)malloc(str_size);
    TESTCASE_MESSAGE(char_ptr, "malloc(%d) failed", str_size);
    if (!char_ptr) return;

    str_ret_val = strncpy(char_ptr, str, str_size);
    TESTCASE(str_ret_val == char_ptr);

    TESTCASE_STRINGS_MATCH(char_ptr, str);

    realloc_ptr = (char*)realloc(char_ptr, str_smaller_size);
    TESTCASE_MESSAGE(realloc_ptr, "realloc(%p, %d) failed", char_ptr, str_smaller_size);
    if (!realloc_ptr) return;

    char_ptr = realloc_ptr;
    TESTCASE_STRINGS_MATCH_LENGTH(char_ptr, str, str_smaller_size);

    free(realloc_ptr);
}

void test_realloc_max_size()
{
    char *realloc_ptr; /* Return value from realloc() */
    char *char_ptr;

    /* See if malloc() fails due to lack of memory */
    errno = 0;
    char_ptr = (char*)malloc(SIZE_MAX);
    TESTCASE_MESSAGE(char_ptr == NULL, "malloc(%d) succeeded %p, should have failed.",
        SIZE_MAX, char_ptr);

    if (char_ptr == NULL) {
        HORIZON_KNOWN_FAILURE("RYNDA-330", errno == ENOMEM);
    } else {
        HORIZON_KNOWN_FAILURE("RYNDA-330", char_ptr == NULL && errno == ENOMEM);
        free(char_ptr);
        return;
    }

    /* Now malloc() some actual memory that should succeed */
    char_ptr = (char*)malloc(20);
    TESTCASE_MESSAGE(char_ptr, "malloc(20) failed");
    if (!char_ptr) return;

    realloc_ptr = (char*)realloc(char_ptr, sizeof(char) * SIZE_MAX);
    if (realloc_ptr == NULL) {
        HORIZON_KNOWN_FAILURE("RYNDA-330", errno == ENOMEM);
        /* Since realloc() failed, this memory should remain intact, so we need
         * to still free() it
         */
        free(char_ptr);
    } else {
        HORIZON_KNOWN_FAILURE("RYNDA-330", realloc_ptr == NULL && errno == ENOMEM);
        free(realloc_ptr);
    }
}


int ntd_extended_realloc(void)
{
  /* Test allocating memory at the start of the program and leaving it allocated
     through the duration of the other tests to test leaving memory in use
     longer term */
    char *char_ptr;
    NTD_TEST_GROUP_START("realloc", 2);
    char_ptr = (char*)malloc(100000);
    TESTCASE_MESSAGE(char_ptr, "malloc(100000) failed");
    if (char_ptr) {
        test_remember_numbers();
        test_realloc_chars();
        test_realloc_strings();
        test_null_argument();
        test_calloc_realloc_ints();
        test_calloc_realloc_doubles();
        test_realloc_smaller_size();
        test_realloc_max_size();

        free(char_ptr);
    }
    return NTD_TEST_GROUP_END("realloc", 2);
}
