﻿/*******************************************************************************

  This file contains various tests for the function perror()

*******************************************************************************/

#include "errno_error_triggers.h"
#include "fileio_helper_functions.h" /* DoesFileExist(), DeleteFile(), OpenFile(), etc. */
/* The header above also includes files this test uses:
     <stdio.h>  // rewind() */
#include <string.h> /* strstr(), strncpy(), strcmp() */
#include <stdlib.h> /* malloc() */
#include <errno.h> /* errno, strerror() */
#include "test.h" /* t_error(), t_status */
#include "malloc_helper_functions.h" /* is_ptr_null() */


/*{{{{------------------------------------------------------------------------*/
/* Redirects all stderr to a file in order to capture the text from perror()
   to make sure perror() is printing the right text.
   (Note, this function is not used as a test. It is a helper function for the
    other tests in this file)
 !!! ---> Be aware that the caller of this function must delete the file that
          stderr gets redirected to. That is, just call
          DeleteFile(filename) at some point after this function
          returns */
void perror_helper__redirect_stderr_to_file(const char *stderr_filename)
{
  FILE *freopen_ptr; /* Return value from freopen() */
  const char *funcname; /* This function's name. For printing in case there's an error */
  funcname = "perror_helper__redirect_stderr_to_file";

  freopen_ptr = freopen(stderr_filename, "w", stderr);
  TESTCASE_MESSAGE(freopen_ptr != NULL, "freopen('%s', 'w', stderr) failed freopen_ptr = %p, errno = %d",
    stderr_filename, freopen_ptr, errno);
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Make sure the filename that will be used in these tests really doesn't exist */
void perror_helper__make_sure_file_doesnt_exist(const char *filename)
{
  if (DoesFileExist(filename))
  {
    DeleteFile(filename);
  }
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Make sure perror() properly formats is message as per the C Standard:
     - Passed-in string is prepended to error message
     - Prepended string is followed by a colon (:) and a space
     - Entire message ends with a newline (\n)
   C Standard: ISO/IEC 9899:TC2, May 6, 2005, WG14/N1124 */
void test_proper_formatting(const char *testname, const char *filename)
{
  FILE *non_exist_file; /* Non-existent file that program tries to open to trigger error */
  FILE *stderr_redirect_file; /* File that contain's stderr's redirected output */
  char *str_ret_val; /* Return value from strstr() */
  char *substr; /* Substring, for comparison later */
  char *fread_buffer;
  const char *perror_text; /* Text to be prepended to perror() message */
  const char *stderr_filename;
  int strlen_perror_text; /* Save strlen(perror_text) value so strlen() isn't caled a bunch */
  #if __NX__
    stderr_filename = "host:/perror.c_stderr_redirect.txt";
  #else
    stderr_filename = "perror.c_stderr_redirect.txt";
  #endif /* if __NX__ */

  perror_helper__redirect_stderr_to_file(stderr_filename);

  /* Attempt to open a non-existing file in order to make sure perror() will
     have something to print */
  non_exist_file = fopen(filename, "r");
  TESTCASE_MESSAGE(non_exist_file == NULL, "fopen('%s','r') opened a file that should be non-existent.",
    filename
  );
  if (non_exist_file != NULL) {
    CloseFile(non_exist_file, filename);
    return;
  }

  perror_text = "Testing perror() - (Expected failure)";
  strlen_perror_text = strlen(perror_text);
  perror(perror_text);
  fflush(stderr);

  /* Obtain the text that perror() wrote to stderr */
  OpenFile(stderr_filename, "r", &stderr_redirect_file);
  ReadEntireFile(stderr_redirect_file, &fread_buffer);
  CloseFile(stderr_redirect_file, stderr_filename); stderr_redirect_file = NULL;

  /* Now check if perror()'s text was formatted properly */

  /* See if 'perror_text' exists inside fread_buffer */
  str_ret_val = strstr(fread_buffer, perror_text);
  TESTCASE_MESSAGE(str_ret_val != NULL,
        "The prepended text was not found in perror()'s message.\n"
        "The argument passed into perror() should be included in the full message.\n"
        "Output of perror(): \"%s\".\n"
        "The text that was expected to be prepended to that output: \"%s\"\n",
        fread_buffer, perror_text);

  /* See if perror()'s prepended message was altered in any way by extracting
     the prepended text from the full perror() message and comparing strings
     (it should be the same as 'perror_text') */
  substr = (char*)malloc(sizeof(char) * (strlen_perror_text + 1)); /* +1 for '\0' */
  if (is_ptr_null(substr)) return;
  str_ret_val = strncpy(substr, str_ret_val, strlen_perror_text);
  if (is_ptr_null(str_ret_val)) return;
  substr[strlen_perror_text] = '\0';
  TESTCASE_STRINGS_MATCH(substr, perror_text);

  /* Make sure there's a colon (:) and a space right after the prepended text */
  TESTCASE_MESSAGE(strlen(fread_buffer) > strlen_perror_text+1
            "perror()'s full message size is less than or equal to the size of "
            "the prepended text. Expected more text after the prepended text.\n"
            "perror()'s message size = %d.\n"
            "Prepended text size = %d.\nperror()'s full message text = \"%s\".\n"
            "Prepended text = \"%s\"\n",
            strlen(fread_buffer), strlen_perror_text, fread_buffer,
            perror_text);
  }
  TESTCASE_MESSAGE(fread_buffer[strlen_perror_text] == ':',
            "There is no colon (:) after the prepended text in perror()'s "
            "message.\nGot '%c'. Expected ':'.\nperror()'s full "
            "message text = \"%s\"\n",
            fread_buffer[strlen_perror_text], fread_buffer);
  }
  TESTCASE_MESSAGE(fread_buffer[strlen_perror_text+1] == ' ',
            "There is no space after the prepended text in perror()'s "
            "message.\nGot '%c'. Expected ' '.\nperror()'s full "
            "message text = \"%s\"\n",
            fread_buffer[strlen_perror_text+1], fread_buffer);
  }

  /* Make sure there's a newline at the very end of perror()'s message */
  TESTCASE_MESSAGE(fread_buffer[strlen(fread_buffer)-1] != '\n',
            "There is no newline after perror()'s message. "
            "Expected newline at the end of it.\nperror()'s full message "
            "text = \"%s\"\n", fread_buffer);
  }

  /* Clean up */
  free(substr);
  free(fread_buffer);
  DeleteFile(stderr_filename);
}
/*------------------------------------------------------------------------}}}}*/




/*{{{{------------------------------------------------------------------------*/
/* Make sure the right error message is printed in relation to the error
   that occurred. Basically just test against strerror() since the C Standard
   says that perror()'s message should be the same as strerror()'s */
void test_correct_error_messages_helper(const char *filename,
                                        int errno_code)
{
  FILE *non_exist_file; /* Non-existent file that program tries to open to trigger error */
  FILE *stderr_redirect_file; /* File that contain's stderr's redirected output */
  char *fread_buffer;
  char *strerror_string; /* Return value of strerror */
  const char *stderr_filename;
  int strlen_fread_buffer; /* Save strlen() value so it doesn't have to be called a lot */
  int saved_errno; /* Save errno value right after error in case errno gets changed later */
  #if __NX__
    stderr_filename = "host:/perror.c_stderr_redirect.txt";
  #else
    stderr_filename = "perror.c_stderr_redirect.txt";
  #endif /* if __NX__ */

  perror_helper__redirect_stderr_to_file(stderr_filename);

  /* Trigger the specified error so that errno will get set and perror()'s
     message will be redirected to the file */
  TriggerErrno(errno_code);
  saved_errno = errno;
  perror(NULL); /* Pass in NULL since we don't need a prepended message */
  fflush(stderr);

  /* Obtain the text that perror() wrote to stderr */
  OpenFile(stderr_filename, "r", &stderr_redirect_file);
  ReadEntireFile(stderr_redirect_file, &fread_buffer);
  CloseFile(stderr_redirect_file, stderr_filename); stderr_redirect_file = NULL;
  strlen_fread_buffer = strlen(fread_buffer);
  TESTCASE_MESSAGE(fread_buffer[strlen_fread_buffer-1] == '\n',
            "There is no newline at the end of perror()'s message. "
            "There should be one.\nperror()'s message: \"%s\"\n",
            fread_buffer);
  }
  /* Get rid of the newline so we don't have to deal with it during
     string comparisons */
  fread_buffer[strlen_fread_buffer-1] = '\0';
  --strlen_fread_buffer; /* Update the new string size */

  strerror_string = strerror(saved_errno);

  TESTCASE_MESSAGE(strcmp(fread_buffer, strerror_string) == 0,
            "perror()'s message differs from strerror()'s. "
            "They should be the same.\nerrno value = %d\nstrerror(%d) = \"%s\"\n"
            "perror() = \"%s\"\n",
            saved_errno, saved_errno, strerror_string, fread_buffer);
  }

  /* Clean up */
  free(fread_buffer);
  DeleteFile(stderr_filename);
}

void test_correct_error_messages(const char *testname, const char *filename)
{
  test_correct_error_messages_helper(filename, ENOENT);
  test_correct_error_messages_helper(filename, EINVAL);
  test_correct_error_messages_helper(filename, ENOMEM);
}
/*------------------------------------------------------------------------}}}}*/




int main(void)
{
  /* Name of file that shouldn't exist to test if perror() prints the right message.
     Define the filename here so we can pass it to the test functions to reuse
     the name */
  const char *non_exist_filename;
  #if __NX__
    non_exist_filename = "host:/perror.c_non_exist_filename.txt";
  #else
    non_exist_filename = "perror.c_non_exist_filename.txt";
  #endif /* if __NX__ */

  perror_helper__make_sure_file_doesnt_exist(non_exist_filename);

  test_proper_formatting(non_exist_filename);
  test_correct_error_messages(non_exist_filename);

  return t_status;
}
